# -*- coding: utf-8 -*-

import copy
import irt.iron.options as iron_opts
from . import funcs as atom_funcs
import yt.wrapper as yt


from irt.bannerland.hosts import get_curr_host, get_host_role
import irt.broadmatching.common_options
from irt.monitoring.common_configs.yt_tables import yt_tables_spec
from irt.monitoring.common_configs.iron_resources import get_srcs, short_src_path

# FQDN текущего хоста
CURR_HOST = get_curr_host()

# Роль текущего хоста
CURR_HOST_ROLE = get_host_role()


def get_atoms_configuration():
    """
    Возвращает основной конфигуратор атомов: какие метрики и на каких хостах мы считаем.
    :return: массив Питоновских dict-ов. Каждый из таких dict-ов имеет следующую структуру:

        • include_host_roles - на хостах каких ролей будет считаться метрика. Тип данных - массив. Если ключ отсутствует,
        метрика будет считаться на всех не idle-хостах. idle-хост - это тот, в роли которого есть подстрока "idle".
        Примеры:
            - ["bannerland", "catalogia-media-scripts"]
            - ["bmfront:master"]

        • include_host_fqdns - на каких ещё хостах, помимо тех, что указаны с ролями из "include_host_role", будут
        считаться метрики. Тип данных - массив. Ключ не обязателен.
        Примеры:
            - ["bmbannerland-yt01e.bm.yandex.ru", "catalogia-media-dev02e.yandex.ru"]
            - ["bmfront01f.bm.yandex.net"]

        • exclude_host_roles - хосты с какими ролями среди указанных будут исключениями для подсчёта метрик.
        Тип данных - массив. Ключ не обязателен.
        Примеры:
            - ["bannerland", "catalogia-media-scripts"]
            - ["bmfront:master"]

        • exclude_host_fqdns - какие отдельные хосты среди оставшихся будут исключениями для подсчёта метрик.
        Тип данных - массив. Ключ не обязателен.
        Примеры:
            - ["bmbannerland-yt01e.bm.yandex.ru", "catalogia-media-dev02e.yandex.ru"]
            - ["bmfront01f.bm.yandex.net"]

        • atom - какой атом будет считаться. Тип данных - функция из funcs.py. Ключ обязателен.
        Примеры:
            - atoms.check_mysql_connection
            - atoms.yt_table_hours_old

        • atom_args - аргументы для атома. Тип данных - словарь с этими аргументами.
        Ключ обязателен (если никаких аргументов в атом не нужно передавать, это просто пустой словарь).
        Примеры:
            - {"yt_client": yt.YtClient(proxy='hahn'), "yt_path": "//home/bannerland/test/perf_avatars_result"}
            - dict()

        • special_handler - если результат работы атома - не чиселка, которую нужно просто отправить в Соломон, то
        указываем отдельную функцию результата обработки атома здесь. Тип данных - функция, принимающая на вход
        единственный обязательный аргумент (результат работы атома), а возвращает массив сенсоров для их отправки
        в Соломон. Ключ не обязателен.
        Пример:
            - lambda atom_res: print_log("Yeah! I have calculated this atom with result '{}'".format(atom_res))

        • solomon_sensor_params: параметры сенсора для отправки в Соломон. Ключ обязателен,
        если не инициализирован "special_handler". Тип данных - словарь, в котором ессть как минимум ключи "cluster",
        "service" и "sensor". Ключ обязателен, если не инициализирован "special_handler", а результат работы атома -
        обычная чиселка, которую нужно отправить в Соломон.
        Пример:
            - {
                  "cluster": "host_info",
                  "service": "system",
                  "sensor": "cpu_load",
                  "labels": UNIVERSAL_SENSOR_LABELS,
              }

        • timeout - эксклюзивный таймаут в секундах для вычисления искомого атома, если не подходит таймаут по умолчанию.
        Тип данных - int или float. Ключ не обязателен.
        Примеры:
            - 100
            - 0.5

        • alert_id - идентификатор алерта в Соломоне (после префикса "atom_"), если хотим навесить алерт на этот атом.
        Ключ не обязателен, если не алертируем атом. Также можно не указывать, если он совпадает с именем сенсора.
        Пример:
            - "check_crontab"

        • crit_thr/warn_thr/ok_thr - пороговые значения для алерта в Соломоне (если его делаем)
         Примеры:
            - 1
            - 0.5
            - "0.5"
            - "1"
            - "5+"
            - "!8"
            - ["0-", "1"]

         • group_by: label-ы сенсора группировки для мультиалерта, если его делаем. Тип данных - массив.
         Пример:
             - ["host"]

    """
    res_configuration = []

    # Конфигурация атомов на железе
    perl_common_options = irt.broadmatching.common_options.get_options()

    res_configuration.extend(get_system_atoms(perl_common_options))
    res_configuration.extend(get_arcadia_atoms())
    res_configuration.extend(get_bannerland_atoms(perl_common_options))
    res_configuration.extend(get_catmedia_atoms(perl_common_options))
    res_configuration.extend(get_bmfront_atoms())
    res_configuration.extend(get_resources_atoms())
    res_configuration.extend(get_yt_tables_atoms())

    return res_configuration

# ************************************************************************
# ************************************************************************
# ************************************************************************


def get_system_atoms(perl_common_options):
    """
    :return: массив системных атомов (диск, CPU и т. д.)
    """
    solomon_template = {
        "cluster": "host_info",
        "service": "system",
        "labels": {"host": CURR_HOST}
    }

    system_atoms = [
        # Загрузка CPU
        conf(atom_funcs.load_cpu, solomon_template=solomon_template, sensor="cpu_load"),

        # Процент свободного места в RAM
        conf(atom_funcs.free_ram, solomon_template=solomon_template, sensor="free_ram_percent"),

        # la (15 min)
        conf(atom_funcs.load_average_15minutes, solomon_template=solomon_template, sensor="load_average_15minutes"),

        # Проверка кронтаба
        conf(
            atom_funcs.check_crontab,
            solomon_template=solomon_template,
            sensor="check_crontab",
            crit_thr=0,
            group_by=["host"],
        ),

        # Проверка atop
        conf(
            atom_funcs.grep_process,
            {"grep_expr": r"/usr/bin/atop"},
            solomon_template=solomon_template,
            sensor="grep_atop_process",
            crit_thr=0,
            group_by=["host"],
        ),

        # Проверка ansible
        conf(
            atom_funcs.file_hours_old,
            {"path": perl_common_options["ansible_mark_file"]},
            exclude_host_roles=[
                'bmbender-front', 'bmcache', 'bmcdict-gen', 'bmcdict-front',
                'catmedia', 'catalogia-media-tasks', 'catalogia-media-front', 'catalogia-media-gen',
            ],
            solomon_template=solomon_template,
            sensor="ansible_last_success",
            warn_thr="4+",
            crit_thr="24+",
            group_by=["host"],
        ),

        # Проверка log-файла atop
        conf(
            atom_funcs.file_hours_old,
            {"path": "/var/log/atop/atop.log"},
            solomon_template=solomon_template,
            sensor="atop_log_hours_old",
            warn_thr="2+",
            crit_thr="24+",
            group_by=["host"],
        ),

        # Проверка log-rotated-файла atop
        conf(
            atom_funcs.file_hours_old,
            {"path": "/var/log/atop/atop.log.1"},
            solomon_template=solomon_template,
            sensor="atop_log_rotated_hours_old",
            warn_thr="25+",
            crit_thr="48+",
            group_by=["host"],
        ),

        # Минимальный из процентов свободного места среди всех разделов диска на хосте
        conf(
            atom_funcs.min_free_disk_percent,
            solomon_template=solomon_template,
            sensor="min_free_disk_percent",
            warn_thr="15-",
            crit_thr="10-",
            group_by=["host"],
        ),

        # TCP States
        conf(
            atom_funcs.tcp_connections_states,
            special_handler=lambda atom_res: [
                dict(
                    solomon_template,
                    **{"sensor": "tcp_state_count", "labels": {"state": state, "host": CURR_HOST}, "value": cc}
                )
                for state, cc in atom_res.items()
            ]
        ),
    ]

    return system_atoms

# ************************************************************************


def get_arcadia_atoms():
    """
    :return: атомы по Аркадии и SVN
    """
    solomon_template = {
        "cluster": "host_info",
        "service": "arcadia",
        "labels": {"host": CURR_HOST}
    }

    def svn_info_handler(atom_res):
        """
        Функция-обраблотчик атома "svn_info"
        """
        if not isinstance(atom_res, dict):
            raise TypeError("Incorrect type of the atom result (waiting for dict).")
        if ("revision" not in atom_res) or ("hours_ago" not in atom_res):
            raise ValueError("Incorrect atom result (can't find key 'revision' or 'hours_ago').")

        return [
            dict(solomon_template, **{"sensor": "svn_revision", "value": atom_res["revision"]}),
            dict(solomon_template, **{"sensor": "svn_hours_ago", "value": atom_res["hours_ago"]}),
        ]

    return [
        # Последняя ревизия в Аркадии и количество часов с момента её обновления
        conf(atom_funcs.svn_info, special_handler=svn_info_handler, timeout=30),

        # Количество диффов в Аркадии
        conf(
            atom_funcs.svn_arcadia_diff,
            solomon_template=solomon_template,
            sensor="svn_diff_count",
            timeout=30,
            warn_thr="1+",
            group_by=["host"],
        ),
    ]


# ************************************************************************


def get_bannerland_atoms(perl_common_options):
    """
    :return: атомы Bannerland-а
    """
    bannerland_atoms = [
        # Коннект с MySQL-базой Bannerland-а
        conf(
            atom_funcs.check_mysql_connection,
            {"dbh_name": "bannerland_dbh"},
            include_host_roles=["bannerland:master"],
            solomon_sensor_params={
                "cluster": "host_info",
                "service": "check_mysql_connection",
                "sensor": "bannerland_dbh",
            },
            alert_id="bl_connect_bannerland_dbh",
            crit_thr="0",
        ),
        conf(
            atom_funcs.check_mysql_connection,
            {"dbh_name": "bannerland_dbh", "dbh_mod": "test"},
            include_host_roles=["bannerland:master"],
            solomon_sensor_params={
                "cluster": "host_info",
                "service": "check_mysql_connection",
                "sensor": "connect_bannerland_test_dbh",
            },
            alert_id="connect_bannerland_test_dbh",
            crit_thr="0",
        )
    ]

    # ******************** Время обновления tskvs для каждого из BL-хостов
    for task_type in ["perf", "dyn"]:
        for days_interval, dir_name in {"1": "from_instances_1day", "30": "from_instances_30days"}.items():

            def tskvs_handler(atom_res, days_interval=days_interval, task_type=task_type):
                if not isinstance(atom_res, dict):
                    raise TypeError("Incorrect type of the atom result (waiting for dict).")

                return [{
                    "cluster": "yt_hahn",
                    "service": "bannerland",
                    "sensor": "tskvs",
                    "value": value,
                    "labels": {
                        "days_interval": days_interval,
                        "task_type": task_type,
                        "host": host,
                    }
                } for host, value in atom_res.items()]

            bannerland_atoms.append(conf(
                atom_funcs.bl_hosts_divide_info,
                {"yt_dir_path": yt.ypath_join("//home/bannerland/data/tskvs", task_type, dir_name), "table_prefix": "{}_tskv_".format(task_type)},
                include_host_roles=["bannerland:master"],
                special_handler=tskvs_handler,
                timeout=20,
            ))

    # ******************** Количество тасок, ожидающих очереди на генерацию
    for task_type in ["perf", "dyn"]:

        def task_waiters_handler(atom_res, task_type=task_type):
            if not isinstance(atom_res, dict):
                raise TypeError("Incorrect type of the atom result (waiting for dict).")
            if len(atom_res) == 0:
                raise ValueError("Incorrect result: no elements for hosts' task waiters.")

            waiters_atoms = [{
                "cluster": "host_info",
                "service": "bannerland",
                "sensor": "task_waiters",
                "value": value,
                "labels": {
                    "host": host,
                    "task_type": task_type,
                }
            } for host, value in atom_res.items()]

            waiters_atoms.append({
                "cluster": "host_info",
                "service": "bannerland",
                "sensor": "avg_task_waiters",
                "value": 1.0 * sum(atom_res.values()) / len(atom_res),
                "labels": {
                    "task_type": task_type,
                }
            })

            return waiters_atoms

        bannerland_atoms.append(conf(
            atom_funcs.task_waiters,
            {"task_type": task_type},
            include_host_roles=["bannerland:master"],
            special_handler=task_waiters_handler,
        ))

    # ******************** dyn_bs_log: время апдейта и количество строк
    def dyn_bs_log_handler(atom_res):
        return [
            {
                "cluster": "yt_hahn",
                "service": "dyn_bs_log",
                "sensor": "last_success_days_ago",
                "value": atom_res["days_ago"],
            },
            {
                "cluster": "yt_hahn",
                "service": "dyn_bs_log",
                "sensor": "row_count",
                "value": atom_res["row_count"],
            },
        ]

    bannerland_atoms.append(conf(
        atom_funcs.dyn_bs_log_info,
        {"yt_path": "//home/bannerland/logs/dyn_bs_log"},
        include_host_roles=["bannerland:master"],
        special_handler=dyn_bs_log_handler,
    ))

    # ******************** Проверка дампа MySQL-таблиц, участвующих в генерации баннеров (в том числе, и в рекоме)
    for table_name in perl_common_options["db_dump"]["bannerland"]["tables"]:
        bannerland_atoms.append(conf(
            atom_funcs.file_size,
            {"path": "/opt/broadmatching/work/db_dump/bannerland/{}.json".format(table_name)},
            include_host_roles=["bannerland:master"],
            solomon_sensor_params={
                "cluster": "host_info",
                "service": "db_tables_dump",
                "sensor": "file_size",
                "labels": {"table_name": table_name}
            },
            crit_thr="1-",
            alert_id="bl_db_tables_dump",
            group_by=["table_name"],
        ))

    # ******************** Проверка файлов банщика в динамиках
    files_to_service = {
        "/opt/broadmatching/work/dynstat/bad_dyn_phrases": "dyn_trashfilter",
    }
    for fname, solomon_service in files_to_service.items():
        bannerland_atoms.append(conf(
            atom_funcs.file_size,
            {"path": fname},
            include_host_roles=["bannerland:master"],
            solomon_sensor_params={
                "cluster": "host_info",
                "service": solomon_service,
                "sensor": "file_size",
                "labels": {"file_name": fname},
            },
            alert_id="bl_{}_file_size".format(solomon_service),
            warn_thr="5000-",
            crit_thr="1000-",
        ))

        bannerland_atoms.append(conf(
            atom_funcs.file_hours_old,
            {"path": fname},
            include_host_roles=["bannerland:master"],
            solomon_sensor_params={
                "cluster": "host_info",
                "service": solomon_service,
                "sensor": "file_hours_old",
                "labels": {"file_name": fname},
            },
            alert_id="bl_{}_file_hours_old".format(solomon_service),
            warn_thr="7008+",
            crit_thr="8760+",
        ))

    # ******************** Проверка количества noload-ов в смартах и динамиках
    for noload_mode in ["perf", "dyn"]:
        bannerland_atoms.append(conf(
            atom_funcs.noload_count,
            {"yt_cluster": "hahn", "mode": noload_mode},
            include_host_roles=["bannerland:master"],
            solomon_sensor_params={
                "cluster": "yt_hahn",
                "service": "bannerland",
                "sensor": "noload_count",
                "labels": {"mode": noload_mode}
            },
            timeout=30,
        ))

    # живые хосты
    for role in ["bannerland", "bannerland-preprod"]:
        # кол-во не idle хостов в hosts.py)
        bannerland_atoms.append(conf(
            atom_funcs.get_bannerland_hosts_count,
            {"role": role},
            include_host_roles=["bmfront:master"],
            solomon_sensor_params={
                "cluster": "host_info",
                "service": role,
                "sensor": "iron_hosts_count",
            }
        ))

        # кол-во живых хостов
        bannerland_atoms.append(conf(
            atom_funcs.get_bannerland_alive_hosts_count,
            {"role": role},
            include_host_roles=["bmfront:master"],
            solomon_sensor_params={
                "cluster": "host_info",
                "service": role,
                "sensor": "alive_iron_hosts_count",
            }
        ))

    # Taskoffers from BL-iron hosts to YT
    def get_offers_delay_by_host_handler(atom_res):
        return [
            {
                "cluster": "yt_hahn",
                "service": "bannerland",
                "sensor": "last_update_export_offers",
                "value": 1,
                "labels": {
                    "host": host,
                    "task_type": task_type,
                    "prod_type": prod_type,
                },
                "ts_datetime": str_datetime,
            }
            for host, prod_type, task_type, str_datetime in atom_res
        ]

    bannerland_atoms.append(conf(
        atom_funcs.get_bannerland_last_offers_export,
        include_host_roles=["bmfront:master"],
        special_handler=get_offers_delay_by_host_handler,
    ))

    return bannerland_atoms

# ************************************************************************


def get_catmedia_atoms(perl_common_options):
    """
    :return: атомы Катмедии
    """
    catmedia_atoms = [
        # Коннект с MySQL-базой "catalogia_media_dbh"
        conf(
            atom_funcs.check_mysql_connection,
            {"dbh_name": "catalogia_media_dbh"},
            include_host_roles=[
                "bannerland",
                "catalogia-media-front",
                "catalogia-media-scripts",
                "catalogia-media-tasks",
                "catmedia",
                "catmedia-dev",
            ],
            solomon_sensor_params={
                "cluster": "host_info",
                "service": "check_mysql_connection",
                "sensor": "catalogia_media_dbh",
                "labels": {"host": CURR_HOST},
            },
            alert_id="connect_catalogia_media_dbh",
            group_by=["host"],
            crit_thr="0",
        ),

        conf(
            atom_funcs.check_dyntable_proxy,
            include_host_roles=[
                "catalogia-media-front",
                "catmedia-dev",
            ],
            solomon_sensor_params={
                "cluster": "host_info",
                "service": "dyntable_proxy",
                "sensor": "access_is_ok",
                "labels": {"host": CURR_HOST},
            },
            alert_id="dyntable_proxy_access_is_ok",
            group_by=["host"],
            crit_thr="0",
        ),

        # Проверяем файл подфраз категорий (на тех машинках, где по крону запущен "prepare_data.pl")
        conf(
            atom_funcs.file_hours_old,
            {"path": perl_common_options["subphraser_params"]["init_data_file"]},
            include_host_roles=[
                "catalogia-media-front",
                "catalogia-media-tasks",
                "catmedia",
            ],
            solomon_sensor_params={
                "cluster": "host_info",
                "service": "file_hours_old",
                "sensor": "subphraser_init_file",
                "labels": {"host": CURR_HOST},
            },
            group_by=["host"],
            warn_thr="26+",
            crit_thr="50+",
        ),
    ]

    # Проверка Kyoto-Cache
    for cache_name in ["broad_kyoto", "ktclient"]:
        catmedia_atoms.append(conf(
            atom_funcs.kyoto_cache_test,
            {"cache_name": cache_name},
            include_host_roles=["catmedia"],
            solomon_sensor_params={
                "cluster": "host_info",
                "service": "kyoto_cache_test",
                "sensor": cache_name,
            },
            alert_id="kyoto_cache_{}".format(cache_name),
            crit_thr="0",
        ))

    return catmedia_atoms

# ************************************************************************


def get_bmfront_atoms():
    """
    :return: атомы bmfront-а
    """
    return [
        # Коннект с MySQL-базой "monitoring_dbh"
        conf(
            atom_funcs.check_mysql_connection,
            {"dbh_name": "monitoring_dbh"},
            include_host_roles=["bmfront"],
            solomon_sensor_params={
                "cluster": "host_info",
                "service": "check_mysql_connection",
                "sensor": "monitoring_dbh",
                "labels": {"host": CURR_HOST},
            },
            alert_id="connect_monitoring_dbh",
            group_by=["host"],
            crit_thr="0",
        ),
    ]

# ************************************************************************


def get_resources_atoms():
    """
    :return: атомы ресурсов, хранящихся на железе (до их переезда в Sandbox)
    """
    resources_atoms = []
    iron_srcs = get_srcs()

    for path, fqdns in iron_srcs.items():
        short_path = short_src_path(path)

        resources_atoms.append(conf(
            atom_funcs.file_size,
            {"path": path},
            include_host_fqdns=fqdns,
            solomon_sensor_params={
                "cluster": "host_info",
                "service": "iron_resources",
                "sensor": "file_size",
                "labels": {"host": CURR_HOST, "file_name": short_path}
            },
            alert_id="iron_srcs_file_size",
            group_by=["host", "file_name"],
            crit_thr="1-",
        ))

        resources_atoms.append(conf(
            atom_funcs.file_hours_old,
            {"path": path},
            include_host_fqdns=fqdns,
            solomon_sensor_params={
                "cluster": "host_info",
                "service": "iron_resources",
                "sensor": "file_hours_old",
                "labels": {"host": CURR_HOST, "file_name": short_path}
            }
        ))

    return resources_atoms

# ************************************************************************


def get_yt_tables_atoms():
    """
    :return: атомы для мониторинга YT-таблиц (количество строк и время последнего апдейта)
    """
    yt_tables_atoms = []

    for yt_table in yt_tables_spec:
        yt_client = yt.YtClient(
            config={
                "proxy": {"url": yt_table["yt_cluster"]},
                "token_path": iron_opts.get("yt_token_path")
            }
        )

        for sensor_name, func in {
            "hours_old": atom_funcs.yt_table_hours_old,
            "row_count": atom_funcs.yt_table_rows_count
        }.items():

            yt_tables_atoms.append(conf(
                func,
                {"yt_client": yt_client, "yt_path": yt_table["yt_path"]},
                include_host_roles=["bmfront:master"],
                solomon_sensor_params={
                    "cluster": yt_table["solomon_cluster"],
                    "service": yt_table["solomon_service"],
                    "sensor": sensor_name,
                    "labels": {"table_name": yt_table["yt_path"]},
                }
            ))

    return yt_tables_atoms

# ************************************************************************


def conf(atom, atom_args=None, **kwargs):
    """
    Вспомогательная фунция, возвращающая словарь-конфиг конкретного атома. Нужна для более удобного формата записи конфига.
    :param atom: функция атома из funcs.py
    :param atom_args: аргументы атома (None, если их нет)
    :param solomon_template: шаблон Соломоновского сенсора с кластером, сервисом и label-ами (если таковые имеются)
    :param sensor: имя Соломоновского сенсора
    :param add_labels: дополнительные лейблы Соломоновского сенсора, если таковые нужны
    :param **kwargs: другие параметры, допустимые форматом конфигурации атома.
    :return: словарь-конфиг атома.
    """
    atom_conf = {"atom": atom}
    if not atom_args:
        atom_conf["atom_args"] = dict()
    else:
        atom_conf["atom_args"] = atom_args

    copy_kwargs_fields = set([
        "alert_id",
        "crit_thr",
        "group_by",
        "include_host_roles",
        "include_host_fqdns",
        "exclude_host_roles",
        "exclude_host_fqdns",
        "ok_thr",
        "special_handler",
        "solomon_sensor_params",
        "timeout",
        "warn_thr",
    ])
    atom_conf.update({key: val for key, val in kwargs.items() if key in copy_kwargs_fields})

    if "solomon_template" in kwargs:
        solomon_sensor_params = copy.deepcopy(kwargs["solomon_template"])

        if "sensor" in kwargs:
            solomon_sensor_params["sensor"] = kwargs["sensor"]
        if "add_labels" in kwargs:
            solomon_sensor_params["labels"] = solomon_sensor_params.get("labels", dict())
            solomon_sensor_params["labels"].update(kwargs["add_labels"])

        atom_conf["solomon_sensor_params"] = solomon_sensor_params

    # Если идентификатор алерта (при алертировании атома) опущен, значит он совпадает с сенсором
    if (len({"crit_thr", "warn_thr", "ok_thr"} & set(kwargs.keys())) > 0) and ("alert_id" not in kwargs):
        atom_conf["alert_id"] = atom_conf["solomon_sensor_params"]["sensor"]

    return atom_conf
