# -*- coding: utf-8 -*-

import collections
import datetime
import dateutil.parser
import dateutil.tz
import gzip
import logging
import os
import psutil
import re
import subprocess
import sqlalchemy as sa
import time
import typing

import yalibrary.svn as svn
import yt.wrapper as yt

import irt.iron.options as iron_opts
from irt.bannerland.options import get_option as get_bannerland_option
from irt.bannerland.hosts import get_hosts_by_role
import irt.broadmatching.common_options
from bm.yt_tools import get_upload_time

if typing.TYPE_CHECKING:
    from typing import List, Tuple


logger = logging.getLogger(__name__)


# корень аркадии, из которой запущен скрипт
def bin_arcadia_root():
    curr_dir = os.getcwd()
    iters = 50
    while not os.path.exists(curr_dir + '/.arcadia.root'):
        curr_dir += '/..'
        iters -= 1
        if iters == 0:
            raise Exception("Can't find arcadia root!")
    return os.path.abspath(curr_dir)


# пытаемся найти корень Аркадии
def get_arcadia_root():
    if 'ARCADIA_ROOT' in os.environ:
        return os.environ['ARCADIA_ROOT']
    else:
        return bin_arcadia_root()


def _run_mysql_query(dbh, sql_query):
    """
    Внутренняя функция текущего модуля, производящая запрос в MySQL-базу и возвращающая результат запроса.
    Логи соединения с БД пишем в /dev/null, дабы не светить в логах параметрами авторизации БД.
    :param dbh: имя БД.
    :param sql_query: SQL-запрос.
    :return: текст ответа в искомом формате.
    """
    # [!]  Этот путь должен быть валидным и для железных
    connect_script = "/opt/broadmatching/scripts/utils/connect-dbh.pl"
    return subprocess.check_output([connect_script, dbh, "-m", "-N -B -e \"{}\"".format(sql_query)], stderr=subprocess.DEVNULL)


def _connect_to_dbh(dbh_name):
    """
    Внутренняя функция. Выполняем коннект к MySQL БД через sqlalchemy
    :param dbh_name: алиас одной из наших БД
    :return: sqlalchemy-сокет подключения
    """
    dbh_opt = iron_opts.get("mysql_dbh_to_common_opts").get(dbh_name)
    if not dbh_opt:
        logger.error("Unknown dbh!")
        return

    dbh_params = irt.broadmatching.common_options.get_options()[dbh_opt]

    password = dbh_params.get("password")
    if not password and dbh_params.get("password_path"):
        with open(dbh_params["password_path"], "r") as f:
            password = f.readline().strip()

    return sa.create_engine("mysql://{user}:{password}@{host}:{port}/{db}".format(
        user=dbh_params["user"],
        password=password,
        host=dbh_params["hosts"][0],
        port=dbh_params["port"],
        db=dbh_params["database"],
    ))


def check_dyntable_proxy():
    """
    Проверяем работоспособность прокси dyntable, ходим в динтаблицу для смартов
    :return: 1 - если поход через проксю был успешен, 0 - если что-то пошло не так.
    """
    check_script = "/opt/broadmatching/scripts/utils/test_dyntable_proxy.pl"
    try:
        subprocess.check_call([check_script])
    except subprocess.CalledProcessError as exc:
        logger.exception("check_dyntable_proxy failed. test_dyntable_proxy.pl was failed with exit-code %s.", exc.returncode)
        return 0

    return 1


def check_mysql_connection(dbh_name, dbh_mod="prod"):
    """
    Проверяем коннект к MySQL-базе.
    Логи коннекта с БД пишем в /dev/null, дабы не светить в логах параметрами авторизации БД.
    :param dbh_name: имя БД.
    :param dbh_mod: prod/test/preprod.
    :return: 1 - если коннект успешен, 0 - если что-то пошло не так.
    """
    # [!]  Этот путь должен быть валидным и для железных
    check_script = "/opt/broadmatching/scripts/utils/test-dbh.pl"

    env = {}
    d = {
        "prod": "",
        "test": "mdb_test_mysql",
        "preprod": "mdb_preprod_mysql",
    }
    if dbh_mod not in d:
        raise ValueError("bad dbh_mod=%s, avaliable %s", dbh_mod, d)

    env["BM_USE_PATCH"] = d[dbh_mod]

    try:
        subprocess.check_call([check_script, dbh_name], env=env)
    except subprocess.CalledProcessError as exc:
        logger.exception("ERROR: Connection with '%s' was finished with exit-code %s.", dbh_name, exc.returncode)
        return 0

    return 1


def svn_arcadia_diff():
    """
    :return: количество диффов в Аркадийной папке.
    """
    irt_dir = os.path.join(get_arcadia_root(), "rt-research")
    return sum(1 for _, status in svn.svn_st(irt_dir).items() if status.get("item") != "unversioned")


def check_crontab():
    """
    Проверка кронтаба.
    :return: 1 - если ОК, 0 - в противном случае (для удобства просмотра статусов на графике).
    """
    crontab_script = os.path.join(get_arcadia_root(), "rt-research/broadmatching/scripts/set-crontab.pl")

    try:
        subprocess.check_call([crontab_script, "--check"])
    except subprocess.CalledProcessError as exc:
        logger.exception("ERROR: crontab checking has been finished with exit code %s.", exc.returncode)
        return 0

    return 1


def free_space(path):
    """
    Вычисляет процент свободного места по заданному пути.
    :param path: путь
    :return: процент свободного места
    """
    if isinstance(path, str):
        return 100 - psutil.disk_usage(path).percent


def min_free_disk_percent():
    """
    :return: Минимальный из процентов свободного места среди всех разделов диска на хосте.
    """
    min_perc = None
    for dp in psutil.disk_partitions():
        free_perc = 100 - psutil.disk_usage(dp.mountpoint).percent
        min_perc = free_perc if min_perc is None else min(min_perc, free_perc)
    return min_perc


def free_ram():
    """
    Вычисляет процент свободного места в оперативной памяти.
    :return: процент свободного места.
    """
    ram_info = psutil.virtual_memory()
    return 100 - ram_info.percent


def load_cpu():
    """
    :return: загрузка CPU в процентах.
    """
    return psutil.cpu_percent()


def file_size(path):
    """
    :param path: путь к файлу
    :return: размер файла в байтах
    """
    f_stat = os.stat(path)
    return f_stat.st_size


def grep_process(grep_expr):
    """
    Грепаем нужный процесс.
    :param grep_expr: выражение для грепа.
    :return: 1, если греп успешен, 0 - в противном случае (для удобства просмотра статусов на графике).
    """
    proc = subprocess.Popen(
        ["pgrep", "-f", grep_expr],
        stdout=subprocess.PIPE
    )
    out_txt, err_txt = proc.communicate()

    return 1 if out_txt else 0


def svn_info():
    """
    Возвращает SVN-информацию по ревизии текущего Аркадийного репозитория.
    :return: словарь с информацией по номеру ревизии Аркадии и количеству часов с момента её создания.
    """
    irt_dir = os.path.join(get_arcadia_root(), "rt-research")
    svn_info = svn.svn_info(irt_dir)
    return {
        "revision": int(svn_info["revision"]),
        "hours_ago": (time.time() - svn_info["timestamp"]) / 3600.0
    }


def yt_table_hours_old(yt_client, yt_path):
    """
    Количество часов, прошедших с последнего обновлоения заданной таблицы (опираемся на наш атрибут "upload_time").
    :param yt_client: инициализированный YT-клиент.
    :param yt_path: YT-путь к нужной таблице.
    :return: количество часов с момента последнего апдейта таблицы.
    """
    upload_time = get_upload_time(yt_path, yt_client)
    if upload_time is not None:
        return (time.time() - upload_time) / 3600.0


def yt_table_rows_count(yt_client, yt_path):
    """
    Количество строк в заданной YT-таблице.
    :param yt_client: инициализированный YT-клиент.
    :param yt_path: YT-путь к нужной таблице.
    :return: количество строк.
    """
    return yt_client.get_attribute(yt_path, "row_count", default=None)


def bl_hosts_divide_info(yt_dir_path, table_prefix=""):
    """
    Агрегированная по хостам Bannerland-а информация относительно tskv и динамического экспорта.
    :param yt_dir_path: YT-путь к нужной папке, где собрана информация для каждой из мащинок.
    :param table_prefix: универсальный префикс для всех таблиц в указанной папке.
    :return: словарь, где ключ - это хост, а значение - кол-во часов с апдейта таблицы по этому хосту.
    """
    hours_stat = dict()
    yt_client = yt.YtClient(proxy="hahn", config=yt.default_config.get_config_from_env())

    for host in get_hosts_by_role("bannerland"):
        short_host_name = host.split(".")[0]
        table_hours_old = yt_table_hours_old(yt_client, yt.ypath_join(yt_dir_path, table_prefix + short_host_name))
        if table_hours_old:
            hours_stat[host] = table_hours_old

    return hours_stat


def dyn_bs_log_info(yt_path, days_ago=30):
    """
    Ищем наиболее старый необновившийся dyn_bs_log, начиная c <days_ago> дней назад.
    :param yt_path: YT-путь к папке с dyn_bs_log.
    :param days_ago: кол-во дней назад, начиная с которого начинаем искать.
    :return: для найденной таблицы - кол-во дней назад (для которого там подсчитана статистика) и число строк в ней.
    """
    date_today = datetime.date.today()
    yt_client = yt.YtClient(proxy="hahn", config=yt.default_config.get_config_from_env())
    dbl_tables = {t for t in yt_client.list(yt_path, absolute=False)}

    res_day_ago = days_ago
    while res_day_ago > -1:
        current_date = (date_today - datetime.timedelta(days=res_day_ago)).strftime("%Y%m%d")
        if current_date in dbl_tables:
            res_day_ago -= 1
            continue
        break

    res_day_ago += 1
    dbl_last_name = (date_today - datetime.timedelta(days=res_day_ago)).strftime("%Y%m%d")
    row_count = yt_table_rows_count(yt_client, yt.ypath_join(yt_path, dbl_last_name))

    return {
        "days_ago": res_day_ago,
        "row_count": row_count,
    }


def task_waiters(task_type):
    """
    Количество тасок на BL-машинках, ожидающих генерации.
    :param task_type: тип тасок ("perf" или "dyn").
    :return: агрегированная по хостам информация о количестве искомых тасок.
    """
    if task_type == "perf":
        db_table = "PerfTasksQueue"
    elif task_type == "dyn":
        db_table = "DynTasksQueue"
    else:
        raise ValueError("Unknown task type for task_waiters.")

    socket = _connect_to_dbh("bannerland_dbh")
    meta = sa.MetaData(bind=socket, reflect=True)
    tq = meta.tables[db_table]

    bl_hosts = set(get_hosts_by_role("bannerland"))
    res_dict = {host: 0 for host in bl_hosts}

    query = sa. \
        select([sa.Column("host"), sa.func.count().label("cc")]). \
        where(tq.c.State == "New"). \
        group_by(sa.Column("host"))

    for row in socket.execute(query):
        if row.host not in bl_hosts:
            continue
        res_dict[row.host] = row.cc

    return res_dict


def file_hours_old(path):
    """
    Считаем число часов назад последней модификации заданного файла.
    :param path: путь к файлу
    :return: сколько часов назад он был модифицирован.
    """
    file_stat = os.stat(path)
    return (time.time() - file_stat.st_mtime) / 3600.0


def last_success_finish_script(log_path, success_mark):
    """
    Ищем в заданном файле лога (а также его log-rotate продолжениях) метку <success_mark> и считаем, сколько часов назад она была установлена.
    :param log_path: путь к файлу лога (наиболее свежему).
    :param success_mark: метка, которую мы ищем в логе.
    :return: количество часов назад, когда метка была поставлена или None, если метка не найдена или лог в некорректном формате.
    """
    file_name = os.path.basename(log_path)
    dir_name = os.path.dirname(log_path)

    all_rotated_logs = [(0, file_name)]
    for ff in os.listdir(dir_name):
        search_res = re.search(r"^{}\.(\d+)(|\.gz)$".format(file_name), ff)
        if search_res:
            all_rotated_logs.append((int(search_res.group(1)), ff))

    all_rotated_logs.sort(key=lambda x: x[0])

    hours_old = None
    for r_file in all_rotated_logs:
        log_fname = r_file[1]
        if re.search(r"\.gz$", log_fname):
            file_lines = gzip.open(os.path.join(dir_name, log_fname)).readlines()
        else:
            file_lines = open(os.path.join(dir_name, log_fname)).readlines()
        for log_line in reversed(file_lines):
            if success_mark in log_line:
                dateutil_mark = dateutil.parser.parse(log_line.split("\t")[0]).timetuple()
                hours_old = (time.time() - time.mktime(dateutil_mark)) / 3600.0
                break
        if hours_old is not None:
            break

    return hours_old


def kyoto_cache_test(cache_name):
    """
    Тест для Kyoto Cache.
    :param cache_name: наименование кэша, который мы тестируем.
    :return: результат теста (1 - если ОК, 0 - в противном случае).
    """
    check_script = os.path.join(get_arcadia_root(), "rt-research/broadmatching/scripts/tests/kyoto/test_cache.pl")

    try:
        subprocess.check_call(["perl", check_script, cache_name])
    except subprocess.CalledProcessError as exc:
        logger.exception("ERROR: Test for Kyoto cache '%s' was finished with exit-code %s.", cache_name, exc.returncode)
        return 0

    return 1


def noload_count(yt_cluster, mode):
    """
    Количество Noload-ов.
    :param yt_cluster: YT-кластер.
    :param mode: тип заданий ("perf" или "dyn").
    :return: число noload-ов.
    """
    if mode == "dyn":
        table = get_bannerland_option("dyn_tasks_source_table")
    elif mode == "perf":
        table = get_bannerland_option("perf_tasks_source_table")
    else:
        raise ValueError("Unknown mode: {}".format(mode))

    noload_field = "Noload"
    orderid_field = "OrderID"

    yt_client = yt.YtClient(proxy=yt_cluster, config=yt.default_config.get_config_from_env())
    noload_ids = set()
    for row in yt_client.read_table(yt.TablePath(table, columns=[noload_field, orderid_field])):
        if noload_field in row and row[noload_field]:
            noload_ids.add(row[orderid_field])

    return len(noload_ids)


def tasks_queue_waiting_time(cmd):
    """
    Максимальное время ожидания невыполненных заданий в таблице TasksQueue базы
    :param cmd: команда, в рамках которой выполнялись задания.
    :return: максимальное время ожидания
    """
    socket = _connect_to_dbh("catalogia_media_dbh")
    meta = sa.MetaData(bind=socket, reflect=True)
    tq = meta.tables["TasksQueue"]

    query = sa. \
        select([sa.func.unix_timestamp(tq.c.ComeTime)]). \
        where(sa.and_(tq.c.Cmd == cmd, tq.c.State.in_(("", "processing")))). \
        order_by(tq.c.ComeTime). \
        limit(1)
    proc_time = socket.execute(query).scalar()

    if proc_time is None:
        return 0
    return (time.time() - proc_time) / 3600.0


def get_bannerland_alive_hosts_count(role):
    """
    Количество Живых хостов в роли (по таблицам на locke), таблицы создаются скриптом bannerland_set_alive
    :param role: str.
    :return: int
    """
    yt_client = yt.YtClient(proxy="locke", config=yt.default_config.get_config_from_env())

    locks_dir = get_bannerland_option("alive_yt_locke_path")
    yt_alive_dir = "{}/{}".format(locks_dir, role)
    return len(yt_client.list(yt_alive_dir))


def get_bannerland_hosts_count(role):
    """
    Количество хостов в роли, нужно чтобы оценить отношение живых хостов к общему количеству
    :param role: str.
    :return: int
    """
    return len(get_hosts_by_role(role))


def tcp_connections_states():
    """
    Return counts of different states for current host's TCP connections.

    :return: Python counter: <TCP state> -> <count>
    :rtype: collections.Counter[str, int]
    """
    return collections.Counter(elem.status.lower() for elem in psutil.net_connections() if elem.status)


def load_average_15minutes():
    """
    Return a host load average for last 15 minutes.

    :return: 15min la
    :rtype: float
    """
    return os.getloadavg()[2]


def get_task_specified_last_offers(
        yt_client,  # type: yt.YtClient
        task_type,  # type: str
        prod_type,  # type: str
):                  # type: (...) -> List[Tuple[str, str, str, str]]
    """
    Return for the specified task's type and prod-type a datetime of last export offers loading from iron BL host to YT.

    :param yt_client: initialized YT-client
    :param task_type: task type
    :param prod_type: task prod type ("prod"/"preprod")
    :return: set of tuples of last export offers datetime timestamp in format: (host, prod_type, task_type, datetime)
    """
    res_timings = []

    for yt_table in yt_client.list(yt.ypath_join(
        "//home",
        "bannerland",
        "{}{}".format(task_type, "" if prod_type == "prod" else "_{}".format(prod_type)),
        "tasks_and_offers",
        "offers_from_instances",
    )):
        host, str_datetime = yt_table.split("_", 1)
        datetime_obj = datetime.datetime.strptime(str_datetime, get_bannerland_option("bannerland_pocket_name_format"))
        if datetime_obj.tzinfo is None:
            datetime_obj = datetime_obj.replace(tzinfo=dateutil.tz.tzlocal())
        res_timings.append((host, prod_type, task_type, datetime_obj.isoformat()))

    return res_timings


def get_bannerland_last_offers_export():  # type: () -> List[Tuple[str, str, str, str]]
    """
    Return for prod/preprod and for each actual BL task_type a datetime of last export offers loading from iron BL host to YT.
    :return: set of tuples of last export offers datetime timestamp in format: (host, prod_type, task_type, datetime)
    """
    res_timings = []
    yt_client = yt.YtClient(proxy="hahn", config=yt.default_config.get_config_from_env())

    for task_type in ["perf", "dyn"]:
        for prod_type in ["prod", "preprod"]:
            res_timings += get_task_specified_last_offers(yt_client, task_type, prod_type)
    return res_timings
