# -*- coding: utf-8 -*-

from irt.bannerland.options import get_option as get_bl_opt
from irt.bannerland.hosts import get_hosts_by_role, get_hosts_by_roles
import irt.broadmatching.common_options

from irt.monitoring.juggler.configs.lib import PROD_TELEGRAM_CHANNEL, \
    get_classic_aggregate, \
    telegram_notification_rules, \
    get_finish_scripts_aggregate, \
    get_iron_hosts_aggregates


def get_checks():
    """
    :return: Возвращает список агрегатов для bmfront-а
    """
    checks = []
    juggler_host = "bannerland"
    module_notifications = telegram_notification_rules(PROD_TELEGRAM_CHANNEL)

    bl_hosts = get_hosts_by_role("bannerland")
    bl_preprod_hosts = get_hosts_by_role("bannerland-preprod")
    bl_master = get_hosts_by_role("bannerland:master")[0]

    # Состояние железных хостов Bannerland-а
    checks += get_iron_hosts_aggregates(
        hosts=get_hosts_by_roles(["bannerland", "bannerland-yt", "bannerland-yt-net"]),
        aggr_host=juggler_host,
        notification_rules=module_notifications,
    )

    # Проверяем успешное выполнение необходимых скриптов
    for hosts, script in [
        (bl_hosts, "get-resources"),
        ([bl_master], "get_kwyt_source_url_title"),
    ]:
        checks.append(get_finish_scripts_aggregate(
            hosts=hosts,
            scripts=[script],
            aggr_host=juggler_host,
            aggr_service=script,
            notification_rules=module_notifications,
        ))

    # Коннект к Мускулю
    raw_events = [
        {"host": bl_master, "service": "bl_connect_bannerland_dbh.atoms.irt"},
        {"host": bl_master, "service": "connect_bannerland_test_dbh.atoms.irt"},
    ]
    for host in bl_hosts:
        raw_events.append(dict(host=host, service="connect_catalogia_media_dbh.atoms.irt"))

    checks.append(get_classic_aggregate(
        aggr_host=juggler_host,
        aggr_service="mysql_connection",
        notification_rules=module_notifications,
        simple_children=raw_events,
        description="BL-hosts connection to MySQL",
    ))

    # tskvs
    raw_events = []
    for task_type in ["perf", "dyn"]:
        for days_interval in [1, 30]:
            for host in bl_hosts:
                raw_events.append(dict(host=host, service="{}_tskvs_{}.bannerland.irt".format(task_type, days_interval)))
    checks.append(get_classic_aggregate(
        aggr_host=juggler_host,
        aggr_service="tskvs",
        notification_rules=module_notifications,
        simple_children=raw_events,
        description="BL_tskvs",
    ))

    # Дамп таблиц на мастере Bannerlnad-а (используется в смартах/динамиках)
    raw_events = []
    for table_name in irt.broadmatching.common_options.get_options()["db_dump"]["bannerland"]["tables"]:
        raw_events.append(dict(host=bl_master, service="bl_db_tables_dump.{}.atoms.irt".format(table_name)))

    checks.append(get_classic_aggregate(
        aggr_host=juggler_host,
        aggr_service="db_dump_bl_tables",
        notification_rules=module_notifications,
        simple_children=raw_events,
        description="Dumps of Bannerland DB tables",
    ))

    # Проверка файла банщика для динамиков
    raw_events = [
        dict(host=bl_master, service="bl_dyn_trashfilter_{}.atoms.irt".format(property))
        for property in ["file_hours_old", "file_size"]
    ]
    checks.append(get_classic_aggregate(
        aggr_host=juggler_host,
        aggr_service="bl_iron_dyn_trashfilter",
        notification_rules=module_notifications,
        simple_children=raw_events,
        description="Bannerland iron trashfilter file",
    ))

    # Мониторинг всех SimDistance
    checks.append(get_classic_aggregate(
        aggr_host="artmon",
        aggr_service="monitor_all_simdistances",
        notification_rules=module_notifications,
        simple_children=[{"host": "artmon_stat", "service": "monitor_all_simdistances.irt"}],
        description="Monitoring of all SimDistances",
    ))

    # Мониторинг денег в Артмоне в разрезе по ContextType и SimDistance
    sim_distances_in_context_types = get_bl_opt("sim_distances_in_context_types")
    non_monitor_sds = set(get_bl_opt("non_monitor_sds"))
    for context_type, context_type_name in [
        (7, "dynamic"),
        (8, "performance"),
    ]:
        raw_events = [
            {"host": "artmon_stat", "service": "sim_distance_{}.diff_ok.irt".format(sd)}
            for sd in sim_distances_in_context_types[context_type] if sd not in non_monitor_sds
        ]
        raw_events.append({"host": "artmon_stat", "service": "context_type_{}.diff_ok.irt".format(context_type)})

        checks.append(get_classic_aggregate(
            aggr_host="artmon",
            aggr_service="{}_money".format(context_type_name),
            notification_rules=module_notifications,
            simple_children=raw_events,
            flaps_config={"stable": 3 * 3600, "critical": 9 * 3600},
            description="Monitoring of {} money in Artmon".format(context_type_name),
        ))

    # Monitoring for actuality of taskoffers export from BL-iron hosts to YT
    for task_type in ["perf", "dyn"]:
        for prod_type in ["prod", "preprod"]:
            if prod_type == "prod":
                hosts = bl_hosts
            else:
                hosts = bl_preprod_hosts

            checks.append(get_classic_aggregate(
                aggr_host="bannerland",
                aggr_service="{}_{}_export_offers".format(task_type, prod_type),
                notification_rules=module_notifications,
                simple_children=[{"host": host, "service": "__ontime__.actuality.export_offers.{}.{}.irt".format(task_type, prod_type)} for host in hosts],
                description="[{}-{}] Monitoring of export offers actuality from BL-iron hosts".format(task_type, prod_type),
                only_value=True,
            ))

    # Monitoring of non-critical size for BL tasks queues
    for task_type in ["perf", "dyn"]:
        checks.append(get_classic_aggregate(
            aggr_host="bannerland",
            aggr_service="{}_tasks_queue_size".format(task_type),
            notification_rules=module_notifications,
            simple_children=[{"host": host, "service": "{}.tasks_queue.irt".format(task_type)} for host in bl_hosts],
            description="[{}] Сriticality of tasks queue size".format(task_type),
        ))

    return checks
