# -*- coding: utf-8 -*-
from juggler_sdk import NotificationOptions, Child, FlapOptions
import copy

# Идентификатор тестового Telegram-канала, куда будут слаться уведомления из Juggler-а
TEST_TELEGRAM_CHANNEL = "IRT_Test_Channel"

# Идентификатор продового Telegram-канала, куда будут слаться уведомления из Juggler-а
PROD_TELEGRAM_CHANNEL = "YT_tables_prepare_data"

# Опции Juggler-мониторинга, актуальные для большинства модулей конфига
JUGGLER_MONITORING_OPTIONS = {

    # Префикс сырого события, отвечающего за протухание метрики
    "ontime_prefix": "__ontime__.",

    # Шаблон стандартного агрегатора (проверки) в Juggler-е для большинства наших проверок
    "check_template": {
        "refresh_time": 90,
        "ttl": 48 * 3600,
        "aggregator": "logic_or",
    },

    # Метки скриптов, выполняющиеся по крону на всех железных машинках
    "universal_iron_scripts": ["check-logs", "put-resources-to-sb"],

    # Логины, для которых будут приходить SMS-нотификации по активной проверке балансера "catmedia.yandex.ru"
    "catmedia_sms_logins": ["sergio", "apovetkin", "fawkes"],

    # Логины, для которых будут приходить SMS-нотификации по активной проверке балансера "bmfront.bm.yandex-team.ru"
    "bmfront_sms_logins": ["sergio", "fawkes", "rekub"],
}

# Массив префиксов метрик, если мониторятся значение и самой метрики, и её протухание.
JUGGLER_MONITORING_OPTIONS["with_nor_ontime"] = [JUGGLER_MONITORING_OPTIONS["ontime_prefix"], ""]


def juggler_option(option_name):
    """
    :param option_name: ключ опции
    :return: значение опции из JUGGLER_MONITORING_OPTIONS
    """
    return JUGGLER_MONITORING_OPTIONS[option_name]


def telegram_notification_rules(telegram_channel):
    """
    Возвращает правила нотификаций для каждого из агрегатов-сервисов в выделенный Telegram-канал.
    Обычные оповещения, если "OK" или "WARN". Повторяющиеся каждые 8 часов оповещения, если статус "CRIT".
    :param telegram_channel: идентификатор зарегестрированного в Juggler-е Telegram-канала
    :return: массив стандартных правил нотификации
    """
    return [
        NotificationOptions(
            template_name="on_status_change",
            template_kwargs={
                "status": ["OK", "WARN"],
                "method": "telegram",
                "login": telegram_channel,
            },
        ),
        NotificationOptions(
            template_name="on_status_change",
            template_kwargs={
                "status": ["CRIT"],
                "method": "telegram",
                "login": telegram_channel,
                "repeat": 8 * 3600,
            },
        ),
    ]


def get_classic_aggregate(aggr_host, aggr_service, simple_children, notification_rules=(), description="",
                          other_children=None, flaps_config=None, only_value=False):
    """
    Формирует классический Juggler-агрегат на основе заданного перечня из сырых событий.
    :param aggr_host: координата хоста искомого агрегата в Juggler-е
    :param aggr_service: координата сервиса искомого агрегата в Juggler-е
    :param notification_rules: правила нотификации
    :param simple_children: стандартные "дети" агрегата - сырые события или подагрегаты (каждый элемент - словарь с ключами "host" и "service")
    :param other_children: массив оставшихся Child-событий агрегата, если таковой нужен
    :param description: описание Juggler-агрегата
    :param flaps_config: настройки флапа, если он нужен (подробнее про флапы - здесь: https://nda.ya.ru/t/7GW_FIns3Vwe9T)
    :param only_value: если True - не включать в агрегат сырые события по протуханию (если False - то включать)
    :return: словарь-конфиг Juggler-агрегата (проверки)
    """
    children = other_children if other_children else []

    if only_value:
        children += [Child(host=child["host"], service=child["service"]) for child in simple_children]
    else:
        for service_prefix in juggler_option("with_nor_ontime"):
            for child in simple_children:
                children.append(Child(
                    host=child["host"],
                    service="{}{}".format(service_prefix, child["service"]),
                ))

    return dict(
        juggler_option("check_template"),
        host=aggr_host,
        service=aggr_service,
        children=children,
        notifications=notification_rules,
        flaps_config=FlapOptions(**flaps_config) if flaps_config else None,
        description=description,
    )


def get_yt_resources_aggregate(yt_cluster, account, notification_rules):
    """
    :param yt_cluster: имя YT-кластера
    :param account: имя аккаунта
    :param notification_rules: правила нотификации
    :return: Juggler-агрегат по ресурсам в YT для заданного аккаунта и YT-кластера
    """
    children = []
    for source_name in ["disk_space", "chunk_count", "node_count"]:
        children.append(Child(
            host="yt_{}".format(yt_cluster),
            service="free_{}_perc.{}.yt.irt".format(source_name, account)
        ))

    return dict(
        juggler_option("check_template"),
        host="yt_{}_sources".format(yt_cluster),
        service="account_{}".format(account),
        children=children,
        notifications=notification_rules,
        flaps_config=FlapOptions(stable=1200, critical=3600),
        description="Ресурсы в YT на кластере '{}' для аккаунта '{}'".format(yt_cluster, account),
    )


def get_iron_hosts_aggregates(hosts, aggr_host, notification_rules, custom_flaps=None, check_ansible=True):
    """
    :param hosts: fqdn-ы хостов, для которых создается агрегат
    :param aggr_host: координата 'host' соответствующего агрегата в Juggler-е
    :param notification_rules: правила нотификации
    :param custom_flaps: кастомные правила флапа для выделенных атомов
    :param check_ansible: проверяем ли актуальность ansible на хосте (актуально только для хостов под L3)
    :return: массив Juggler-агрегатов: главный агрегат по главным ресурсам железа + SVN, а также подагрегаты под ним
    """
    result_aggregates = []
    main_aggr_children = []
    other_children = []
    atoms_format = "{}.atoms.irt"

    system_atoms = [
        "atop_log_hours_old",
        "atop_log_rotated_hours_old",
        "grep_atop_process",
        "min_free_disk_percent",
        "svn_hours_ago",
    ]
    if check_ansible:
        system_atoms.append("ansible_last_success")

    for host in hosts:
        for system_atom in system_atoms:
            main_aggr_children.append({
                "host": host,
                "service": atoms_format.format(system_atom),
            })
        other_children.append(Child(host=host, service="collect_logs.separate_host.atoms.irt"))

    # Флапающие атомы (только по значению!) выделяем в отдельные подагрегаты в составе агрегата главного
    for flap_check in [
        {"atom": "check_crontab", "stable": 600, "critical": 3000},
        {"atom": "svn_diff_count", "stable": 600, "critical": 3000},
    ]:
        atom = flap_check["atom"]
        if (custom_flaps is not None) and (atom in custom_flaps):
            stable_time = custom_flaps[atom]["stable"]
            critical_time = custom_flaps[atom]["critical"]
        else:
            stable_time = flap_check["stable"]
            critical_time = flap_check["critical"]

        subaggr_service = "subaggr_{}".format(atom)
        other_children.append(Child(
            host=aggr_host,
            service=subaggr_service,
        ))
        result_aggregates.append(get_classic_aggregate(
            aggr_host=aggr_host,
            aggr_service=subaggr_service,
            simple_children=[{"host": host, "service": atoms_format.format(atom)} for host in hosts],
            flaps_config={"stable": stable_time, "critical": critical_time},
            only_value=True,
        ))
        for host in hosts:
            other_children.append(Child(
                host=host,
                service=juggler_option("ontime_prefix") + atoms_format.format(atom),
            ))

    other_children += get_finish_scripts_children(hosts=hosts, scripts=juggler_option("universal_iron_scripts"))

    result_aggregates.append(get_classic_aggregate(
        aggr_host=aggr_host,
        aggr_service="hosts",
        notification_rules=notification_rules,
        simple_children=main_aggr_children,
        other_children=other_children,
        description="Агрегат системных атомов для '{}'".format(aggr_host),
    ))

    return result_aggregates


def get_finish_scripts_aggregate(hosts, scripts, aggr_host, aggr_service, notification_rules, at_least_one=False):
    """
    :param hosts: хосты
    :param scripts: метки скриптов, которые должны успешно завершиться на этих хостах
    :param aggr_host: координата 'host' соответствующего агрегата в Juggler-е (если возвращаем агрегат)
    :param aggr_service: координата 'host' соответствующего агрегата в Juggler-е (если возвращаем агрегат)
    :param notification_rules: правила нотификации (если возвращаем агрегат)
    :param at_least_one: правило, когда достаточно успешное завершение скрипта хотя бы на одном хосте среди заданных
    :return: Juggler-агрегат по скриптам, которые ожидаются выполниться на заданных хостах
    """
    children = get_finish_scripts_children(hosts, scripts)
    res_aggr = dict(
        juggler_option("check_template"),
        host=aggr_host,
        service=aggr_service,
        children=children,
        notifications=notification_rules,
        description="Успешное выполнение всех нужных скриптов для '{}'".format(aggr_host),
    )
    if at_least_one:
        res_aggr["aggregator"] = "logic_and"

    return res_aggr


def get_finish_scripts_children(hosts, scripts):
    """
    :param hosts: хосты
    :param scripts: метки скриптов, которые должны успешно завершиться на этих хостах
    :return: массив словарей сырых событий по заданным хостам и скриптам
    """
    children = []

    for host in hosts:
        for script in scripts:
            children.append(Child(
                host=host,
                service="{}.success_finish.irt".format(script)
            ))

    return children


def raw_events_by_selector(host_selector, service_selector):
    """
    :param host_selector: селектор по хосту сырых событий в Juggler-е
    :param service_selector: селектор по сервису сырых событий в Juggler-е
    :return: Child-объект агрегата в Juggler-е, созданный на основе сырых событий, попадающих под заданные селекторы
    """
    return Child(
        host="host={} & service={}".format(host_selector, service_selector),
        service="all",
        instance="all",
        group_type="EVENTS",
    )


def get_balancer_aggregate(balancer, standard_notifications, sms_logins=None):
    """
    :param balancer: имя балансера
    :param standard_notifications: стандартные правила нотификации в Telegram
    :param sms_logins: Логины на Стаффе, для которых нужны дополнительные нотификации по SMS (если это актуально)
    :return: агрегат активной проверки заданного балансера
    """
    notify_options = copy.deepcopy(standard_notifications)
    if sms_logins:
        notify_options.append(NotificationOptions(
            template_name="on_status_change",
            template_kwargs=dict(
                status=["OK", "WARN", "CRIT"],
                method=["sms"],
                login=sms_logins,
            ),
        ))

    return dict(
        host=balancer,
        service="https",
        refresh_time=90,
        ttl=900,
        active="https",
        active_kwargs=dict(port=443, warn_expire=30),
        description="Активная проверка балансера '{}'".format(balancer),
        notifications=notify_options,
    )
