import argparse
import logging
import os
from typing import List  # NOTE: Module dill can'n pickle whole module typing

import juggler_sdk

import irt.broadmatching.common_options
import irt.common.logging
import irt.monitoring.juggler.configs
import irt.monitoring.solomon.sensors


logger = logging.getLogger(__name__)


# Универсальная метка робота, с которой будут обновляться все наши агрегаты в рамках созданной Juggler-сессии.
# Агрегаты с другими метками не будут трогаться роботом.
IRT_JUGGLER_ROBOT_MARK = os.getenv("JUGGLER_PROD_MARK", "test_irt_checks")

# Неймспейс, в рамках которого робот обновляет агрегаты. По-умолчанию проставляется тестовый
NAMESPACE = os.getenv("JUGGLER_PROD_NAMESPACE", "irt.test")


def get_all_check_configs(
        prod_mode,           # type: bool
        notification_login,  # type: str
):                           # type: () -> List[juggler_sdk.Check]
    """
    Возвращает список всех актуальных агрегатов (проверок)
    :param prod_mode: флажок продового режима (True - если продовый, False - если тестовый)
    :param notification_login: кастомный логин для отправки нотификаций об агрегатах
    :return:
    """
    all_configs = []
    aggregates_set = set()

    for config_module in irt.monitoring.juggler.configs.get_all_configs():
        module_configs = config_module.get_checks()
        for conf in module_configs:
            aggregates_set.add((conf["host"], conf["service"]))

            conf["namespace"] = NAMESPACE
            if not prod_mode:
                conf["service"] = "__test__{}".format(conf["service"])

            if "notifications" in conf:
                if (isinstance(notification_login, str)) and (len(notification_login) > 0):
                    for notif_elem in conf["notifications"]:
                        notif_elem.template_kwargs["login"] = notification_login
                elif not prod_mode:
                    del conf["notifications"]

            all_configs.append(conf)

    if not prod_mode:
        for conf in all_configs:
            for child in conf.get("children", []):
                if (child.host, child.service) in aggregates_set:
                    child.service = "__test__{}".format(child.service)

    return [juggler_sdk.Check(**conf) for conf in all_configs]


def main():
    irt.common.logging.init_logging()
    argparser = argparse.ArgumentParser()

    argparser.add_argument("--dry_run", action="store_true", help="Juggler 'dry_run' for validation of a Juggler checks config")
    argparser.add_argument("--notification_login", type=str, help="Custom non-empty Telegram login for notification in Juggler checks")

    args = argparser.parse_args()
    notification_login = args.notification_login

    # Включенный "dry_run" позволяет не отправлять проверки на Juggler-сервер.
    # Просто проверяем корректность заданной конфигурации агрегатов. В "dry_run"-режиме токен не нужен.
    dry_run = args.dry_run

    # OAuth-токен клиента. По нему можно понять, кто именно ходит с запросами в Juggler.
    oauth_token = None

    # Флаг, который указывает на то, отправляем ли мы продовую конфигурацию агрегатов.
    prod_mode = False

    # В "dry_run"-режиме токен не нужен, а поднятый флаг prod_mode позволит проверить корректность примененения новой
    # конфигурации к текущей продовой (реально же на сервер Juggler-а ничего не отправится)
    if dry_run:
        prod_mode = True

    # На основе инициализированных двух переменных окружения, которые в проде ставятся по крону, понимаем, что это прод.
    elif ("JUGGLER_PROD_MARK" in os.environ) and ("JUGGLER_PROD_NAMESPACE" in os.environ):
        prod_mode = True
        try:
            tokens_folder = os.path.join(irt.broadmatching.common_options.get_options()["secrets"]["server_dir"], "tokens")
            token_file = os.path.join(tokens_folder, "juggler_oauth_token")
            with open(token_file, "r") as f:
                oauth_token = f.read().strip()
        except Exception as err:
            logger.exception("ERROR: failed with getting Juggler OAuth token: '%s'", err)
            exit(1)

    # Иначе ищем пользовательский OAuth-токен Juggler-а
    # [!!!] НЕ ИСПОЛЬЗУЙТЕ ТОКЕН РОБОТА В ОТЛАДОЧНЫХ ЦЕЛЯХ (в противном случае Вы рискуете перезатереть прод)!
    elif "JUGGLER_OAUTH_TOKEN" in os.environ:
        oauth_token = os.getenv("JUGGLER_OAUTH_TOKEN")

    # Иначе падаем с ошибкой, потому что без токена и с опущенным флагом "dry_run" API Juggler-а не работает.
    else:
        logger.error("[!] Please, add your OAUTH-token. Don't use robot token in debug mode!")
        exit(1)

    # Конфигурируем в рамках текущей Juggler-сессии все наши агрегаты
    with juggler_sdk.JugglerApi(
            "http://juggler-api.search.yandex.net",
            oauth_token=oauth_token,
            dry_run=dry_run,
            mark=IRT_JUGGLER_ROBOT_MARK,
            downtime_changed=True
    ) as api:
        for check in get_all_check_configs(prod_mode, notification_login):
            result = api.upsert_check(check)
            update_mark = ""
            if result.changed:
                update_mark = " + UPDATE"
            logger.info("[UPSERT%s] host='%s'; service='%s'", update_mark, check.host, check.service)

    logger.info("SEND_JUGGLER_CHECKS_OK")
    solomon_client = irt.monitoring.solomon.sensors.SolomonAgentSensorsClient()
    solomon_client.set_success_script_finish("update_juggler_checks")


if __name__ == "__main__":
    main()
