# -*- coding: utf-8 -*-

import datetime
import holidays
import irt.logging

from irt.bannerland.options import get_option as get_bl_option
from collections import Counter
from irt.artmon import ArtmonAPI
from time import timezone

# Универсальный формат даты для перевода в строку в этом модуле
DATE_FORMAT = "%Y-%m-%d"

# Список праздников в РФ помимо стандартных праздников
EXTRA_HOLIDAYS = [
    "2021-12-31",
    "2022-03-07",
    "2022-05-02",
    "2022-05-03",
    "2022-05-10",
    "2022-06-13",
]

# Критические WOW-пороги в процентах для денег в разрезах по SimDistance/ContextType
MONEY_THRS = {
    "context_type": {
        7: 25,
        8: 30,
    },

    "sim_distance": {
        700025: 35,
        700001: 40,
        700050: 50,

        804002: 30,
        804050: 40,
        2: 80,
        800333: 80,
        800025: 95,
    },

    "night_or_holiday": {
        "sim_distance": 95,
        "context_type": 75,
    },
}


def is_holiday(dt):
    """
    Проверяет, является ли этот день прадничным (выходные не считаются)
    :param dt: datetime.datetime
    :return: True, если является, и False в противном случае
    """
    return (dt in holidays.CountryHoliday("Russia")) or (dt.strftime(DATE_FORMAT) in EXTRA_HOLIDAYS)


def get_main_dt(dt_now):
    """
    Возвращает главную дату для сравнения (точнее datetime c ней) на основе текущего datetime
    :param dt_now: текущий datetime
    :return: искомый главный datetime для сравнения
    """
    return (dt_now - datetime.timedelta(days=1)) if dt_now.hour < 2 else dt_now


def calculate_money_stat(rows, max_unixtime):
    """
    Считает количество денег, заработанных в разрезе по SimDistance, с начала суток до max_unixtime
    :param rows: временной ряд со статистикой по SD
    :param max_unixtime: unix-время в мс., до которого нужно производить вычисления
    :return: два Counter-объекта: первый - для ContextType, второй - для SimDistance
    """
    sd_counter = Counter()

    for row in rows:
        if "cost" not in row:
            continue
        if (max_unixtime is not None) and (row["utc"] > max_unixtime):
            continue

        sd = int(row["series_id"].split(",")[1])
        sd_counter[sd] += row["cost"]

    return sd_counter


def calculate_diff(curr_money, cmp_money):
    """
    Вычисляет денежный дифф WOW в процентах
    :param curr_money: текущее денежное значение
    :param cmp_money: денежное значение для сравнения
    :return: дифф в процентах
    """
    if cmp_money == 0:
        if curr_money == 0:
            diff = 0
        else:
            diff = 100
    else:
        diff = 100.0 * (curr_money - cmp_money) / cmp_money
    return diff


def get_wow_statistics(artmon_token):
    """
    Вычисляет в Артмоне денежную статистику week-of-week для наших ContextType-ов (7 и 8), а также SimDistnce-ов,
    входящих в их состав. Возвращает агрегированную статистику по WOW-диффу, а также того, мониторятся ли все SD.

    Подробную логику работы данного функционала можно прочитать здесь: https://st.yandex-team.ru/DYNSMART-1095#5ebe781aa0994838e3f9c111

    :param artmon_token: OAuth-токен для хождения в API Артмона
    :return tuple из 3-х элементов: словарь диффов, метка мониторинга всех SD, ISO-timestamp
    """
    log = irt.logging.getLogger(irt.logging.BANNERLAND_PROJECT, __name__)

    sim_distances_in_context_types = get_bl_option("sim_distances_in_context_types")
    non_monitor_sds = set(get_bl_option("non_monitor_sds"))

    # Обозначаем сравниваемые даты. Ищем ближайшую WOW-дату для сравнения (за последние 4 недели), чтобы она не была праздником
    dt_now = datetime.datetime.now()

    if dt_now.hour == 0:
        # В первом часу ночи ничего не делаем - ждем до 01:00 полную статистику за прошедшие сутки
        log.info("Waiting for full statistics for yesterday... Exit.")
        return

    main_dt = get_main_dt(dt_now)
    for week_ago in range(1, 5):
        cmp_dt = main_dt - datetime.timedelta(weeks=week_ago)
        if not is_holiday(cmp_dt):
            break

    # Формируем запрос для Артмона
    log.info("Comparing '%s' with '%s'", main_dt.strftime(DATE_FORMAT), cmp_dt.strftime(DATE_FORMAT))
    artmon_api = ArtmonAPI(artmon_token)
    req_res = artmon_api.do_request(
        main_start=main_dt,
        main_end=main_dt,
        cmp_start=cmp_dt,
        cmp_end=cmp_dt,
        timegroup="min",
        add_filters={"group_context_type": ["7", "8"]},
    )
    if req_res is None:
        log.error("Artmon request is failed!")
        return

    # Последние три точки во временном ряде (15 минут) иногда содержат неполную информацию, их не берём в счёт
    # Исключение - если мы считаем полную статистику за прошедшие сутки
    utc_arr = sorted(set(row["utc"] for row in req_res["items"]["rows"]))
    if dt_now.day == main_dt.day:
        max_unixtime = utc_arr[-4]
    else:
        max_unixtime = utc_arr[-1]
    iso_maxtime = (datetime.datetime.fromtimestamp(max_unixtime / 1000 + 300) + datetime.timedelta(0, timezone)).isoformat()

    # Считаем деньги в разрезе по SimDistance и ContextType
    main_sd_counter = calculate_money_stat(req_res["items"]["rows"], max_unixtime)
    cmp_sd_counter = calculate_money_stat(req_res["items"]["compared"], max_unixtime - 1000 * (main_dt - cmp_dt).total_seconds())

    main_ct_counter, cmp_ct_counter = Counter(), Counter()
    for ct, sd_list in sim_distances_in_context_types.items():
        for sd in sd_list:
            main_ct_counter[ct] += main_sd_counter[sd]
            cmp_ct_counter[ct] += cmp_sd_counter[sd]

    # Если среди обозначенных SD есть тот, который мы не мониторим
    monitor_all_sds = True
    artmon_sds = set(main_sd_counter) - non_monitor_sds

    thr_sd_diff = artmon_sds ^ (set(MONEY_THRS["sim_distance"]) - non_monitor_sds)
    if thr_sd_diff:
        log.info("Artmon SD list doesn't match with SDs in the threshold config! Diff: {}".format(", ".join(str(x) for x in sorted(thr_sd_diff))))
        monitor_all_sds = False

    ct_sd_diff = artmon_sds ^ (set(sim_distances_in_context_types[7] + sim_distances_in_context_types[8]) - non_monitor_sds)
    if ct_sd_diff:
        log.info("Artmon SD list doesn't match with our SD list for CTs! Diff: {}".format(", ".join(str(x) for x in sorted(ct_sd_diff))))
        monitor_all_sds = False

    # Вычисляем WOW-дифф для каждого SimDistance. Также проверяем дифф на критичность
    value_diff = {
        "sim_distance": dict(),
        "context_type": dict(),
    }
    for sd in artmon_sds:
        diff = calculate_diff(main_sd_counter[sd], cmp_sd_counter[sd])
        value_diff["sim_distance"][sd] = diff

    # Вычисляем WOW-дифф для каждого ContextType. Также проверяем дифф на критичность
    for ct in [7, 8]:
        diff = calculate_diff(main_ct_counter[ct], cmp_ct_counter[ct])
        value_diff["context_type"][ct] = diff

    return value_diff, monitor_all_sds, iso_maxtime


def evaluate_diff(value_diff):
    """
    Оценивает WOW-дифф в разрезе по SimDistance/ContextType
    :param value_diff: словарь со значениями WOW-диффа
    :return: словарь меток успешности диффов
    """
    dt_now = datetime.datetime.now()
    main_dt = get_main_dt(dt_now)
    ok_diff = {
        "sim_distance": dict(),
        "context_type": dict(),
    }

    nigth_or_holiday = False
    if is_holiday(main_dt) or ((dt_now.day == main_dt.day) and (dt_now.hour < 10)):
        nigth_or_holiday = True

    for sd in value_diff["sim_distance"]:
        if sd not in MONEY_THRS["sim_distance"]:
            continue
        threshold = MONEY_THRS["sim_distance"][sd]
        if nigth_or_holiday:
            threshold = MONEY_THRS["night_or_holiday"]["sim_distance"]
        ok_diff["sim_distance"][sd] = int(value_diff["sim_distance"][sd] > -threshold)

    for ct in [7, 8]:
        threshold = MONEY_THRS["context_type"][ct]
        if nigth_or_holiday:
            threshold = MONEY_THRS["night_or_holiday"]["context_type"]
        ok_diff["context_type"][ct] = int(value_diff["context_type"][ct] > -threshold)

    return ok_diff


def get_stat_solomon_sensors(value_diff, ok_diff, monitor_all_sds, iso_timestamp):
    """
    На основе агрегированной WOW-статистики из Артмона возвращает массив сенсоров для отправки в Соломон
    :param value_diff: словарь с численным WOW-диффом
    :param ok_diff: численные метки того, не критичен ли WOW-дифф
    :param monitor_all_sds: метка того, мониторятся ли все актуальные SimDistance
    :param iso_timestamp: строковый timestamp в формате ISO для отправки в Соломон
    :return: массив сенсоров Соломона
    """
    solomon_sensors = []
    sensor_template = {
        "cluster": "artmon",
        "service": "money_statistics",
    }

    # Сенсор того, мониторим ли мы все SimDistance
    solomon_sensors.append(dict(sensor_template, sensor="monitor_all_simdistances", value=int(monitor_all_sds)))

    # Формируем словарь вида: SimDistance -> сет ContextType-ов
    sd_dict = dict()
    for ct, sd_list in get_bl_option("sim_distances_in_context_types").items():
        for sd in sd_list:
            ct_set = sd_dict.get(sd, set())
            ct_set.add(ct)
            sd_dict[sd] = ct_set

    # Сенсоры статистики по SimDistance
    for sd in value_diff["sim_distance"]:
        for ct in sd_dict.get(sd, set()):
            sensor_labels = {
                "sim_distance": str(sd),
                "context_type": str(ct),
            }

            solomon_sensors.append(dict(
                sensor_template,
                sensor="sd_diff",
                value=value_diff["sim_distance"][sd],
                labels=sensor_labels,
                ts_datetime=iso_timestamp,
            ))
            solomon_sensors.append(dict(
                sensor_template,
                sensor="ok_sd_diff",
                value=ok_diff["sim_distance"][sd],
                labels=sensor_labels,
                ts_datetime=iso_timestamp,
            ))

    # Сенсоры статистики по ContextType
    for ct in value_diff["context_type"]:
        sensor_labels = {"context_type": str(ct)}

        solomon_sensors.append(dict(
            sensor_template,
            sensor="ct_diff",
            value=value_diff["context_type"][ct],
            labels=sensor_labels,
            ts_datetime=iso_timestamp,
        ))
        solomon_sensors.append(dict(
            sensor_template,
            sensor="ok_ct_diff",
            value=ok_diff["context_type"][ct],
            labels=sensor_labels,
            ts_datetime=iso_timestamp,
        ))

    return solomon_sensors
