# -*- coding: utf-8 -*-

import ast
import copy
import deepdiff
import json
import logging
import requests

import irt.bannerland.hosts

logger = logging.getLogger(__name__)

# Поля по-умолчанию Соломоновских алертов, которые важны для нас при обновлении
ALERT_REQUIRED_FIELDS = {"annotations", "delaySeconds", "description", "id", "name", "notificationChannels",
                         "periodMillis", "type", "groupByLabels"}


class SolomonAlertSync:
    def __init__(self, project="irt", oauth_token="", env="prestable", required_fields=ALERT_REQUIRED_FIELDS):
        self.api_url = self._get_solomon_api_url(env) + "/projects/{project}/alerts/".format(project=project)
        self.oauth_token = oauth_token
        self.required_fields = required_fields
        self.headers = {
            "Content-Type": "application/json;charset=UTF-8",
            "Authorization": "OAuth {}".format(oauth_token),
        }

    # Создаёт новый алерт в текущем проекте Соломона. Возвращает 0 в случае успеха, 1 - в случае неуспеха
    # В функции алерт не проверяется на уже существующий, наличие необходимых полей и т. д. (за нас это делает Соломон)
    def create_alert(self, alert_dict):
        res_code = 1

        try:
            req = requests.post(self.api_url, headers=self.headers, data=json.dumps(alert_dict))
            if req.status_code == 200:
                logger.info("Alert '%s' has been created.", alert_dict["id"])
                res_code = 0
            else:
                self._print_request_error(req)

        except Exception as err:
            logger.exception("[create_alert]: Fail with creating a new alert: '%s'", err)

        return res_code

    # Удаляет алерт из уже существующих. Возвращает 0 - в случае успеха, 1 - в случае неудачи
    def delete_alert(self, alert_id):
        res_code = 1

        if type(alert_id) is not str:
            logger.error("[delete_alert]: Bad input!")
            return res_code

        try:
            req = requests.delete(self.api_url + alert_id, headers=self.headers)
            if req.status_code == 204:
                logger.info("Alert '%s' has been deleted.", alert_id)
                res_code = 0
            else:
                self._print_request_error(req)

        except Exception as err:
            logger.exception("[delete_alert]: fail for alert '%s': '%s'", alert_id, err)

        return res_code

    # Обновляет алерт. Возвращает 0 - в случае успеха, 1 - в случае неудачи
    def update_alert(self, main_alert, updated_params):
        res_code = 1

        if (type(main_alert) is not dict) or (type(updated_params) is not dict):
            logger.error("[update_alert]: Bad input!")
            return res_code

        alert_id = main_alert.setdefault("id", None)
        if (type(alert_id) is not str) or (("id" in updated_params) and (updated_params["id"] != alert_id)):
            logger.error("[update_alert]: Invalid alert id!")
            return res_code

        updated_alert = dict(main_alert)
        for key in updated_params.keys():
            updated_alert[key] = updated_params[key]

        try:
            req = requests.put(self.api_url + alert_id, headers=self.headers, data=json.dumps(updated_alert))
            if req.status_code == 200:
                logger.info("Alert '%s' has been updated.", alert_id)
                res_code = 0
            else:
                self._print_request_error(req)

        except Exception as err:
            logger.exception("[update_alert]: fail for alert '%s': '%s'", alert_id, err)

        return res_code

    # Возвращает сет из id-шников всех алертов текущего проекта. Либо None, если что-то пошло не так
    def get_all_alerts(self):
        try:
            req = requests.get(self.api_url + '?pageSize=2147483646', headers=self.headers)
            if req.status_code == 200:
                data = req.json()
                ids = set()
                if "items" not in data:
                    return ids
                for alert_item in data["items"]:
                    ids.add(str(alert_item["id"]))
                return ids
            else:
                self._print_request_error(req)
        except Exception as err:
            logger.error("[get_all_alerts]: Fail '%s'", err)

        return None

    # Возвращает информацию об алерте по его id. К сожалению, API для получения всех алертов не предоставляет всех
    # спецификаций алертов, их можно получить для каждого алерта отдельно другим API.
    # Возвращает словарь со спецификациями алерта в случае успеха или None в случае неуспеха.
    def get_alert_by_id(self, alert_id):

        if type(alert_id) is not str:
            logger.error("[get_alert_by_id]: Bad input!")
            return None

        try:
            req = requests.get(self.api_url + alert_id, headers=self.headers)
            if req.status_code == 200:
                return ast.literal_eval(req.text)
            else:
                self._print_request_error(req)
        except Exception as err:
            logger.exception("[get_alert_by_id]: Fail '%s'", err)

        return None

    # Обновляет все алерты в рамках текущего проекта, то есть делает актуальными алерты из словаря "new_alerts"
    # alerts_list - это список словарей, каждый из которых описывает отдельный алерт с его спецификациями.
    # Функция возвращает 0, если успех, 1 - если что-то пошло не так.
    def update_all_alerts(self, alerts_list):
        res_code = 0

        if type(alerts_list) is not list:
            logger.error("[update_all_alerts]: Bad input!")
            return 1

        # 1. Для удобства создаем словарь вида: "alert_id" -> alert_dict, где alert_dict - словарь алерта.
        # Также создаем сет из id-шников актуального массива алертов и проверяем их на уникальность
        new_ids = set()
        upd_alerts = dict()
        for alert in alerts_list:
            if (type(alert) is not dict) or ("id" not in alert):
                logger.error("[update_all_alerts]: Bad alert specification!")
                return 1
            if alert["id"] in new_ids:
                logger.error("[update_all_alerts]: alert id '%s' is not unique!", alert["id"])
                return 1
            new_ids.add(alert["id"])
            upd_alerts[alert["id"]] = alert

        # 2. Получаем id-шники тех алертов, что сейчас в Соломоне
        current_ids = self.get_all_alerts()
        if current_ids is None:
            logger.error("[update_all_alerts]: Can't get current project alerts")
            return 1

        # 3. Удаляем алерты, которых нет в upd_alerts, но которые существовали до этого
        for alert_id in (current_ids - new_ids):
            res = self.delete_alert(alert_id)
            if res != 0:
                logger.error("[update_all_alerts]: Can't delete alert '%s'", alert_id)
                res_code = 1

        # 4. Создаем алерты, которые не существовали до этого, но появились в upd_alerts
        for alert_id in (new_ids - current_ids):
            res = self.create_alert(upd_alerts[alert_id])
            if res != 0:
                logger.error("[update_all_alerts]: Can't create alert '%s'", alert_id)
                res_code = 1

        # 5. Проверяем спецификации уже существующих алертов и обновляем те из них, что изменились
        for alert_id in (current_ids & new_ids):
            cur_alert = self.get_alert_by_id(alert_id)
            if self._is_alerts_eq(cur_alert, upd_alerts[alert_id]):
                continue
            # Старые спецификации алерта cur_alert нужны, так как для обновления требуются некоторые служебные поля
            res = self.update_alert(cur_alert, upd_alerts[alert_id])
            if res != 0:
                logger.error("[update_all_alerts]: Can't update alert '%s'", alert_id)
                res_code = 1

        return res_code

    # По имени окружения возвращает урл для API (подробнее - https://wiki.yandex-team.ru/solomon/api/push/ )
    def _get_solomon_api_url(self, env):
        env_url_dict = {
            "production": "http://solomon.yandex.net/api/v2",
            "prestable": "http://solomon-prestable.yandex.net/api/v2",
            "testing": "http://solomon-test.yandex.net/api/v2",
        }
        return env_url_dict.setdefault(env, None)

    # Печатает сообщение об ошибке, пришедшей после неудачного обращения в API Соломона (когда код возврата не 200)
    def _print_request_error(self, req):
        logger.error("Solomon alerting API-request has been failed with code '%s'", req.status_code)
        try:
            response = req.json()
            if "message" in response:
                logger.error('%s', response["message"])
        except Exception as exception:
            logger.exception("Unknown error %s", exception)

    # Сравнивает алерты на основе тех полей, которые для нас важны в алертах. True - если равны, False - если нет.
    def _is_alerts_eq(self, alert1, alert2):
        clear_alert1 = dict((key, value) for key, value in alert1.items() if key in self.required_fields)
        clear_alert2 = dict((key, value) for key, value in alert2.items() if key in self.required_fields)
        return deepdiff.DeepDiff(clear_alert1, clear_alert2) == dict()


def juggler_annotations(**kwargs):
    """
    :param kwargs: параметры аннотаций Solomon-алерта для Juggler-а.
    :return: словарь аннотаций в нужном формате.
    """
    juggler_dict = {"juggler_{}".format(key): value for key, value in kwargs.items()}
    if isinstance(juggler_dict["juggler_tags"], list):
        juggler_dict["juggler_tags"] = ", ".join(x for x in juggler_dict["juggler_tags"])
    return juggler_dict


def solomon_ts(**kwargs):
    """
    :param kwargs: параметры сенсора.
    :return: словарь временного ряда сенсора.
    """
    req_fields = set(["cluster", "service", "sensor"])
    if len(req_fields) != len(req_fields & set(kwargs.keys())):
        logger.error("No enough fields for Solomon sensor withing an alert creating.")
        return

    ts = copy.deepcopy(kwargs)
    if "project" not in ts:
        ts["project"] = "irt"

    return ts


def str_solomon_ts(solomon_ts):
    """
    Преобразует словарь временного ряда в строку, понятную Соломону. # Порядок ключей в записи важен, так как алерт
    по сенсору с ключами, записанными в другом порядке, тратит излишние накладные ресурсы по переделыванию того же алерта
    в рамках синхронизации.
    :param solomon_ts: Python-словарь Solomon-овского временного ряда (в простейшем случае - сенсора)
    :return: строку в нужном формате
    """
    return "{{{ts}}}".format(ts=", ".join("{}='{}'".format(key, solomon_ts[key]) for key in sorted(solomon_ts.keys())))


def get_threshold_predicates(thr, status):
    """
    Формирует предикат(ы) для Соломоновского алерта типа threshold
    :param thr: заданный диапазон значений (точка или луч) в строковом формате. Либо массив этих строк.
    Например: "5", ["3.14-", "100+"], "!0".
    [!] Учтите, что порядок имеет значение. Чем первее в массиве, тем выше приоритет предиката.
    :param status: тип Соломон-предиката ("ALARM", "WARN", "OK")
    :return: предикат в формате Python-словаря
    """
    if isinstance(thr, list):
        thr_arr = thr
    else:
        thr_arr = [thr]

    predicates = []
    for raw_thr_item in thr_arr:
        thr_item = str(raw_thr_item)

        if thr_item[-1] == "+":
            value = float(thr_item[:-1])
            comparison = "GTE"
        elif thr_item[-1] == "-":
            value = float(thr_item[:-1])
            comparison = "LTE"
        elif thr_item[0] == "!":
            value = float(thr_item[1:])
            comparison = "NE"
        else:
            value = float(thr_item)
            comparison = "EQ"

        predicates.append({
            "thresholdType": "LAST_NON_NAN",
            "comparison": comparison,
            "threshold": value,
            "targetStatus": status,
        })

    return predicates


def get_value_alert(**kwargs):
    """
    Возвращает алерт по значению метрики.
    :param id: Идентификатор алерта. Обязательный аргумент. id каждого алерта должен быть уникальным!
    :param name: Имя алерта. Обязательный аргумент. Строка в свободном формате.
    :param type: Тип алерта. Обязательный аргумент. Одно из двух значений: "threshold" или "expression".
    :param solomon_ts: Временной ряд Соломона. Обязательный аргумент для threshold-алерта. Python-словарь.
    :param ok_thr, warn_thr, crit_thr: предикат(ы) для OK/WARN/CRIT порогов соответственно. Обязательный аргумент для threshold-алерта.
    :param expression: выражение для проверки в синтаксисе Solomon-алерта. Обязательный аргумент для expression-алерта.
    :param check_expression: проверочное выражение для expression-алерта. Необязательный аргумент, если предикат - уже в поле "expression"
    :param ontime_interval: интервал окна, в котором вычисляется алерт (соответствует интервалу протухания метрики). Обязательный аргумент.
    :param juggler_annotations: Juggler-конфигурация соотвествующего сырого события.
    :param notification_channels: массив каналов нотификации. По-умолчанию - ["Juggler"].
    :param delay_seconds: допустимое расширение в прошлое временного окна алерта. По-умолчанию - 0.
    :param description: Описание алерта. Необязательный аргумент. Строка в свободном формате (по-умолчанию - пустая строка).
    :return: Python-словарь искомого алерта.
    """
    for req_field in ["id", "name", "type", "juggler_annotations"]:
        if req_field not in kwargs:
            logger.error("No '%s' field for value-alert configuration!", req_field)
            return

    if len(kwargs["id"]) > 63:
        logger.error("Too long alert id '%s' (length = %s > 63)", kwargs["id"], len(kwargs["id"]))
        return

    if kwargs["type"] == "threshold":
        if ("crit_thr" not in kwargs) and ("warn_thr" not in kwargs) and ("ok_thr" not in kwargs):
            logger.error("Not enough alert params for threshold configuration.")
            return
        if "solomon_ts" not in kwargs:
            logger.error("There is no Solomon time-series for threshold alert.")
            return

        alert_subconf = {
            "selectors": str_solomon_ts(kwargs["solomon_ts"]),
            "timeAggregation": "LAST_NON_NAN",
            "predicateRules": [],
        }
        for conf_key, status in [("crit_thr", "ALARM"), ("warn_thr", "WARN"), ("ok_thr", "OK")]:
            if conf_key in kwargs:
                alert_subconf["predicateRules"].extend(get_threshold_predicates(kwargs[conf_key], status))
        alert_subconf["predicate"] = alert_subconf["predicateRules"][0]["comparison"]
        alert_subconf["threshold"] = alert_subconf["predicateRules"][0]["threshold"]

    elif kwargs["type"] == "expression":
        if "expression" not in kwargs:
            logger.error("There is no Solomon expression text for expression alert.")
            return
        alert_subconf = {
            "program": kwargs["expression"],
            "checkExpression": kwargs.get("check_expression", ""),
        }

    else:
        logger.error("Unknown type for value alert.")
        return

    value_alert = {
        "id": kwargs["id"],
        "name": kwargs["name"],
        "periodMillis": kwargs["ontime_interval"] * 1000,
        "delaySeconds": kwargs.get("delay_seconds", 0),
        "description": kwargs.get("description", ""),
        "notificationChannels": kwargs.get("notification_channels", ["Juggler"]),
        "annotations": kwargs["juggler_annotations"],
        "type": {kwargs["type"]: alert_subconf},
    }
    modify_groups_and_annots(value_alert, kwargs)

    return value_alert


def get_ontime_alert(**kwargs):
    """
    Возвращает алерт по протуханию. [!!!] Словарь параметров алерта kwargs должен быть идентичен параметрам алерта по значению.
    Все необходимые меодификации (с id, name и т. д. алерта происходят в рамках этой функции)
    :param id: Идентификатор алерта. Обязательный аргумент.
    :param name: Имя алерта. Обязательный аргумент. Строка в свободном формате.
    :param ontime_interval: Время протухания метрики в секундах. Обязательный аргумент.
    :param solomon_ts: Временной ряд Соломона. Обязательный аргумент. Python-словарь.
    :param juggler_annotations: Juggler-конфигурация соотвествующего сырого события. (!!! должна быть идентична Juggler-конфигурации для алерта по значению)
    :param description: Описание алерта. Необязательный аргумент. Строка в свободном формате (по-умолчанию - пустая строка).
    :return: Python-словарь искомого алерта.
    """
    for req_field in ["id", "name", "ontime_interval", "solomon_ts", "juggler_annotations"]:
        if req_field not in kwargs:
            logger.error("No '%s' field for on-time alert configuration!", req_field)
            return

    ontime_alert_id = "__ontime__" + kwargs["id"]
    if len(ontime_alert_id) > 63:
        logger.error("Too long alert id '%s' (length = %s > 63)", ontime_alert_id, len(ontime_alert_id))
        return

    annotations = copy.deepcopy(kwargs["juggler_annotations"])
    annotations["juggler_service"] = "__ontime__." + annotations["juggler_service"]

    ontime_alert = {
        "id": "__ontime__" + kwargs["id"],
        "name": "[On-Time] " + kwargs["name"],
        "periodMillis": kwargs["ontime_interval"] * 1000,
        "delaySeconds": 0,
        "description": "[ONTIME] " + kwargs.get("description", ""),
        "notificationChannels": ["Juggler_Ontime"],
        "annotations": annotations,
    }
    modify_groups_and_annots(ontime_alert, kwargs, consider_grouping_by_dc=False)

    ontime_alert["type"] = {
        "expression": {
            "program": "let ts = group_lines('count', {});\n alarm_if(count(ts) == 0);".format(str_solomon_ts(kwargs["solomon_ts"])),
            "checkExpression": "",
        },
    }

    return ontime_alert


def get_universal_alerts(**kwargs):
    """
    Возвращает универсальную конфигурацию для алерта по заданным параметрам.
    :param kwargs: параметры алертирования.
    :return: Массив из двух алертов. Первый алерт - для значения метрики, второй - по протуханию.
    """
    # Проверяем наличие обязательных полей
    for req_field in ["id", "name", "type", "solomon_ts", "ontime_interval"]:
        if req_field not in kwargs:
            logger.error("No '%s' field for alert configuration!", req_field)
            return

    value_alert = get_value_alert(**kwargs)
    ontime_alert = get_ontime_alert(**kwargs)
    if (not value_alert) or (not ontime_alert):
        return

    return [value_alert, ontime_alert]


def modify_groups_and_annots(alert, kwargs, consider_grouping_by_dc=True):
    """
    Модифицирует словарь алерта по параметрам группировки и, если нужно, аннотациям
    :param alert: словарь алерта
    :param kwargs: сырые параметры алерта
    :param consider_grouping_by_dc: учитывать ли ДЦ в группировке алерта
    """
    grouped_labels = set(kwargs["group_by"]) if "group_by" in kwargs else set()

    dc_prefix = "a_geo_"
    juggler_host = alert["annotations"].get("juggler_host")
    if consider_grouping_by_dc and (juggler_host == "{{labels.host}}"):
        grouped_labels.add("datacenter")
        alert["annotations"]["datacenter"] = dc_prefix + "{{labels.datacenter}}"
    else:
        dc = irt.bannerland.hosts.get_host_datacenter(juggler_host)
        if dc:
            alert["annotations"]["datacenter"] = dc_prefix + dc

    # Через настройки канала мы не можем прокинуть id нужного алерта (для обычных и мультиалертов он имеет разный смысл).
    # Закрепляем нужный идентификатор в аннотациях для обоих видов алертов
    alert["annotations"]["alert_id"] = "{{alert.id}}"

    if grouped_labels:
        alert["groupByLabels"] = sorted(grouped_labels)
