# -*- coding: utf-8 -*-

import logging

from irt.bannerland.hosts import get_hosts_by_role
from irt.monitoring.solomon.alerts.alert_client import solomon_ts as sts, get_universal_alerts, juggler_annotations
from irt.monitoring.atoms.config import get_atoms_configuration


logger = logging.getLogger(__name__)


def get_alerts():
    """
    :return: массив алертов для атомов
    """
    alerts = []
    atom_alerts_ontime = 3600
    alert_thr_keys = set(["crit_thr", "warn_thr", "ok_thr"])

    # Поскольку некоторые конфигурации атомов повторяются в цикле по label-ам, отсекаем дупликаты алертов
    uniq_atom_alerts = dict()

    # Алерты по классическим атомам (результатом работы которых есть чиселка, отправляющаяся в соответствующий временной ряд)
    for atom_conf in get_atoms_configuration():
        if "alert_id" not in atom_conf:
            continue
        if len(set(atom_conf.keys()) & alert_thr_keys) == 0:
            continue

        # Извлекаем поля группировки, если таковые имеются
        grouped_labels = set(atom_conf.get("group_by", []))

        # Инициализируем временной ряд Solomon-а (или их совокупность)
        solomon_sensor_params = atom_conf["solomon_sensor_params"]
        solomon_ts = {key: solomon_sensor_params[key] for key in ["cluster", "service", "sensor"]}
        solomon_ts["project"] = solomon_sensor_params.get("project", "irt")
        for label, val in solomon_sensor_params.get("labels", dict()).items():
            if label in grouped_labels:
                continue
            solomon_ts[label] = val

        # Проверяем, что мы не добавляем дважды алерт с одинаковым id и идентичной конфигурацией
        alert_tuple = tuple(
            ["__ts__" + solomon_ts[key] for key in ["project", "cluster", "service", "sensor"]] +
            ["__label__" + lbl for lbl in sorted(solomon_ts.get("labels", dict()).keys())] +
            ["__group_by__" + grpb for grpb in sorted(grouped_labels)]
        )
        if atom_conf["alert_id"] not in uniq_atom_alerts:
            uniq_atom_alerts[atom_conf["alert_id"]] = alert_tuple
        elif uniq_atom_alerts[atom_conf["alert_id"]] != alert_tuple:
            logger.error("Two (or more) different atom alerts, but with equal id='%s'", atom_conf["alert_id"])
            return
        else:
            continue

        # Инициализируем пороговые значения алерта, а также группировку (если она есть)
        extra_alert_params = {key: atom_conf[key] for key in alert_thr_keys if key in atom_conf}
        if grouped_labels:
            extra_alert_params["group_by"] = sorted(grouped_labels)

        # Инициализируем хост сырого события в Juggler-е
        if "host" in grouped_labels:
            juggler_host = "{{labels.host}}"
        else:
            hosts = set()
            if "include_host_fqdns" in atom_conf:
                hosts |= set(atom_conf["include_host_fqdns"])
            if "include_host_roles" in atom_conf:
                for host_role in atom_conf["include_host_roles"]:
                    hosts |= set(get_hosts_by_role(host_role))
            if "exclude_host_fqdns" in atom_conf:
                hosts -= set(atom_conf["exclude_host_fqdns"])
            if "exclude_host_roles" in atom_conf:
                for host_role in atom_conf["exclude_host_roles"]:
                    hosts -= set(get_hosts_by_role(host_role))

            if len(hosts) == 1:
                juggler_host = hosts.pop()
            else:
                logger.error("Can't define host for Juggler raw event")
                return

        if len(juggler_host) > 192:
            logger.error("Too much length of Juggler host '%s' (length = %s)", juggler_host, len(juggler_host))
            return

        # Инициализируем сервис сырого события в Juggler-е, а также идентификатор алерта
        juggler_service = format(atom_conf["alert_id"])
        labels_substr = ".".join("{{{{labels.{}}}}}".format(x) for x in sorted(grouped_labels) if x != "host")
        if labels_substr:
            juggler_service += ".{}".format(labels_substr)
        juggler_service += ".atoms.irt"

        if len(juggler_service) > 128:
            logger.error("Too much length of Juggler service '%s' (length = %s)", juggler_service, len(juggler_service))
            return

        # Добавляем алерты по атомы
        alerts += get_universal_alerts(
            id="atom_{}".format(atom_conf["alert_id"]),
            name="[Atom] {}".format(atom_conf["alert_id"]),
            solomon_ts=solomon_ts,
            type="threshold",
            ontime_interval=atom_alerts_ontime,
            juggler_annotations=juggler_annotations(
                host=juggler_host,
                service=juggler_service,
                tags=["atoms"],
            ),
            **extra_alert_params
        )

    # Время прошедшее с последнего обновления кода
    alerts += get_universal_alerts(
        id="atom_svn_hours_ago",
        name="[Atom] svn_hours_ago",
        solomon_ts=sts(cluster="host_info", service="arcadia", sensor="svn_hours_ago"),
        group_by=["host"],
        type="threshold",
        warn_thr="240+",
        ontime_interval=atom_alerts_ontime,
        juggler_annotations=juggler_annotations(
            host="{{labels.host}}",
            service="svn_hours_ago.atoms.irt",
            tags=["atoms"],
        ),
    )

    return alerts
