# -*- coding: utf-8 -*-

from irt.monitoring.solomon.alerts.alert_client import solomon_ts as sts, get_universal_alerts, get_ontime_alert, juggler_annotations, str_solomon_ts

EXPORT_OFFERS_CRIT_THRESHOLDS_SECONDS = {
    "dyn": 6 * 3600,  # more, than in perf because of site-crawlers and content-filters
    "perf": 3 * 3600,
}


def get_alerts():
    """
    :return: массив алертов Bannerland-а, кроме тех, что алертируются в стандартных атомах (результаты которых - чиселки)
    """
    alerts = []
    atom_alerts_ontime = 900

    # tskvs
    alerts += get_universal_alerts(
        id="tskvs",
        name="[Bannerland] tskvs",
        solomon_ts=sts(
            cluster="yt_hahn",
            service="bannerland",
            sensor="tskvs",
        ),
        type="threshold",
        warn_thr="40+",
        crit_thr="50+",
        ontime_interval=atom_alerts_ontime,
        group_by=["host", "days_interval", "task_type"],
        juggler_annotations=juggler_annotations(
            host="{{labels.host}}",
            service="{}_tskvs_{}.bannerland.irt".format("{{labels.task_type}}", "{{labels.days_interval}}"),
            tags=["tskvs_{}".format("{{labels.days_interval}}"), "bannerland", "{{labels.task_type}}"],
        ),
    )

    # dyn_bs_log
    alerts += get_universal_alerts(
        id="dyn_bs_log_last_success_days_ago",
        name="[Bannerland] dyn_bs_log (last_success_days_ago)",
        solomon_ts=sts(
            cluster="yt_hahn",
            service="dyn_bs_log",
            sensor="last_success_days_ago",
        ),
        type="threshold",
        warn_thr="4+",
        crit_thr="5+",
        ontime_interval=atom_alerts_ontime,
        juggler_annotations=juggler_annotations(
            host="yt_hahn",
            service="dyn_bs_log.last_success_days_ago.bannerland.irt",
            tags=["dyn_bs_log", "bannerland"],
        ),
    )
    alerts += get_universal_alerts(
        id="dyn_bs_log_row_count",
        name="[Bannerland] dyn_bs_log (row_count)",
        solomon_ts=sts(
            cluster="yt_hahn",
            service="dyn_bs_log",
            sensor="row_count",
        ),
        type="threshold",
        warn_thr="100000000-",
        crit_thr="50000000-",
        ontime_interval=atom_alerts_ontime,
        juggler_annotations=juggler_annotations(
            host="yt_hahn",
            service="dyn_bs_log.row_count.bannerland.irt",
            tags=["dyn_bs_log", "bannerland"],
        ),
    )

    # Мониторим ли мы все SimDistance
    alerts += get_universal_alerts(
        id="monitor_all_simdistances",
        name="[Artmon] Monitoring for all SimSistances",
        solomon_ts=sts(
            cluster="artmon",
            service="money_statistics",
            sensor="monitor_all_simdistances",
        ),
        type="threshold",
        crit_thr="0",
        ontime_interval=3 * 3600,
        juggler_annotations=juggler_annotations(
            host="artmon_stat",
            service="monitor_all_simdistances.irt",
            tags=["artmon_stat"],
        ),
    )

    # Мониторинг денег в Artmon-е в разрезе по SimDistance и ContextType
    for slice_short, slice_full, grouped_labels in [
        ("ct", "context_type", ["context_type"]),
        ("sd", "sim_distance", ["context_type", "sim_distance"]),
    ]:
        alerts += get_universal_alerts(
            id="{}_diff_ok".format(slice_short),
            name="[Artmon] Is OK for '{}' diff".format(slice_full),
            solomon_ts=sts(
                cluster="artmon",
                service="money_statistics",
                sensor="ok_{}_diff".format(slice_short),
            ),
            type="threshold",
            crit_thr="0",
            ontime_interval=3 * 3600,
            group_by=grouped_labels,
            juggler_annotations=juggler_annotations(
                host="artmon_stat",
                service="{slice_full}_{{{{labels.{slice_full}}}}}.diff_ok.irt".format(slice_full=slice_full),
                tags=["artmon_stat", "ok_diff", slice_full],
            ),
        )

    # Monitoring for actuality of taskoffers export from BL-iron hosts to YT
    for task_type, ontime_interval in EXPORT_OFFERS_CRIT_THRESHOLDS_SECONDS.items():
        alerts.append(get_ontime_alert(
            id="{}_bannerland_export_offers_actuality".format(task_type),
            name="[{}] Actuality for export taskoffers from each of iron Bannerland host".format(task_type.capitalize()),
            ontime_interval=ontime_interval,
            solomon_ts=sts(
                cluster="yt_hahn",
                service="bannerland",
                sensor="last_update_export_offers",
                task_type=task_type,
            ),
            group_by=["host", "prod_type"],
            juggler_annotations=juggler_annotations(
                host="{{labels.host}}",
                service="actuality.export_offers.{}.{{{{labels.prod_type}}}}.irt".format(task_type),
                tags=["export_offers", "bannerland"],
            ),
        ))

    # Monitoring of non-critical size for BL tasks queues
    for task_type in ["dyn", "perf"]:
        time_series = {
            "cluster": "host_info",
            "service": "bannerland",
            "sensor": "task_waiters",
            "task_type": task_type,
        }
        alerts += get_universal_alerts(
            id="{}_tasks_queue_size".format(task_type),
            name="[{}] Сriticality of tasks queue size".format(task_type.capitalize()),
            ontime_interval=9000,
            type="expression",
            expression="let queue_size = {};".format(str_solomon_ts(time_series)),
            check_expression="sum(queue_size) == 0",
            solomon_ts=time_series,
            group_by=["host"],
            juggler_annotations=juggler_annotations(
                host="{{labels.host}}",
                service="{}.tasks_queue.irt".format(task_type),
                tags=["tasks_queue", "bannerland"],
            ),
        )

    return alerts
