# -*- coding: utf-8 -*-

from irt.monitoring.solomon.alerts.alert_client import get_universal_alerts, get_value_alert, get_ontime_alert, juggler_annotations, str_solomon_ts


def get_ts_template():
    """
    :return: настройки для Соломона
    """
    return {
        "project": "irt",
        "cluster": "yt_hahn",
        "service": "bannerland_yt",
    }


def get_alerts():
    """
    :return: массив алертов по генерации Bannerland-а в YT
    """
    return get_alerts_for_task_type("dyn") + \
           get_alerts_for_task_type("perf") + \
           get_archive_worker_success()


def get_alerts_for_task_type(task_type):
    """
    :return: массив алертов по генерации Bannerland-а в YT для заданного типа тасок
    """
    alerts = []
    ts_template = get_ts_template()

    # Make banners
    alerts += get_universal_alerts(
        id="{}_make_banners".format(task_type),
        name="[Bannerland-YT] [{}] make_banners".format(task_type),
        solomon_ts=dict(ts_template, sensor="{}.make_banners.error_in_last_pocket".format(task_type)),
        type="threshold",
        crit_thr="!0",
        ontime_interval=1800,
        notification_channels=["Juggler"],
        juggler_annotations=juggler_annotations(
            host="yt_hahn",
            service="make_banners.{}.bannerland_yt.irt".format(task_type),
            tags=["make_banners", "bannerland", "bannerland_yt", task_type, "yt"],
            dashboard="monitor_yt_tables",
        ),
    )

    # avatars delay
    ontime_interval = 1800
    ts_avatars_delay = dict(ts_template, sensor="{}.prepare_input.avatars_cache_hours_old".format(task_type))
    alert_expr = "let ava_d = {};".format(str_solomon_ts(ts_avatars_delay))
    alerts.append(get_value_alert(
        id="{}_avatars_delay".format(task_type),
        name="[Bannerland-YT] [{}] Avatars delay".format(task_type),
        type="expression",
        expression=alert_expr,
        check_expression="last(ava_d) > 5",
        ontime_interval=ontime_interval,
        juggler_annotations=juggler_annotations(
            host="yt_hahn",
            service="avatars_delay.{}.bannerland_yt.irt".format(task_type),
            tags=["avatars_delay", "bannerland", "bannerland_yt", task_type, "yt"],
            dashboard="monitor_yt_tables",
        ),
    ))
    alerts.append(get_ontime_alert(
        id="{}_avatars_hours_old".format(task_type),
        name="Sensor '{}'".format(ts_avatars_delay["sensor"]),
        ontime_interval=ontime_interval,
        solomon_ts=ts_avatars_delay,
        juggler_annotations=juggler_annotations(
            host="yt_hahn",
            service="{}.irt".format(".".join(x for x in reversed(ts_avatars_delay["sensor"].split(".")))),
            tags=["avatars_delay", "bannerland", "bannerland_yt", task_type, "yt"],
            dashboard="monitor_yt_tables",
        ),
    ))

    # /put success
    ts_avatars_total = dict(ts_template, sensor="{}.avatars.put_requests_count".format(task_type))
    ts_avatars_success = dict(ts_template, sensor="{}.avatars.put_responses_success_count".format(task_type))
    alert_expr = "let ava_total = {}; \nlet ava_success = {};".format(
        str_solomon_ts(ts_avatars_total), str_solomon_ts(ts_avatars_success)
    )
    alerts.append(get_value_alert(
        id="{}_avatars_put_success_ratio".format(task_type),
        name="[Bannerland-YT] [{}] Avatars put success ratio".format(task_type),
        type="expression",
        expression=alert_expr,
        check_expression="sum(tail(ava_success, 6h)) / sum(tail(ava_total, 6h)) < 0.1",
        ontime_interval=12 * 3600,
        juggler_annotations=juggler_annotations(
            host="yt_hahn",
            service="avatars_put.{}.bannerland_yt.irt".format(task_type),
            tags=["avatars", "bannerland", "bannerland_yt", task_type, "yt"],
            dashboard="monitor_yt_tables",
        ),
    ))

    # make_banners_last_archive_alert
    if task_type == "perf":
        last_archive_thr = 6
    elif task_type == "dyn":
        last_archive_thr = 6
    else:
        raise ValueError("Task_type {} not supported".format(task_type))

    alerts += get_universal_alerts(
        id="{}_make_banners_last_archive".format(task_type),
        name="[Bannerland-YT] [{}] make_banners: last archive".format(task_type),
        type="threshold",
        solomon_ts=dict(ts_template, sensor="{}.make_banners.last_archive_hours_old".format(task_type)),
        crit_thr="{}+".format(last_archive_thr),
        ontime_interval=3600,
        delay_seconds=30,
        juggler_annotations=juggler_annotations(
            host="yt_hahn",
            service="last_archive_make_banners.{}.bannerland_yt.irt".format(task_type),
            tags=["last_archive", "bannerland", "bannerland_yt", task_type, "yt"],
            dashboard="monitor_yt_tables",
        ),
    )

    # Full State
    if task_type == "perf":
        ontime_interval = 20 * 3600
        critical_thresholds = {
            "phrases_count": "650000000-",
            "domains_count": "3500-",
            "orders_count": "6500-",
            "tasks_count": "20000-",
            "children_count": "80000000-",
        }
    elif task_type == "dyn":
        ontime_interval = 20 * 3600
        critical_thresholds = {
            "phrases_count": "850000000-",  # 0.85b
            "domains_count": "6000-",
            "orders_count": "20000-",
            "tasks_count": "90000-",
            "children_count": "200000000-",
        }
    else:
        raise ValueError("Task_type {} not supported".format(task_type))

    alerts.append(get_ontime_alert(
        id="bannerland_yt_full_state_{}_actuality".format(task_type),
        name="[{} Full state] Actuality".format(task_type),
        ontime_interval=ontime_interval,
        solomon_ts=dict(ts_template, sensor="{}.full_state.actuality".format(task_type)),
        juggler_annotations=juggler_annotations(
            host="yt_hahn",
            service="actuality.{}.full_state.bannerland_yt.irt".format(task_type),
            tags=["full_state", "bannerland", "bannerland_yt", task_type, "actuality", "yt"],
            dashboard="monitor_yt_tables",
        ),
    ))
    for metric, threshold in critical_thresholds.items():
        alerts.append(get_value_alert(
            id="bannerland_yt_{}_full_state_{}".format(task_type, metric),
            name="[{} Full state] {}".format(task_type, metric),
            solomon_ts=dict(ts_template, sensor="{}.full_state.{}".format(task_type, metric)),
            type="threshold",
            crit_thr=threshold,
            ontime_interval=ontime_interval,
            juggler_annotations=juggler_annotations(
                host="yt_hahn",
                service="{}.{}.full_state.bannerland_yt.irt".format(metric, task_type),
                tags=[metric, "full_state", "bannerland", "bannerland_yt", task_type, "yt"],
                dashboard="monitor_yt_tables",
            ),
        ))

    return alerts


def get_archive_worker_success():
    """
    :return: массив алертов по успешности завершения заданных архивариусов
    """

    alerts = []
    for cluster in ["yt_hahn", "yt_arnold"]:
        alerts += get_universal_alerts(
            id="{}_archive_worker_ok".format(cluster),
            name="[Bannerland-YT] Archive worker OK",
            type="threshold",
            solomon_ts={"project": "irt", "cluster": cluster, "service": "bannerland_yt", "sensor": "archive_worker_ok"},
            crit_thr=0,
            ontime_interval=3600 * 48,  # на всякий случай - вдруг сломается отправка в Solomon
            group_by=["archive_worker", "task_type", "cypress_mode"],
            juggler_annotations=juggler_annotations(
                host=cluster,
                service="{}.{}.{}.archive_worker_ok.bannerland_yt.irt".format("{{labels.task_type}}", "{{labels.archive_worker}}", "{{labels.cypress_mode}}"),
                tags=["bannerland", "bannerland_yt", "archive_worker_ok", "{{labels.archive_worker}}", "{{labels.cypress_mode}}"],
            ),
        )
    return alerts
