# -*- coding: utf-8 -*-

from irt.monitoring.solomon.alerts.alert_client import get_value_alert, juggler_annotations, str_solomon_ts


def get_alert_expr(ts, warn_threshold):
    """
    Возвращает expression-выражение для Соломон-алерта по актуальности завершения скрипта
    :param ts: словарь временного ряда Соломона
    :param warn_threshold: warning-порог в секундах для пришедшего значения
    :return: expression-выражение для Соломон-алерта, мониторящего актуальность завершения того или иного скрипта
    """
    return """
        let ts = {ts};
        alarm_if(count(ts) == 0);
        warn_if(time_interval_end() - last(get_timestamps(ts)) > {warn_threshold});
    """.format(ts=str_solomon_ts(ts), warn_threshold=warn_threshold)


def get_alerts():
    """
    Возвращает массив алертов для идентификации успешного завершения скриптов.
    :return: массив искомых алертов
    """
    alerts = []

    for script_name, warn_threshold, crit_threshold in [
        ("check-logs", 4, 5),
        ("collect-logs", 2, 3),
        ("find_banners_benchmark_categs_diff", 768, 780),
        ("get-resources", 24, 36),
        ("get_kwyt_source_url_title", 130, 150),
        ("get_log_percantage_slices", 36, 60),
        ("put-resources-to-sb", 18, 24),
        ("restart_services", 36, 48),
        ("sync_solomon_alerts", 1, 3),
        ("transfer-essential-tables", 12, 24),
        ("update_juggler_checks", 1, 2),
        ("update_service_tickets", 3, 6),
    ]:
        alerts.append(get_value_alert(
            id="{}_success_finish".format(script_name),
            name="[Success Finish Script] {}".format(script_name),
            type="expression",
            expression=get_alert_expr(
                ts=dict(
                    cluster="host_info",
                    service="scripts",
                    sensor="success_finish",
                    script_name=script_name,
                    host="*",
                    datacenter="*",
                ),
                warn_threshold=3600 * warn_threshold,
            ),
            check_expression="",
            ontime_interval=3600 * crit_threshold,
            juggler_annotations=juggler_annotations(
                host="{{labels.host}}",
                service="{}.success_finish.irt".format(script_name),
                tags=["success_finish_script", script_name],
            ),
            notification_channels=["Juggler_Ontime"],
            group_by=["host"],
        ))

    # collect-logs для каждой машинки по отдельности
    alerts.append(get_value_alert(
        id="collect_logs_from_host",
        name="[Collect Logs] Hosts separately",
        type="expression",
        expression=get_alert_expr(
            ts=dict(
                cluster="host_info",
                service="scripts",
                sensor="collect_logs_from_host",
            ),
            warn_threshold=3600 * 2,
        ),
        ontime_interval=3600 * 3,
        group_by=["host"],
        juggler_annotations=juggler_annotations(
            host="{{labels.host}}",
            service="collect_logs.separate_host.atoms.irt",
            tags=["collect_logs"],
        ),
        notification_channels=["Juggler_Ontime"],
    ))
    return alerts
