import argparse
import logging
import os

import yenv

import irt.broadmatching.common_options
import irt.common.logging
import irt.monitoring.solomon.alerts.alert_client
import irt.monitoring.solomon.alerts.configs
import irt.monitoring.solomon.sensors


logger = logging.getLogger(__name__)


def main():
    irt.common.logging.init_logging(use_json=False)

    # Парсинг аргументов командной строки
    argparser = argparse.ArgumentParser()
    argparser.add_argument("--prod", action="store_true", help="Use production mode.")
    argparser.add_argument("--force_notify_channels", action='store_true', help="[! PLEASE, BE CAREFUL !] Using this option doesn't remove alert notifying in debug mode.")
    args = argparser.parse_args()

    # Определяем режим работы скрипта (тестовый или продовый).
    # Для прода необходимо явное прописывание опции "--prod", а также отсутствие debug-значений в BM_USE_PATCH и DEBUG.
    debug_mode = yenv.type != yenv.STABLE or not args.prod

    # Получаем OAuth-токен. В проде - это токен робота, в debug-режиме - личный токен, указанный в env-переменной.
    # Наличие OAuth-токена обязательно! Без него Solomon не позволяет работать с API алертов (даже в prestable-режиме).
    if debug_mode:
        oauth_token = os.getenv("SOLOMON_OAUTH_TOKEN", "")
        if len(oauth_token) == 0:
            logger.error("Empty Solomon OAuth token. Please, specify it in 'SOLOMON_OAUTH_TOKEN' env-variable.")
            exit(1)
    else:
        try:
            tokens_folder = os.path.join(irt.broadmatching.common_options.get_options()["secrets"]["server_dir"], "tokens")
            token_file = os.path.join(tokens_folder, "solomon_oauth_token")
            with open(token_file, "r") as f:
                oauth_token = f.read().strip()
        except Exception as err:
            logger.exception("failed with getting Solomon OAuth token: '%s'", err)
            exit(1)

    # Получаем все алерты для нашего проекта. Идем в цикле по группам алертов в рамках конфига.
    project_alerts = []
    for config_module in irt.monitoring.solomon.alerts.configs.get_all_configs():

        alerts = config_module.get_alerts()
        if alerts is None:
            logger.error("Fail with getting alerts from '%s' configuration", config_module.__name__)
            exit(-1)
        else:
            project_alerts.extend(alerts)

    for alert_conf in project_alerts:
        if not isinstance(alert_conf, dict):
            logger.error("Invalid type of project alert: '%s'", alert_conf)
            exit(-1)

    # Если мы в debug-режиме, отключаем каналы нотификации, во избежание пересечения с продом
    if debug_mode and not args.force_notify_channels:
        for alert in project_alerts:
            if "notificationChannels" in alert:
                alert["notificationChannels"] = []

    # Создаем объект класса алерта (в зависимости от того, в какой среде мы находимся - продовой или тестовой)
    solomon_environment = "prestable"
    if not debug_mode:
        solomon_environment = "production"

    alert_obj = irt.monitoring.solomon.alerts.alert_client.SolomonAlertSync("irt", oauth_token, solomon_environment)
    if alert_obj.update_all_alerts(project_alerts) != 0:
        return

    logger.info("SYNC_ALERTS_OK")
    solomon_client = irt.monitoring.solomon.sensors.SolomonAgentSensorsClient()
    solomon_client.set_success_script_finish("sync_solomon_alerts")


if __name__ == "__main__":
    main()
