# -*- coding: utf-8 -*-

import datetime
import dateutil.parser
import logging
import requests
import time

import solomon
import yenv

import irt.bannerland.hosts


logger = logging.getLogger(__name__)


# Наш клиент для отправки сенсоров в Соломон
class SolomonSensorsClient:
    def __init__(self, project, cluster, service, force_test_mode=False):

        self.api_url = "http://solomon.yandex.net"
        if yenv.type != yenv.STABLE or force_test_mode:
            self.api_url = "http://solomon-prestable.yandex.net"

        # Интервал в секундах, в течение которого будут периодически отправляться сенсоры в рамках единого батча
        self.push_interval = 1

        # Лимит сенсоров на батч. Вообще Соломон допускает лимит до 10К сенсоров в батче, но для надежности поставим 5K
        self.batch_limit = 5000

        # Множитель, указывающий сколько раз по batch_limit мы спим, дожидаясь отправки батча в Соломон
        self.multiplier = 2.5

        # Инициализация клиента
        self.client = solomon.ThrottledPushApiReporter(
            project=project,
            cluster=cluster,
            service=service,
            url=self.api_url,
            push_interval=self.push_interval,
        )

        logger.info(
            "Use Solomon API Url '%s' for [project='%s' | cluster='%s' | service='%s']",
            self.api_url,
            project,
            cluster,
            service,
        )

    # Отправка сенсоров Соломона батчами (один push-запрос на один батч вместо единичного push-запроса на каждый сенсор)
    # sensors - массив сенсоров, каждый из которых - словарь {<имя сенсора>: {"value": X [, "labels": <Python dict>]}.
    def push_sensors(self, sensors):

        if (sensors is None) or (len(sensors) == 0):
            logger.info("No Solomon sensors for pushing...")
            return

        cc = 0
        for sensor in sensors:
            if ("ts_datetime" in sensor) and (type(sensor["ts_datetime"]) is not datetime.datetime):
                sensor["ts_datetime"] = dateutil.parser.parse(str(sensor["ts_datetime"]))
            self.client.set_value(**sensor)

            logger.info("Set Solomon sensor: [%s]", sensor)
            cc += 1

            # Если наш лимит превышен, тогда ждём, когда собранный батч отправится.
            if cc == self.batch_limit:
                time.sleep(self.multiplier * self.push_interval)
                cc = 0
        time.sleep(self.multiplier * self.push_interval)

        logger.info("End Solomon pushing.")


class SolomonAgentSensorsClient:
    """
    Класс для отправки сенсоров в Solomon-агент
    """

    def __init__(self, retries=3):
        """
        Инициализация объекта класса
        :param retries: количество попыток переотправки сенсоров в рамках одного запроса
        """
        self.retries = retries
        self.agent_port = 10060 if yenv.type != yenv.STABLE else 10050
        self.post_url = "http://localhost:{}".format(self.agent_port)

    def push_single_sensor(self, cluster, service, sensor, value, labels=None, ts_datetime=None, project="irt"):
        """
        Отправка только одного сенсора в агент.
        :param project: проект
        :param cluster: кластер
        :param service: сервис
        :param sensor: сенсор
        :param value: значение сенсора
        :param labels: словарь label-ов сенсора
        :param ts_datetime: строковый timestamp в iso-формате
        :return: 0 в cлучае успеха, 1 - в противном случае
        """

        return self.push_sensors([{
            "project": project,
            "cluster": cluster,
            "service": service,
            "sensor": sensor,
            "value": value,
            "labels": labels,
            "ts_datetime": ts_datetime,
        }])

    def push_sensors(self, sensors):
        """
        Отправка массива сенсоров в агент. Каждый сенсор - словарь с ключами (формат сенсора - в функции "push_single_sensor "выше)
        :param sensors: массив сенсоров
        :return: 0 в cлучае успеха, 1 - в противном случае
        """
        if (not isinstance(sensors, list)) or (len(sensors) == 0):
            logger.info("No Solomon sensors for pushing...")
            return 1

        converted_sensors = [self._convert_sensor_dict(single_sensor) for single_sensor in sensors]
        return self._push_agent_sensors(converted_sensors)

    def set_success_script_finish(self, script_name):
        """
        Отправляет в Соломон сенсор, символизирующий успешное завергение скрипта.
        :param script_name: имя скрипта
        """
        self.push_single_sensor(
            cluster="host_info",
            service="scripts",
            sensor="success_finish",
            value=1,
            labels={
                "host": irt.bannerland.hosts.get_curr_host(),
                "script_name": script_name,
            }
        )

    def _convert_sensor_dict(self, sensor):
        """
        Преобразовывает словарь сенсора из привычного нам формата в формат, воспринимаемый Соломон-агентом.
        Внутренняя функция класса.
        :param sensor: словарь сенсора в понятном нам формате (формат - в функции "push_single_sensor "выше)
        :return: словарь сенсора в понятном Solomon-агенту формате
        """
        if sensor.get("project") is None:
            sensor["project"] = "irt"

        converted_sensor = {
            "labels": {
                "project_id": sensor["project"],
                "cluster_id": sensor["cluster"],
                "service_id": sensor["service"],
                "sensor": sensor["sensor"],
            },
            "value": sensor["value"],
        }

        if sensor.get("ts_datetime") is not None:
            converted_sensor["ts"] = sensor["ts_datetime"]

        labels = sensor.get("labels")
        if (labels is not None) and (isinstance(labels, dict)):
            converted_sensor["labels"].update(labels)

        if ("datacenter" not in converted_sensor["labels"]) and ("host" in converted_sensor["labels"]):
            dc = irt.bannerland.hosts.get_host_datacenter(converted_sensor["labels"]["host"])
            if dc is not None:
                converted_sensor["labels"]["datacenter"] = dc

        return converted_sensor

    def _push_agent_sensors(self, sensors):
        """
        Отправка сенсоров (в формате Соломон-агента) в Соломон-агент.
        :param sensors: массив сенсоров в понятном Solomon-агенту формате
        :return: 0 в cлучае успеха, 1 - в противном случае
        """

        for retry_cc in range(self.retries):
            if retry_cc > 0:
                logger.info("Pushing to Solomon: attempt number %d...", retry_cc + 1)

            try:
                req = requests.post(
                    self.post_url,
                    json={"sensors": sensors},
                )

                if req.status_code == 200:
                    logger.info("OK: send sensors to agent: %s", sensors)
                    return 0
                else:
                    logger.error("Request to Solomon agent has been failed with code '%d'", req.status_code)

            except Exception as err:
                logger.error("Fail for sending sensors to Solomon agent: '%s'", err)

            time.sleep(1)
        return 1
