import argparse
import os
import warnings

from irt.multik import index_tables as it
from irt.multik.pylib.yt_orm.registry import registry
from sandbox.projects.release_machine.core import const as rm_const
import yt.wrapper as yt


ENV_CHOICES = [
    rm_const.ReleaseStatus.unstable,
    rm_const.ReleaseStatus.testing,
    rm_const.ReleaseStatus.prestable,
    rm_const.ReleaseStatus.stable,
]

DYN_TABLES_FOR_INTEGRATION_TESTS = '//home/multik/_integration_tests'
YT_DEFAULT_TABLES_DIR = '//home/multik/_default_tables'


def parse_arguments():
    parser = argparse.ArgumentParser(description='Create/symlink tables for new MuliK stand')
    parser.add_argument('--path-from', default='//home/multik/banners', help='Create symlinks for large tables from')
    parser.add_argument('--path-to', required=True, help='Path for new tables')
    parser.add_argument('--exists-ok', action='store_true', help='Do not fail if table already exists')
    parser.add_argument(
        '--environments',
        nargs='+',
        choices=ENV_CHOICES,
        default=ENV_CHOICES,
        help='List of environments to create tables for',
    )
    return parser.parse_args()


def copy_dyn_yt_table(src_path, dst_path):
    yt.unmount_table(src_path, sync=True)
    yt.copy(src_path, dst_path)
    yt.mount_table(src_path, sync=True)
    yt.mount_table(dst_path, sync=True)


def main():
    args = parse_arguments()

    if rm_const.ReleaseStatus.stable not in args.environments:
        warnings.warn('No stable environment specified, would still create symlinks to it')

    registry.set_yt_client(yt.YtClient(proxy=os.getenv('YT_PROXY')))

    for stage in args.environments:
        prefix = yt.ypath_join(args.path_to, stage)
        yt.mkdir(prefix, recursive=True)
        for table in [it.Index, it.Label, it.LabelTree]:
            table._table_prefix = prefix
            copy_dyn_yt_table(
                yt.ypath_join(YT_DEFAULT_TABLES_DIR, table._table_name),
                yt.ypath_join(args.path_to, stage, table._table_name),
            )

        if stage == rm_const.ReleaseStatus.stable:
            it.BannerLabels._table_prefix = prefix
            copy_dyn_yt_table(
                yt.ypath_join(YT_DEFAULT_TABLES_DIR, it.BannerLabels._table_name),
                yt.ypath_join(args.path_to, stage, it.BannerLabels._table_name),
            )
        else:
            target_path = yt.ypath_join(args.path_to, rm_const.ReleaseStatus.stable, it.BannerLabels._table_name)
            link_path = yt.ypath_join(prefix, it.BannerLabels._table_name)
            yt.link(target_path, link_path, ignore_existing=args.exists_ok)

        if stage == rm_const.ReleaseStatus.testing:
            for yt_table in yt.list(DYN_TABLES_FOR_INTEGRATION_TESTS):
                copy_dyn_yt_table(
                    yt.ypath_join(DYN_TABLES_FOR_INTEGRATION_TESTS, yt_table),
                    yt.ypath_join(args.path_to, stage, yt_table)
                )

        for table in [
            it.Banner,
            it.BannerData,
            it.BannerCID,
            it.BannerPID,
            it.BannerBID,
            it.BannerBannerID,
            it.BannerBannerhash,
        ]:
            target_path = yt.ypath_join(args.path_from, table._table_name)
            link_path = yt.ypath_join(prefix, table._table_name)
            yt.link(target_path, link_path, ignore_existing=args.exists_ok)


if __name__ == '__main__':
    main()
