import argparse
import json
import re
import shlex
import six
import subprocess

from library.python import resource
import jinja2

from sandbox.projects.release_machine.core import const as rm_const


STAGE_UNSTABLE = rm_const.ReleaseStatus.unstable
STAGE_TESTING = rm_const.ReleaseStatus.testing
STAGE_PRESTABLE = rm_const.ReleaseStatus.prestable
STAGE_STABLE = rm_const.ReleaseStatus.stable
STAGES_ALL = [STAGE_UNSTABLE, STAGE_TESTING, STAGE_PRESTABLE, STAGE_STABLE]
STAGES_DEFAULT = [STAGE_TESTING, STAGE_PRESTABLE, STAGE_STABLE]

ENV_TYPES = {
    STAGE_UNSTABLE: 'development',
    STAGE_TESTING: 'testing',
    STAGE_PRESTABLE: 'prestable',
    STAGE_STABLE: 'production',
}

MAN = 'man'
SAS = 'sas'
VLA = 'vla'

RULES = {
    'frontend': {'task_type': 'BUILD_MULTIK_SERVER_FRONTEND', 'layer_ref': 'multik_static'},
    'nginx': {'task_type': 'BUILD_MULTIK_NGINX', 'layer_ref': 'nginx_layer'},
    'webserver': {'task_type': 'BUILD_MULTIK_WEB_SERVER', 'layer_ref': 'multik_bin'},
    'worker': {'task_type': 'BUILD_MULTIK_WORKER', 'layer_ref': 'worker_bin_layer'},
}


def parse_arguments():
    parser = argparse.ArgumentParser(description='Deploy new MultiK stand in YDeploy')
    parser.add_argument('--stand-name', required=True, help='Name of new stand')
    parser.add_argument(
        '--stages', nargs='+', choices=STAGES_ALL, default=STAGES_DEFAULT, help='List of stages',
    )
    parser.add_argument('--dc-list', nargs='+', choices=[SAS, VLA, MAN], default=[SAS, VLA], help='List of datacenters')
    parser.add_argument(
        '--hostname', default='in.yandex-team.ru', help='Hostname postfix',
    )
    parser.add_argument(
        '--net-macro', default='_MULTIK_NETS_', help='Net macro for dev/stable/prestable stages',
    )
    parser.add_argument(
        '--testing-net-macro', default='_MULTIK_TEST_NETS_', help='Net macro for testing stage',
    )
    parser.add_argument(
        '--tvm2-client-id', default='2016155', help='Client id for dev/stable/prestable stages',
    )
    parser.add_argument(
        '--testing-tvm2-client-id', default='2020395', help='Client id for testing stage',
    )
    parser.add_argument(
        '--yav-secret',
        default='sec-01e34jcrgb5zbvae6f8cqy3rwx:ver-01e34jcrh1x3x1kd5shv91zd76',
        help='YAV secret with tokens. format: sec-...:ver-...',
    )
    parser.add_argument(
        '--tvm-secret',
        default='sec-01dp47g0s9f5wa9y1rasewss1j:ver-01e34j2se2eb5431jmbkhe6j99',
        help='Secret for TMV for dev/stable/prestable stages. format: sec-...:ver-...',
    )
    parser.add_argument(
        '--testing-tvm-secret',
        default='sec-01e8y925cqc34tdmcgwybtfs26:ver-01e8y925dmdxn77q4fybms3zwr',
        help='Secret for TMV for testing stage. format: sec-...:ver-...',
    )
    parser.add_argument(
        '--multik-secret-id',
        required=True,
        help='Secret with multik tokens and passwords. format: sec-...',
    )
    parser.add_argument('--db-host', required=True, help='Postgresql host to connect to')
    parser.add_argument('--yt-table-prefix', default='//home/multik', help='Path to MultiK YT tables')
    parser.add_argument('--graph-fallback-resource-id', required=True, help='Resource ID of fallback(default) graph')
    return parser.parse_args()


def run(command: str, stdin: str = None) -> bytes:
    proc = subprocess.Popen(
        shlex.split(command), stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.STDOUT,
    )
    out, _ = proc.communicate(six.ensure_binary(stdin) if stdin is not None else None)
    if proc.returncode:
        raise RuntimeError(
            'Command "{}" failed. return code: {}, output: {}'.format(command, proc.returncode, six.ensure_text(out))
        )
    # TODO(k-zaitsev): Change to logging
    print(six.ensure_str(out).strip())
    return out


def _parse_secret(secret: str):
    if not re.match('^sec-\w{26}:ver-\w{26}$', secret):
        raise ValueError('Secret {} should be in format "sec-\w{{26}}:ver-\w{{26}}"'.format(secret))
    _id, ver = secret.split(':')
    return {
        'alias': secret,
        'secret_id': _id,
        'secret_version': ver,
        'delegation_token': {},
    }


def main():
    args = parse_arguments()

    project_template = jinja2.Template(six.ensure_str(resource.find('yaml/project_template.yaml')))
    project_name = args.stand_name.lower()
    project_data = project_template.render(project_name=project_name)
    # TODO: Either change this to a programmatic call to YT or to API call
    run('ya tool dctl put project -', project_data)

    stage_template = jinja2.Template(six.ensure_str(resource.find('yaml/stage_template.yaml')))
    for stage in args.stages:
        hostname_parts = [stage] if stage != STAGE_STABLE else []
        hostname_parts += [project_name, args.hostname]
        api_hostname = '.'.join(hostname_parts)

        stage_id = project_name + '-' + stage
        secrets = {
            'yav': _parse_secret(args.yav_secret),
            'tvm': _parse_secret(args.testing_tvm_secret if stage == STAGE_TESTING else args.tvm_secret),
        }
        for key, secret in secrets.items():
            deploy_units = ['backend']
            if key == 'yav':
                deploy_units.append('monitoring')
            for deploy_unit in deploy_units:
                out = run(
                    'ya vault create token {secret_id} -tvm 2001151 -s {stage_id}.{deploy_unit} --json'.format(
                        secret_id=secret['secret_id'], stage_id=stage_id, deploy_unit=deploy_unit)
                )
                token = json.loads(out.strip())
                secret['delegation_token'][deploy_unit] = token['token']
        logrotate_conf = resource.find('conf/multik.logrotate')

        template_args = {
            'stage_id': stage_id,
            'project_name': project_name,
            'api_hostname': api_hostname,
            'secrets': secrets,
            'env_type': ENV_TYPES[stage],
            'net_macro': args.net_macro,
            'tvm2_client_id': args.tvm2_client_id,
            'dc_list': args.dc_list,
            'db_host': args.db_host,
            'multik_secret_id': args.multik_secret_id,
            'yt_table_prefix': args.yt_table_prefix,
            # NOTE(k-zaitsev): json.dumps gives us a double-quoted string with any
            # double-quotes escaped and newlines represented as \n. Exactly what we need for
            # raw config of deploy logrotate.
            'logrotate_conf': json.dumps(logrotate_conf.decode()),
            'graph_fallback_resource_id': args.graph_fallback_resource_id,
        }
        if stage == STAGE_TESTING:
            template_args['net_macro'] = args.testing_net_macro
            template_args['tvm2_client_id'] = args.testing_tvm2_client_id

        stage_data = stage_template.render(**template_args)
        run('ya tool dctl put stage -', stage_data)

        rule_template = jinja2.Template(six.ensure_str(resource.find('yaml/release_rule_template.yaml')))
        rule_data = rule_template.render(
            rule_id='{}-rule'.format(stage_id),
            stage_id=stage_id,
            release_type=stage,
            project_name=project_name,
        )
        run('ya tool dctl put release_rule -', rule_data)


if __name__ == '__main__':
    main()
