import urllib

from juggler import bundles
import irt.utils as irt_utils
import requests

from irt.multik.monitoring.checks import config
from irt.multik.monitoring.checks import npm_checks


def _http_check(path, auth=True):
    url = urllib.parse.SplitResult(
        scheme='https',
        netloc=config.get_config()['monitoring']['hostname'],
        path=path,
        query=None,
        fragment=None,
    ).geturl()

    headers = {}
    if auth:
        token = config.get_config()['secrets']['token']
        headers = {'Authorization': f'OAuth {token}'}

    resp = requests.get(url, headers=headers)
    if not resp.ok:
        return bundles.Event(bundles.Status.CRIT, f'Got {resp.status_code} status for {url}')
    data = resp.json()
    if not len(data):
        return bundles.Event(bundles.Status.CRIT, f'Got empty response, {resp.status_code} status for {url}')

    if 'items' in data and not data['items']:
        return bundles.Event(bundles.Status.CRIT, f'Got empty items, {resp.status_code} status for {url}')

    return bundles.Event(bundles.Status.OK, f'Got {resp.status_code} status for {url}')


@bundles.as_check(interval=60)
def api_ping():
    path = '/api/v1/ping/'
    return _http_check(path, auth=False)


# NOTE: juggler inspects function signature and expects parametrized checks to declare varargs
@bundles.as_check(interval=60)
def api_auth(*arguments):
    if not arguments:
        raise RuntimeError('Path for auth_http not provided')
    path = arguments[0]
    return _http_check(path)


def create_manifest():
    ri = irt_utils.ReleaseInfo()
    build_number = str(ri.major) if ri.minor is None else f'{ri.major}-{ri.minor}'
    manifest = bundles.Manifest(name='multik-pythonic-checks-bundle', build_number=build_number)
    manifest.register(api_ping)
    manifest.register(api_auth)
    manifest.register(npm_checks.smoke)
    manifest.register(npm_checks.infuse)
    return manifest


def main():
    bundles.cli(obj=create_manifest())


if __name__ == '__main__':
    main()
