import datetime
import os
import pwd
import shlex
import shutil
import subprocess
import time
import urllib

from juggler import bundles
import ylock
import yt.wrapper as yt

import irt.utils as irt_utils

from irt.multik.monitoring.checks import config


STALE_NPM_TIMEOUT = 10
USER = os.getenv('NPM_CHECKS_USER', 'monitor')
LOCATION = os.getenv('NPM_CHECKS_LOCATION', '/web_integration_tests')
YT_TABLE_PREFIX = os.getenv('YT_TABLE_PREFIX', '//home/multik/monitoring')

YT_POSTFIX = os.getenv('DEPLOY_STAGE_ID', 'multik')
YT_LOCK_NAME = os.getenv('YT_LOCK_NAME', f'infuse-lock-{YT_POSTFIX}')


def demote(user_uid, user_gid):
    def inner():
        os.setgid(user_gid)
        os.setuid(user_uid)

    return inner


@bundles.as_check(interval=600, timeout=120)
def infuse():
    lock_manager = ylock.create_manager(
        backend='yt', prefix=YT_TABLE_PREFIX, token=config.get_config()['secrets']['token']
    )
    with lock_manager.lock(YT_LOCK_NAME, block=False) as has_lock:
        if not has_lock:
            return bundles.CheckResult(events=[])

        client = yt.YtClient(
            proxy=os.getenv('YT_PROXY', 'locke'),
            token=config.get_config()['secrets']['token'],
        )
        attr_name = f'infuse-next-run-{YT_POSTFIX}'

        now = int(datetime.datetime.utcnow().strftime("%s"))
        next_run = client.get(YT_TABLE_PREFIX, attributes=(attr_name,)).attributes.get(attr_name, 0)

        force_skip = os.getenv('MULTIK_SKIP_LOCK_CHECK') == 'true'
        if not force_skip and next_run > now:
            return bundles.CheckResult(events=[])
        client.set(yt.ypath_join(YT_TABLE_PREFIX, f'@{attr_name}'), now + 500)

        command = 'npm run --prefix /web_integration_tests test-infuse-index'
        test_type = 'infuse'

        return _run_npm(command=command, test_type=test_type)


@bundles.as_check(interval=300, timeout=120)
def smoke():
    command = 'npm run --prefix /web_integration_tests smoke'
    test_type = 'smoke'
    return _run_npm(command=command, test_type=test_type)


def _run_npm(command, test_type):
    pwd_record = pwd.getpwnam(USER)
    env = os.environ.copy()
    env['PUPPETEER_IGNORE_HTTPS'] = 'true'
    env['MULTIK_TEST_HOST'] = urllib.parse.SplitResult(
        scheme='https',
        netloc=config.get_config()['monitoring']['hostname'],
        path='',
        query=None,
        fragment=None,
    ).geturl()
    env['MULTIK_TEST_USER_PASSWORD'] = config.get_config()['secrets']['password']
    results_path = os.path.join(pwd_record.pw_dir, f'{test_type}_results.json')
    env['JEST_RESULTS'] = results_path
    env['HOME'] = pwd_record.pw_dir

    start_time = int(time.time())
    proc = subprocess.Popen(
        shlex.split(command),
        preexec_fn=demote(pwd_record.pw_uid, pwd_record.pw_gid),
        stdout=subprocess.PIPE,
        stderr=subprocess.PIPE,
        cwd=pwd_record.pw_dir,
        env=env,
    )
    out, err = proc.communicate()

    try:
        results = irt_utils.from_json(results_path)
    except (FileNotFoundError, ValueError):
        results = None

    if proc.returncode:
        for name, data in [('out', out), ('err', err)]:
            fname = os.path.join(pwd_record.pw_dir, f'{test_type}.{name}')
            with open(fname, "w") as f:
                f.write(data.decode())

        if results is None:
            return bundles.Event(
                bundles.Status.CRIT,
                f'{test_type} tests failed with {proc.returncode} code. '
                f'Could not load {results_path}. out: {out}, err: {err}',
            )
        shutil.copy(results_path, f'{results_path}.failed')

        if results['success']:
            return bundles.Event(
                bundles.Status.CRIT,
                f'{test_type} tests failed with {proc.returncode} code, but jest reported success: {results["success"]}',
            )
        failed_tests = ', '.join(
            sorted(
                [
                    test['fullName']
                    for suite in results['testResults']
                    for test in suite['assertionResults']
                    if test['status'] == 'failed'
                ]
            )
        )

        return bundles.Event(
            bundles.Status.CRIT,
            f'{test_type} tests failed with {proc.returncode} code. Failed {len(failed_tests)} test(s): {failed_tests}.',
        )

    if results is None:
        return bundles.Event(
            bundles.Status.CRIT,
            f'{test_type} tests succeeded, but I could not load {results_path}. out: {out}, err: {err}',
        )

    npm_start_time = results['startTime'] / 1000
    if abs(start_time - npm_start_time) > STALE_NPM_TIMEOUT:
        shutil.copy(results_path, f'{results_path}.failed')
        return bundles.Event(
            bundles.Status.CRIT,
            f'{test_type} tests succeeded, but I have stale {results_path} file. '
            f'My start time vs npm start time is {start_time - npm_start_time:.2f}.',
        )

    if not results['numPassedTests']:
        shutil.copy(results_path, f'{results_path}.failed')
        return bundles.Event(bundles.Status.CRIT, f'{test_type} tests succeeded, but no tests were run')

    hostname = config.get_config()['monitoring']['hostname']
    return bundles.Event(
        bundles.Status.OK, f'{test_type} tests for {hostname} succeeded. {results["numPassedTests"]} test(s) passed'
    )
