import hashlib
import six


BANNERHASH_FIELDS = ('title', 'body', 'href')
BANNER_UID_FIELDS = ('bid', 'cid', 'pid')


def _get(obj, key):
    try:
        return getattr(obj, key)
    except (AttributeError, TypeError):
        try:
            return obj[key]
        except (KeyError, TypeError):
            raise ValueError('Could not get value for key={} for hash from {}'.format(key, obj))


def _hash(*fields):
    data = b''.join(force_bytes(field) for field in fields)
    return hashlib.md5(data).hexdigest()


def _extract_obj(keys, *args, **kwargs):
    if len(args) > 1:
        raise ValueError('Too many arguments for hash')
    elif len(args) == 1 and len(kwargs) != 0:
        raise ValueError('Passing argument with keywords arguments to hash')
    elif len(args) == 1:
        return args[0]
    elif not set(kwargs.keys()) >= set(keys):
        raise ValueError('Not enough values to compute hash. Need {{{}}}, got {{{}}}'.format(', '.join(keys), ', '.join(kwargs)))
    else:
        return kwargs


def banner_uid(*args, **kwargs):
    obj = _extract_obj(BANNER_UID_FIELDS, *args, **kwargs)
    return _hash(*(_get(obj, field_name) for field_name in BANNER_UID_FIELDS))


def bannerhash(*args, **kwargs):
    obj = _extract_obj(BANNERHASH_FIELDS, *args, **kwargs)
    return _hash(*(_get(obj, field_name) for field_name in BANNERHASH_FIELDS))


def force_text(s, encoding='utf-8', errors='strict'):
    """
    Similar to django force_text, but with no django-specific checks
    or django-specific esceptions
    """
    # Handle the common case first for performance reasons.
    if issubclass(type(s), six.text_type):
        return s
    if not issubclass(type(s), six.string_types):
        if six.PY3:
            if isinstance(s, bytes):
                s = six.text_type(s, encoding, errors)
            else:
                s = six.text_type(s)
        elif hasattr(s, '__unicode__'):
            s = six.text_type(s)
        else:
            s = six.text_type(bytes(s), encoding, errors)
    else:
        s = s.decode(encoding, errors)
    return s


def force_bytes(s, encoding='utf-8', strings_only=False, errors='strict'):
    """
    Similar to django force_bytes, but with no django-specific checks
    or django-specific esceptions
    """
    # Handle the common case first for performance reasons.
    if isinstance(s, bytes):
        if encoding == 'utf-8':
            return s
        else:
            return s.decode('utf-8', errors).encode(encoding, errors)
    if not isinstance(s, six.string_types):
        try:
            if six.PY3:
                return six.text_type(s).encode(encoding)
            else:
                return bytes(s)
        except UnicodeEncodeError:
            if isinstance(s, Exception):
                # An Exception subclass containing non-ASCII data that doesn't
                # know how to print itself properly. We shouldn't raise a
                # further exception.
                return b' '.join(force_bytes(arg, encoding, strings_only, errors)
                                 for arg in s)
            return six.text_type(s).encode(encoding, errors)
    else:
        return s.encode(encoding, errors)


__all__ = ['force_bytes', 'force_text', 'bannerhash', 'banner_uid']
