from abc import abstractmethod, ABCMeta
from six import add_metaclass, string_types

import irt.common.serialization_protocol as serialization_protocol


@add_metaclass(ABCMeta)
class KeyValueStore(serialization_protocol.SerializationProtocol):
    @abstractmethod
    def __init__(self, namespaces, keys_cols, data_cols):
        if not isinstance(namespaces, tuple):
            raise TypeError("namespaces is not a tuple")
        for namespace in namespaces:
            if not isinstance(namespace, string_types):
                raise TypeError("Namespace is not a string")

        if not isinstance(keys_cols, tuple):
            raise TypeError("keys_cols is not a tuple")
        if not keys_cols:
            raise ValueError("No keys columns")
        for keys_col in keys_cols:
            if not isinstance(keys_col, string_types):
                raise TypeError("Key column name is not a string")

        if not isinstance(data_cols, tuple):
            raise TypeError("data_cols is not a tuple")
        if not data_cols:
            raise ValueError("No data columns")

        for data_col in data_cols:
            if not isinstance(data_col, string_types):
                raise TypeError("Data column name is not a string")

        if set(keys_cols) & set(data_cols):
            raise ValueError('Data and key columns have intersection: {}'.format(', '.join(set(keys_cols) & set(data_cols))))

        self.namespaces, self.keys_cols, self.data_cols = namespaces, keys_cols, data_cols

    def validate_namespace(self, namespace):
        if not isinstance(namespace, tuple):
            raise TypeError("namespace is not a tuple")
        if len(namespace) > len(self.namespaces):
            raise ValueError('Wrong namespace')

    def validate_key(self, key):
        if not isinstance(key, tuple):
            raise TypeError("key is not a tuple")
        if len(key) != len(self.keys_cols):
            raise ValueError('Wrong key')

    @abstractmethod
    def upsert(self, namespace, key, data):
        self.validate_namespace(namespace)
        self.validate_key(key)

    @abstractmethod
    def read(self, namespace, key, column_list=None):
        self.validate_namespace(namespace)
        self.validate_key(key)

        return dict()

    @abstractmethod
    def update(self, namespace, key, data):
        self.validate_namespace(namespace)
        self.validate_key(key)

    @abstractmethod
    def delete(self, namespace, key):
        self.validate_namespace(namespace)
        self.validate_key(key)

    @abstractmethod
    def init_store(self, *args):
        pass

    @abstractmethod
    def connect_store(self):
        pass

    @abstractmethod
    def create_store(self, namespace_types, key_types, data_types):
        if len(namespace_types) != len(self.namespaces):
            raise ValueError('Size of namespace types not equal to namespaces count')
        if len(key_types) != len(self.keys_cols):
            raise ValueError('Size of key types not equal to keys count')
        if len(data_types) != len(self.data_cols):
            raise ValueError('Size of data types not equal to data columns count')

    @abstractmethod
    def drop_store(self):
        pass

    @classmethod
    def from_dict(cls, data, path=None):
        return super(KeyValueStore, cls).from_dict(data, path=path)

    def dict(self, *args):
        return super(KeyValueStore, self).dict({
            'namespaces': list(self.namespaces),
            'keys_cols': list(self.keys_cols),
            'data_cols': list(self.data_cols)
        }, *args)

    @abstractmethod
    def files(self):
        return []

    @abstractmethod
    def transaction(self):
        pass

    def Request(self, *args, **kwargs):
        instance = self

        class Request(object):
            """
            Class for request in KV-storage.

            All values in request validated and we don't need to check it in filter
            """
            def __init__(self, kv, keys, namespaces, column_list=None):
                """
                Construct Request object with validating `keys` and `namespaces` argument.

                :param kv: KV instance
                :param keys: iterable object with tuples of keys
                :param namespaces: iterable object with tuples of keys
                :param column_list: iterable object with names of columns
                """
                self.request = {}

                for n, k in zip(namespaces, keys):
                    kv.validate_namespace(n)
                    kv.validate_key(k)

                    if k not in self.request:
                        self.request[k] = set()

                    self.request[k].add(n)

                if column_list is None:
                    self.column_list = None
                else:
                    self.column_list = set(column_list)

        return Request(instance, *args, **kwargs)

    @abstractmethod
    def filter(self, request=None):
        """
        Iterate over KV with filter by keys and namespaces.

        :param request:
        :type request: KeyValueStore.Request
        """

        return
        yield

    @classmethod
    def migrate(cls, src, ns_types, key_types, data_types, init_store_args=(), request=None):
        kv = cls(src.namespaces, src.keys_cols, src.data_cols)
        kv.init_store(*init_store_args)

        kv.create_store(ns_types, key_types, data_types)
        kv.connect_store()

        def _ns(obj):
            return tuple(obj.pop(n) for n in src.namespaces)

        def _key(obj):
            return tuple(obj.pop(k) for k in src.keys_cols)

        for row in src.filter(request=request):
            kv.upsert(_ns(row), _key(row), row)

        return kv
