import logging

from .dyn_table import DynTableKeyValueStore, yt_wrapper
from . import error


log = logging.getLogger(__name__)


class ReplicatedDynTableKeyValueStore(DynTableKeyValueStore):
    class Client(object):
        def __init__(self, proxy_or_client, token):
            if isinstance(proxy_or_client, yt_wrapper.YtClient):
                self._client = proxy_or_client
                self._proxy = self._client.config['proxy']['url']
            else:
                self._proxy = proxy_or_client
                self._client = yt_wrapper.YtClient(self._proxy,
                                                   token,
                                                   config={
                                                       'backend': 'rpc',
                                                       'dynamic_table_retries': {'enable': False}})

            self._config_diff = ReplicatedDynTableKeyValueStore.find_config_diff(self._client.config)
            self._config_diff.pop('token', None)

        @property
        def client(self):
            return self._client

        @property
        def proxy(self):
            return self._proxy

        def __hash__(self):
            return id(self)

        def serialize(self):
            return self._config_diff

        @classmethod
        def deserialize(cls, data, token):
            return cls(yt_wrapper.YtClient(token=token, config=data), token)

    def __init__(self, *args, **kwargs):
        super(ReplicatedDynTableKeyValueStore, self).__init__(*args, **kwargs)

        self._writer = None
        self._readers = list()
        self._table_paths = {}

    def _select_rows(self, *args, **kwargs):
        return self._readers[0].client.select_rows(*args, **kwargs)

    def _lookup_rows(self, *args, **kwargs):
        return self._readers[0].client.lookup_rows(self.table_path(self._readers[0]), *args, **kwargs)

    def _insert_rows(self, data):
        return self._writer.client.insert_rows(self.table_path(self._writer), data, require_sync_replica=False, aggregate=True)

    def _delete_rows(self, *args, **kwargs):
        return self._writer.client.delete_rows(self.table_path(self._writer), *args, require_sync_replica=False, **kwargs)

    def _exists(self, *args, **kwargs):
        return self._writer.client.exists(self.table_path(self._writer)) and all(reader.exists(self.table_path(reader)) for reader in self._readers)

    def _mount_table(self, *args, **kwargs):
        self._writer.mount_table(self.table_path(self._writer),  *args, **kwargs)
        for reader in self._readers:
            reader.client.mount_table(self.table_path(reader), *args, **kwargs)

    def _unmount_table(self, *args, **kwargs):
        self._writer.unmount_table(self.table_path(self._writer),  *args, **kwargs)
        for reader in self._readers:
            reader.client.unmount_table(self.table_path(reader), *args, **kwargs)

    def _create(self, schema):
        writer_table_path = self.table_path(self._writer)
        log.info('Creatin table on master: %s', writer_table_path)

        self._writer.client.create(
            'replicated_table',
            writer_table_path,
            attributes={
                'dynamic': True,
                'schema': schema,
                'enable_replicated_table_tracker': True,
                'sync_replica_count': 1
            },
            recursive=True
        )

        for reader in self._readers:
            reader_table_path = self.table_path(reader)
            log.info('Creating table_replica for replica: %s.%s', reader.proxy, reader_table_path)
            replica_id = self._writer.client.create('table_replica',
                                                    attributes={'table_path': writer_table_path,
                                                                'cluster_name': reader.proxy,
                                                                'replica_path': reader_table_path})

            log.info('Creating table on replica: %s.%s with upstream %s', reader.proxy, reader_table_path, replica_id)
            reader.client.create('table', reader_table_path, attributes={
                'dynamic': True,
                'schema': schema,
                'upstream_replica_id': replica_id
            })
            log.info('alter_table_replica')
            self._writer.client.alter_table_replica(replica_id, True)

            log.info('Mounting table on replica')
            reader.client.mount_table(reader_table_path, sync=True)

        log.info('mount_table master')

        self._writer.client.mount_table(writer_table_path, sync=True)
        log.info('Creation done')

    def _remove(self, *args, **kwargs):
        self._writer.remove(self.table_path(self._writer))
        for reader in self._readers:
            reader.remove(self.table_path(reader))

    def transaction(self):
        if self._writer is None:
            log.error('YT client is None')
            raise RuntimeError('YT client is None')
        log.info('Start transaction')
        return self._writer.Transaction(type='tablet')

    def init_store(self, table_paths_proxies, token=None):
        table_path_proxy_list = []
        for table_path_proxy in table_paths_proxies:
            table_path, proxy = table_path_proxy
            table_path_proxy_list.append((table_path, proxy))

        if len(table_path_proxy_list) < 2:
            raise ValueError('Replicated table needs at least two tables')

        self.init_clients(table_path_proxy_list[0], table_path_proxy_list[1:])

    def init_clients(self, replicated_table_path_client_token, replica_table_paths_clients_tokens):
        if yt_wrapper is None:
            raise error.NoYtModuleException()
        if len(replicated_table_path_client_token) == 3:
            replicated_table_path, replicated_proxy_or_client, token = replicated_table_path_client_token
        elif len(replicated_table_path_client_token) == 2:
            token = None
            replicated_table_path, replicated_proxy_or_client = replicated_table_path_client_token
        else:
            raise ValueError('Wrong size of init_clients param {}'.format(replicated_table_path_client_token))

        self._writer = self.Client(replicated_proxy_or_client, token)
        self._add_table_path(self._writer, replicated_table_path)

        for path_proxy_token in replica_table_paths_clients_tokens:
            if len(replicated_table_path_client_token) == 3:
                path, proxy, token = path_proxy_token
            elif len(replicated_table_path_client_token) == 2:
                token = None
                path, proxy = path_proxy_token
            else:
                raise ValueError('Wrong size of init_clients param {}'.format(path_proxy_token))
            cl = self.Client(proxy, token)
            self._readers.append(cl)
            self._add_table_path(cl, path)

    @classmethod
    def from_dict(cls, data, path=None):
        kv = cls(tuple(data['namespaces']), tuple(data['keys_cols']), tuple(data['data_cols']))
        replicated_data = [(cls.Client.deserialize(data['writer'], None), data['writer_table_path'])]
        replica_data = [
            (cls.Client.deserialize(reader['reader'], None), reader['table_path']) for reader in data['readers']
        ]
        kv.init_store(replicated_data + replica_data, None)
        kv.connect_store()
        return kv

    def dict(self, *args):
        return super(DynTableKeyValueStore, self).dict({
            'writer': self._writer.serialize(),
            'writer_table_path': self.table_path(self._writer),
            'readers': [
                {'reader': reader.serialize(), 'table_path': self.table_path(self._writer)} for reader in self._readers
            ]
        }, *args)

    def _add_table_path(self, cl, path):
        self._table_paths[cl] = path

    def table_path(self, cl):
        return self._table_paths[cl]
