from contextlib import contextmanager
import logging
import os.path
from six.moves import zip_longest

from irt.utils import from_json

from .base import KeyValueStore
from . import error

logger = logging.getLogger(__name__)


class MemoryKeyValueStore(KeyValueStore):
    def __init__(self, namespaces, keys_cols, data_cols):
        super(MemoryKeyValueStore, self).__init__(namespaces, keys_cols, data_cols)

        self._data = {}
        self._path = None
        self._modified = False

    def upsert(self, namespace, key, data):
        super(MemoryKeyValueStore, self).upsert(namespace, key, data)

        if key not in self._data:
            self._data[key] = {}

        self._data[key][namespace] = data
        self._modified = True

    def read(self, namespace, key, column_list=None):
        super(MemoryKeyValueStore, self).read(namespace, key, column_list)

        if key not in self._data or len(self._data[key]) == 0:
            raise error.NoSuchNamespaceException(namespace, key)

        ns = tuple()
        data = {k: v for k, v in zip(self.keys_cols, key)}
        data.update({k: None for k in self.namespaces})

        for x in namespace:
            ns = ns + (x, )
            if ns in self._data[key]:
                data.update({k: v for k, v in zip(self.namespaces, ns)})
                data.update(self._data[key][ns])

        if column_list is not None:
            pass

        return data

    def filter(self, request=None):
        if request is None:
            logger.info('request is None, getting all data')
            for key, ns_storage in self._data.items():
                for ns, d in ns_storage.items():
                    d = dict({k: None for k in self.data_cols}, **d)
                    d.update({k: v for k, v in zip(self.keys_cols, key)})
                    d.update({k: v for k, v in zip_longest(self.namespaces, ns)})
                    yield d
        else:
            if request.column_list:
                logger.info('Requesting fields %s in document', request.column_list)
                key_cols = tuple(kc for kc in self.keys_cols if kc in request.column_list)
                namespaces = tuple(nc for nc in self.namespaces if nc in request.column_list)
                data_cols = tuple(request.column_list - set(key_cols + namespaces))
            else:
                logger.info('Requesting all fields in document')
                key_cols = self.keys_cols
                namespaces = self.namespaces
                data_cols = None

            for key in request.request:
                logger.debug('Processing key %s', key)
                if key not in self._data:
                    logger.debug('Key is not presented in storage')
                    continue

                for key_ns in request.request[key]:
                    logger.debug('Processing namespace %s', key_ns)
                    ns = tuple()
                    data = {k: v for k, v in zip(self.keys_cols, key) if k in key_cols}
                    data.update({k: None for k in (data_cols or self.data_cols)})
                    data.update({k: None for k in namespaces})

                    for x in key_ns:
                        ns = ns + (x, )
                        if ns in self._data[key]:
                            data.update({k: v for k, v in zip(self.namespaces, ns) if k in namespaces})
                            data.update({k: self._data[key][ns].get(k) for k in (data_cols or self._data[key][ns])})
                    yield data

    def update(self, namespace, key, data):
        super(MemoryKeyValueStore, self).update(namespace, key, data)

        if key not in self._data or namespace not in self._data[key]:
            raise error.NoSuchNamespaceException(namespace, key)

        self._modified = True
        self._data[key][namespace].update(data)

    def delete(self, namespace, key):
        super(MemoryKeyValueStore, self).delete(namespace, key)

        if key not in self._data or namespace not in self._data[key]:
            raise error.NoSuchNamespaceException(namespace, key)
        data = self._data[key].pop(namespace)
        data.update({k: v for k, v in zip(self.keys_cols, key)})
        data.update({k: v for k, v in zip_longest(self.namespaces, namespace)})

        self._modified = True
        return data

    def init_store(self, path):
        # TODO fix function, actually we not support tuples keys in JSON
        self._path = path
        if path is not None:
            self._data = from_json(path)

    def connect_store(self):
        pass

    def create_store(self, *args):
        pass

    def drop_store(self):
        self._data = {}

    @classmethod
    def from_dict(cls, data, path=None):
        kv = cls(tuple(data['namespaces']), tuple(data['keys_cols']), tuple(data['data_cols']))
        if 'path' in data and data['path'] is not None:
            kv.init_store(data['path'] if path is None else os.path.join(path, data['path']))
            kv.connect_store()
        elif 'path' in data:
            pass
        elif 'data' in data:
            for ns, key, d in data['data']:
                kv.upsert(tuple(ns), tuple(key), d)
        else:
            raise ValueError('Wrong format of data object. Keys in data: {}'.format(', '.join(data.keys())))

    def dict(self, *args):
        if self._modified or self._path is None:
            data = [
                (list(ns), list(key), self._data[key][ns]) for key in self._data for ns in self._data[key]
            ]

            return super(MemoryKeyValueStore, self).dict({
                'data': data
            }, *args)
        else:
            return super(MemoryKeyValueStore, self).dict({
                'path': self._path
            }, *args)

    def files(self):
        return [self._path] if self._path is not None else []

    @contextmanager
    def transaction(self):
        logger.warning('Transactions not supported in memory key value')
