import logging

import irt.utils
from http.server import BaseHTTPRequestHandler, HTTPServer
import json

DEFAULT_PORT = 8000

logger = logging.getLogger(__name__)


class LivenessServerHandler(BaseHTTPRequestHandler):
    def __init__(self, request, client_address, server):
        self._uri_map = {
            '/alive': self.alive,
            '/ping': self.ping,
            '/status': self.status,
            '/version': self.version,
        }
        self.error_content_type = 'application/json'
        self.error_message_format = '''{
    "code": "%(code)d",
    "message": "%(message)s",
    "explain": "%(explain)s"
}'''

        BaseHTTPRequestHandler.__init__(self, request, client_address, server)

    def _set_headers(self):
        self.send_response(200)
        self.send_header('Content-type', 'application/json')
        self.end_headers()

    @staticmethod
    def _encode(msg):
        if hasattr(msg, 'encode'):
            return msg.encode('utf8')
        return msg

    def _json(self, data):
        self._set_headers()
        self.wfile.write(self._encode(json.dumps(data)))

    def version(self):
        self._json({
            'version': irt.utils.get_version()
        })

    def ping(self):
        self._json({
            'ping': 'pong'
        })

    def check_alive(self):
        try:
            return self.server.liveness_callback()
        except Exception as e:
            logger.exception('Exception during checking alive status {}'.format(e))
            return False

    def alive(self):
        self._json({
            'alive': 'ok' if self.check_alive() else 'fail'
        })

    def status(self):
        self._json({
            'alive': 'ok' if self.check_alive() else 'fail',
            'version': irt.utils.get_version()
        })

    def error_404(self):
        self.send_error(404)

    def do_GET(self):
        self._uri_map.get(self.path, self.error_404)()


class LivenessServer(object):
    class _HTTPServer(HTTPServer):
        def __init__(self, liveness_callback, server_address, RequestHandlerClass):
            self.liveness_callback = liveness_callback
            HTTPServer.__init__(self, server_address, RequestHandlerClass)

    def __init__(self, liveness_callback=None, port=None, host=None):
        self._port = port or DEFAULT_PORT
        self._host = host or 'localhost'

        server_address = (self._host, self._port)
        callback = liveness_callback if callable(liveness_callback) else lambda: True

        self._httpd = self._HTTPServer(callback, server_address, LivenessServerHandler)

    def run(self):
        logger.info('Starting httpd server on {}:{}'.format(self._host, self._port))
        self._httpd.serve_forever()

    def shutdown(self):
        self._httpd.shutdown()
