# coding: utf-8
from django.contrib.auth.models import User
from django.db import models
from django.contrib.postgres.fields import JSONField


class Queue(models.Model):
    id = models.BigAutoField(primary_key=True)
    enqueued_at = models.DateTimeField()
    dequeued_at = models.DateTimeField(blank=True, null=True)
    expected_at = models.DateTimeField(blank=True, null=True)
    schedule_at = models.DateTimeField(blank=True, null=True)
    q_name = models.TextField()
    data = models.TextField()

    class Meta:
        managed = False
        db_table = 'queue'


class Operation(models.Model):
    STATUS_CREATED = "created"
    STATUS_SCHEDULED = "scheduled"
    STATUS_STARTED = "started"
    STATUS_IN_PROGRESS = "in_progress"
    STATUS_FINISHED = "finished"
    STATUS_CANCELED = "canceled"
    STATUS_ERROR = "error"
    STATUS_RESCHEDULED = "rescheduled"

    ALL_STATUSES = [STATUS_CREATED, STATUS_SCHEDULED, STATUS_STARTED, STATUS_IN_PROGRESS, STATUS_FINISHED, STATUS_CANCELED, STATUS_ERROR, STATUS_RESCHEDULED]
    FINAL_STATUSES = [STATUS_FINISHED, STATUS_ERROR, STATUS_CANCELED]

    TYPE_INFUSE = "infuse"
    TYPE_OTHER = "other"

    ALL_TYPES = [TYPE_INFUSE]

    user = models.ForeignKey(User, on_delete=models.PROTECT, null=True)
    operation_type = models.CharField(max_length=256, choices=[
        (TYPE_INFUSE, "Infuse"), (TYPE_OTHER, "Other"),
    ], default=TYPE_OTHER)
    operation_status = models.CharField(max_length=256, choices=[
        (STATUS_CREATED, "Created"),
        (STATUS_SCHEDULED, "Scheduled"),
        (STATUS_STARTED, "Started"),
        (STATUS_IN_PROGRESS, "In Progress"),
        (STATUS_FINISHED, "Finished"),
        (STATUS_CANCELED, "Canceled"),
        (STATUS_ERROR, "Error"),
        (STATUS_RESCHEDULED, "Rescheduled"),
    ], default=STATUS_CREATED)

    created_at = models.DateTimeField(auto_now_add=True, db_index=True)
    updated_at = models.DateTimeField(auto_now=True, db_index=True)
    started_at = models.DateTimeField(null=True, blank=True)
    queue = models.OneToOneField(Queue,
                                 on_delete=models.PROTECT, null=True)

    master_fqdn = models.TextField()
    master_host = models.TextField()
    worker = models.TextField(null=True)
    worker_host = models.TextField(null=True)
    worker_fqdn = models.TextField(null=True)

    # JSON fields for results and errors if any
    result = JSONField(null=True)
    errors = JSONField(null=True)

    def set_error(self, error, exception=None, tb=None):
        self.operation_status = Operation.STATUS_ERROR
        self.errors = self.errors or []
        description = {'job_id': self.queue_id, 'error': error}
        if exception:
            description['exception'] = {'type': type(exception).__name__,
                                        'text': str(exception),
                                        'error': repr(exception),
                                        'traceback': tb}
        self.errors.append(description)
        self.save()

    def set_result(self, result):
        self.result = result
        self.operation_status = Operation.STATUS_FINISHED
        self.save()

    def set_progress(self):
        self.operation_status = Operation.STATUS_IN_PROGRESS
        self.save()


class OperationMessage(models.Model):
    operation = models.ForeignKey(Operation, related_name='messages', on_delete=models.CASCADE)
    created_at = models.DateTimeField(auto_now_add=True, db_index=True)
    message = models.TextField()
    module_name = models.TextField()
    lineno = models.IntegerField(default=0)
