import logging
import os
import sys

from irt.utils import suppress
from pythonjsonlogger import jsonlogger

from .common import *  # noqa
from .common import LOG_LEVEL, get_db_name, POSTGRESQL_DB_PASSWORD

bin_name = sys.executable.split(os.path.sep)[-1]
LOG_PATH = os.getenv('LOG_PATH', '/var/log/multik/{}'.format(bin_name))

with suppress(OSError):
    os.makedirs(LOG_PATH)


class CustomJsonFormatter(jsonlogger.JsonFormatter):
    def add_fields(self, log_record, record, message_dict):
        super(CustomJsonFormatter, self).add_fields(log_record, record, message_dict)
        log_record['@timestamp'] = record.asctime
        log_record['@fields'] = {
            'module': record.module,
            'filename': record.filename,
            'lineno': record.lineno,
        }
        log_record['loggerName'] = record.name
        log_record['threadName'] = record.thread
        log_record['levelStr'] = record.levelname
        log_record['level'] = record.levelno
        if record.exc_text:
            log_record['stackTrace'] = record.exc_text


LOGGING = {
    'version': 1,
    'disable_existing_loggers': False,
    'formatters': {
        'verbose': {
            'format': '{levelname} {asctime} {name} {module} {process:d} {thread:d} {message}',
            'style': '{',
        },
        'json': {
            '()': CustomJsonFormatter,
            'format': '%(message)s'
        },

    },
    'handlers': {
        'debug-file': {
            'level': 'DEBUG',
            'class': 'logging.handlers.WatchedFileHandler',
            'formatter': 'verbose',
            'filename': os.path.join(LOG_PATH, 'debug.log'),
        },
        'error-file': {
            'level': 'ERROR',
            'class': 'logging.handlers.WatchedFileHandler',
            'formatter': 'verbose',
            'filename': os.path.join(LOG_PATH, 'error.log'),
        },
        'console': {
            'level': 'DEBUG',
            'class': 'logging.StreamHandler',
            'formatter': 'json',
            'stream': sys.stdout,
        },
    },
    'loggers': {
        'django': {
            'handlers': ['debug-file', 'error-file', 'console'],
            'level': LOG_LEVEL,
            'propagate': False,
        },
        'pq': {
            'handlers': ['debug-file', 'error-file', 'console'],
            'level': logging.ERROR,
            'propagate': False,
        },
        '': {
            'handlers': ['debug-file', 'error-file', 'console'],
            'level': LOG_LEVEL,
            'propagate': False,
        },
    },
}


DATABASES = {
    'default': {
        'ENGINE': 'django.db.backends.postgresql_psycopg2',
        'NAME': os.getenv('POSTGRESQL_DB_NAME_OVERRIDE', get_db_name()),
        'USER': os.getenv('POSTGRESQL_DB_USER'),
        'PASSWORD': POSTGRESQL_DB_PASSWORD,
        'HOST': os.getenv('POSTGRESQL_DB_HOST'),
        'PORT': os.getenv('POSTGRESQL_DB_PORT', 6432),
        'OPTIONS': {
            'target_session_attrs': 'read-write'
        }
    }
}

INSTALLED_APPS = [
    'django.contrib.contenttypes',
    'django.contrib.auth'
]

AUTH_USER_MODEL = 'auth.User'
