import collections

import inflection
import six

import yt.wrapper as yt

from . import fields
from . import manager
from .schema import Schema
from .registry import registry


def _get_default_table_prefix():
    try:
        from django.conf import settings
        if settings.configured:
            return settings.YT_TABLE_PREFIX
        else:
            raise ImportError
    except ImportError:
        from irt.multik.settings import django
        return django.YT_TABLE_PREFIX


class TableMetaclass(type):
    """Metaclass for Tables

    Inspects all Fields declared on a model and stores them in a ._fields object
    Adds .objects manager, just like Django ORM does

    Calls .contribute_to_class on any object that is being added to the table to give
    objects chance to customise themselves on creation. For example store a backreference
    to current Table class.
    """

    @classmethod
    def _get_declared_fields(cls, bases, attrs):
        my_fields = [(field_name, attrs.pop(field_name))
                     for field_name, obj in list(attrs.items())
                     if isinstance(obj, fields.Field)]
        # keep fields ordered the same way as they're declared in Table class
        my_fields.sort(key=lambda x: x[1]._creation_counter)
        return collections.OrderedDict(my_fields)

    @property
    def _table_path(cls):
        return yt.ypath_join(cls._table_prefix, cls._table_name)

    def __new__(cls, name, bases, attrs):
        my_fields = cls._get_declared_fields(bases, attrs)

        new_class = super(TableMetaclass, cls).__new__(cls, name, bases, attrs)
        for field_name, field in my_fields.items():
            new_class.add_to_class(field_name, field)

        new_class.add_to_class('_schema', Schema(my_fields))

        abstract = attrs.get('__abstract__', False)
        if not abstract:
            table_name = attrs.get('_table_name', inflection.underscore(new_class.__name__))
            registry.register_table(table_name, new_class)

            new_class.add_to_class('_table_prefix', attrs.get('_table_prefix', _get_default_table_prefix()))
            new_class.add_to_class('_table_name', table_name)

        # NOTE: Keeping 'objects' public, similar to django ORM
        new_class.add_to_class('objects', manager.Manager())
        # We need access to some fields of the model from many different places
        # Thus storing them all in a single well known place
        new_class.add_to_class('_fields', fields.FieldStore(my_fields))

        return new_class

    def add_to_class(cls, name, value):
        """Hook to perform custom initialization for classes added to our Table"""
        if hasattr(value, 'contribute_to_class'):
            value.contribute_to_class(cls, name)
        else:
            setattr(cls, name, value)


@six.add_metaclass(TableMetaclass)
class Table(object):
    """Base class for all yt_orm tables"""

    __abstract__ = True

    def __init__(self, **kwargs):
        # First init whatever set in kwargs
        all_fields = self._fields.all
        rogue_fields = {}
        ignored = getattr(self, '_ignore_rogue_fields', [])
        for field_name, value in kwargs.items():
            if field_name in all_fields:
                setattr(self, field_name, value)
            elif field_name in ignored:
                continue
            else:
                rogue_fields[field_name] = value
        if len(rogue_fields):
            raise ValueError('Could not init {}. Got rogue fields: {}'.format(self.__class__.__name__, rogue_fields))

        # All fields that remain recieve respective defaults
        default_fields = set(all_fields) - set(kwargs)
        for field_name in default_fields:
            field = self._fields.all[field_name]
            setattr(self, field_name, field.get_default())

    @classmethod
    def deserialize(cls, **kwargs):
        """Pass all kwargs through relevant .to_python field

        This method is used when deserializing objects retrieved from select queries.
        It allows pre-processing complex types, not supported by YT, like datetime.
        Will raise ValueError if it recieves extra fields, but will silently ignore missing fields
        """
        all_fields = cls._fields.all

        obj_kwargs = {
            field_name: all_fields[field_name].to_python(value)
            for field_name, value in kwargs.items()
            if field_name in all_fields
        }
        if len(obj_kwargs) != len(kwargs):
            ignored = getattr(cls, '_ignore_rogue_fields', [])
            rogue_fields = {
                field_name: value
                for field_name, value in kwargs.items()
                if field_name not in all_fields
                and field_name not in ignored
            }
            if len(rogue_fields):
                raise ValueError('Could not deserialize into {}. Got rogue fields: {}'.format(
                    cls.__name__, rogue_fields))
        return cls(**obj_kwargs)

    def _get_state(self):
        """Returns current object state. I.e. a mapping of field names into their values"""
        return collections.OrderedDict(
            (field_name, field.to_yt(getattr(self, field_name)))
            for field_name, field in self._fields.all.items()
        )

    def save(self):
        """Create or update the object"""
        state = self._get_state()
        return self.objects.save(state)

    def delete(self):
        ns = (field.to_yt(getattr(self, field_name))
              for field_name, field in self._fields.namespace.items())
        key = (field.to_yt(getattr(self, field_name))
               for field_name, field in self._fields.key.items())
        return self.objects.delete(tuple(ns), tuple(key))

    def __str__(self):
        fields = ['{}={}'.format(field_name, getattr(self, field_name)) for field_name in self._fields.all]
        return '{}({})'.format(
            self.__class__.__name__, ', '.join(fields))

    def update(self, **kwargs):
        if not (set(kwargs) < set(self._fields.all)):
            raise ValueError('Parameters {} for update not suitable for table {}'.format(', '.join(set(self._fields.all) - set(kwargs)), self.__class__.__name__))
        for key in kwargs:
            setattr(self, key, kwargs[key])

        return self

    __repr__ = __str__

    @classmethod
    def table_exists(cls):
        return registry.get_yt_client().exists(cls._table_path)

    @classmethod
    def create_table(cls, exists_ok=False):
        registry.get_yt_client().create('table', cls._table_path, attributes=cls._schema.yt_attributes, ignore_existing=exists_ok)

    @classmethod
    def mount_table(cls):
        registry.get_yt_client().mount_table(cls._table_path, sync=True)

    @classmethod
    def table_path(cls, suffix=''):
        return cls._table_path + suffix


class SecondaryIndexTableMetaclass(TableMetaclass):
    def __new__(cls, name, bases, attrs):
        abstract = attrs.get('__abstract__', False)
        if not abstract:
            destination_table = attrs.get('_destination_table', None)
            if destination_table is None or not issubclass(destination_table, Table):
                raise TypeError('Table {} does not contain valid destination_table {} attribute'.format(name, destination_table))

            my_fields = cls._get_declared_fields(bases, attrs.copy())
            pk = tuple(my_fields)

            for field in pk:
                if not my_fields[field].key:
                    raise TypeError('Field {} in index table {} is not key'.format(field, name))

            fk = tuple(destination_table._fields.key)

            if set(fk) & set(pk):
                raise TypeError('Duplication of key fields ({}) in {}'.format(set(fk) & set(pk), name))

            for field_name in fk:
                attrs[field_name] = destination_table._fields.key[field_name].copy()

            if '_stub' in attrs or '_pk' in attrs or '_fk' in attrs:
                raise TypeError('Creation of _stub, _pk, _fk attributes is forbidden in {} subclass of SecondaryIndexTable'.format(name))

            attrs['_stub'] = fields.CharField(default='')
            attrs['_pk'] = pk
            attrs['_fk'] = fk

            if registry.get_secondary_index(destination_table, pk) is not None:
                raise RuntimeError('Such index table with pk {} for {} already registered'.format(pk, destination_table))

            table_path = attrs.get('_table_name', None)
            if table_path is None:
                attrs['_table_name'] = '{}-pk:{}'.format(destination_table._table_name, ','.join(pk))

            new_class = super(SecondaryIndexTableMetaclass, cls).__new__(cls, name, bases, attrs)
            registry.register_secondary_index_table(new_class, destination_table, pk)
            return new_class

        return super(SecondaryIndexTableMetaclass, cls).__new__(cls, name, bases, attrs)


@six.add_metaclass(SecondaryIndexTableMetaclass)
class SecondaryIndexTable(Table):
    __abstract__ = True

    _destination_table = None
