import collections
import datetime
import logging
import six
import string
import time

from irt.multik.common import force_text


log = logging.getLogger('yt_orm.manager')


class NOT_SET:
    pass


class Field(object):
    """Base field class for yt_orm fields"""
    _creation_counter = 0

    def __init__(self, namespace=False, key=False, default=NOT_SET, allow_null=True, name=None):
        # helps keep track of order of creation of fields
        self._creation_counter = Field._creation_counter
        Field._creation_counter += 1

        self.name = name
        self.namespace = namespace
        self.key = key
        self.default = default
        self.allow_null = allow_null

    def copy(self):
        return type(self)(namespace=self.namespace, key=self.key, default=self.default, allow_null=self.allow_null, name=self.name)

    def contribute_to_class(self, cls, name):
        """Create backreferense from Field to table"""
        self.table = cls
        self.name = self.name or name
        if not getattr(cls, self.name, None):
            setattr(cls, self.name, self)

    @property
    def get_default(self):
        if self.default is NOT_SET:
            return lambda: None

        if callable(self.default):
            return self.default
        return lambda: self.default

    def to_yt(self, value):
        """Convert from python to yt-compatible type

        This method is applied when saving/deleting objects and when using object
        inside SQL queries. This ensures that ints are ints, string are strings and
        is usefull to convert custom types like datetime to int conversion
        """
        # None is an always valid value for any field in YT
        # Otherwise apply field-specific conversion
        return None if value is None else self._to_yt(value)

    def _to_yt(self, value):
        return value

    def to_query(self, value):
        """Return value representation suitable for YT dyn-table SQL"""
        return 'null' if value is None else self._to_query(value)

    def _to_query(self, value):
        return force_text(self.to_yt(value))

    def to_python(self, value):
        return value

    def get_type(self):
        # Value returned by this function is passed to create_table for
        # automatic schema generation
        # TODO: validate that python type is set in a child class. Or that get_type is
        # overridden
        return self.python_type


class IntegerField(Field):
    python_type = int

    def _to_yt(self, value):
        return int(value)


class UIntegerField(IntegerField):
    def get_type(self):
        return 'uint64'


_punctuation = set(string.punctuation)


class CharField(Field):
    python_type = str

    def _to_yt(self, value):
        return force_text(value)

    def _to_query(self, value):
        new_value = super(CharField, self)._to_query(value)
        return u"'{}'".format(self._escape(new_value))

    # Simplistic escaping to wrap string in single quotes and
    # not break yt.select_rows
    def _escape(self, value):
        "Escape all non-alphanumeric characters in pattern."
        s = list(u'{}'.format(value))
        for i, c in enumerate(value):
            if c in _punctuation:
                s[i] = "\\" + c
        return ''.join(s)


class ListField(Field):
    python_type = list

    def __init__(self, *args, **kwargs):
        child_field = kwargs.pop('child_field', None)
        super(ListField, self).__init__(*args, **kwargs)
        self.child_field = child_field

    def copy(self):
        copy = super(ListField, self).copy()
        copy.child_field = self.child_field
        return copy

    # TODO: implement to_query, with respct to child_field
    # TODO: handle children of multiple types, possibly with a subclass


class BooleanField(Field):
    python_type = bool


class DateTimeField(Field):
    python_type = datetime.datetime

    def _to_yt(self, value):
        if isinstance(value, six.integer_types):
            return value
        elif isinstance(value, datetime.datetime):
            return int(time.mktime(value.timetuple()))
        else:
            raise TypeError("Don't know how to convert {} to timestamp".format(value))

    def to_python(self, value):
        """Convert yt representation to python object

        YT dyn tables do not have builtin date type, so we should get uint64 from YT
        """
        if value is None:
            return value
        elif isinstance(value, six.integer_types):
            try:
                return datetime.datetime.fromtimestamp(value)
            except Exception:
                error = "Could not convert {} to valid datetime object".format(value)
                log.exception(error)
                raise TypeError(error)
        else:
            raise TypeError("Don't know how to convert {} to datetime".format(value))


class FieldStore(object):
    """Helper class with field accessor methods

    Used as a single interface to access table fields.
    """
    def __init__(self, fields):
        self.fields = fields

    @property
    def namespace(self):
        if not hasattr(self, '_ns_fields'):
            self._ns_fields = collections.OrderedDict(
                (name, field) for name, field in self.fields.items()
                if field.namespace)
        return self._ns_fields

    @property
    def key(self):
        if not hasattr(self, '_key_fields'):
            self._key_fields = collections.OrderedDict(
                (name, field) for name, field in self.fields.items()
                if field.key)
        return self._key_fields

    @property
    def data(self):
        if not hasattr(self, '_data_fields'):
            self._data_fields = collections.OrderedDict(
                (name, field) for name, field in self.fields.items()
                if not field.key and not field.namespace)
        return self._data_fields

    @property
    def all(self):
        return self.fields
