from irt.utils import list_like

_lookups = {}

LOOKUP_SEP = '__'


class BadLookup(Exception):
    pass


def register(lookup):
    if lookup.operator in _lookups:
        raise ValueError("Lookup for operator {} is already registered".format(lookup.operator))
    _lookups[lookup.operator] = lookup
    return lookup


def get_lookup(operator):
    if operator not in _lookups:
        raise BadLookup("No lookup registered for operator {}".format(operator))
    return _lookups[operator]


class Lookup(object):
    comparator = operator = None  # type: str
    allow_multiple = True
    check_values = True

    def __init__(self, table, clause_parts, values):
        self.table = table
        self.clause_parts = clause_parts
        self.values = values if list_like(values) else [values]
        self.template = u"{}" if len(self.clause_parts) == 1 else u"({})"

        if not self.clause_parts:
            raise BadLookup("No fields requested in lookup")
        if self.check_values and len(self.clause_parts) != len(self.values):
            raise BadLookup(u"Got {} as fields for lookup and {} as values".format(self.clause_parts, self.values))
        if not self.allow_multiple and len(self.clause_parts) > 1:
            raise BadLookup(u"Lookup '{}' does not allow multiple parameters, but got {}".format(
                self.operator, self.clause_parts))

    def get_fields(self):
        table_fields = self.table._fields.all
        for field_name in self.clause_parts:
            if field_name not in table_fields:
                raise BadLookup(u"No field '{}' in table {}. Choices are: {}".format(
                    field_name, self.table, list(table_fields.keys())))
        return [table_fields[field_name] for field_name in self.clause_parts]

    def _resolve_field_names(self, fields):
        return self.template.format(", ".join(field.name for field in fields))

    def _resolve_values(self, fields):
        return self.template.format(", ".join(field.to_query(value) for field, value in zip(fields, self.values)))

    def resolve(self):
        fields = self.get_fields()
        return u"{} {} {}".format(
            self._resolve_field_names(fields),
            self.comparator,
            self._resolve_values(fields),
        )


@register
class Exact(Lookup):
    operator = 'exact'
    comparator = '='


@register
class NotEquals(Lookup):
    operator = 'ne'
    comparator = '!='


@register
class Greater(Lookup):
    operator = 'gt'
    comparator = '>'


@register
class GreaterOrEquals(Lookup):
    operator = 'gte'
    comparator = '>='


@register
class Less(Lookup):
    operator = 'lt'
    comparator = '<'


@register
class LessOrEquals(Lookup):
    operator = 'lte'
    comparator = '<='


@register
class In(Lookup):
    operator = 'in'
    comparator = 'IN'
    check_values = False

    def _resolve_values(self, fields):
        elements = []
        if len(fields) == 1:
            elements.append(
                u"({})".format(", ".join(fields[0].to_query(val) for val in self.values))
            )
        else:
            for values in self.values:
                if not list_like(values):
                    raise BadLookup(u"Can't use {} as values for {}".format(values, self.clause_parts))
                # make sure values has len and check we have enough values and fields
                values = list(values)
                if len(values) != len(fields):
                    raise BadLookup(u"Can't use {} as values for {}".format(values, self.clause_parts))
                # pass each value through respective field.to_query method
                elements.append(u"({})".format(", ".join(fields[idx].to_query(val) for idx, val in enumerate(values))))
        # join everything back
        return self.template.format(", ".join(elements))


@register
class ListContains(Lookup):
    operator = 'list_contains'
    allow_multiple = False

    def resolve(self):
        field = self.get_fields()[0]
        value = field.child_field.to_query(self.values[0]) if field.child_field else self.values[0]
        return u"list_contains({}, {})".format(field.name, value)
