import ads.multik.pylib.key_value_store as kv

from . import query
from .registry import registry

import logging


log = logging.getLogger('yt_orm.manager')


class ObjectDoesNotExist(Exception):
    pass


class Manager(object):

    @property
    def store(self):
        if not hasattr(self, '_store'):
            client = registry.get_yt_client()

            store = kv.DynTableKeyValueStore(
                tuple(self.table._fields.namespace.keys()),
                tuple(self.table._fields.key.keys()),
                tuple(self.table._fields.data.keys()),
            )
            store.init_store((self.table._table_path, client))
            self._store = store
        return self._store

    def contribute_to_class(self, table, name):
        self.table = table
        setattr(table, name, self)

    def create_table(self, exists_ok=False):
        log.info("Creating table for {}".format(self.table))
        self.table.create_table(exists_ok=exists_ok)
        log.info("Mounting table for {}".format(self.table))
        self.store.connect_store()

    def read(self, **kwargs):
        ns = tuple(kwargs.get(k) for k in self.table._fields.namespace)
        key = tuple(kwargs.get(k) for k in self.table._fields.key)
        row = self.store.read(ns, key)
        return self.table(**row)

    def get(self, *args, **kwargs):
        # NOTE: We might want behaviour similar to django. i.e. raise exception
        # on multiple objects returned. For now returning single object if multiple
        # found
        obj = next(self.filter(*args, **kwargs).limit(1).eval(), None)
        if obj is None:
            raise ObjectDoesNotExist("No {} for {}, {}".format(self.table, args, kwargs))
        return obj

    def save(self, state):
        """Save object to store"""
        return self.store.upsert(
            tuple(state[k] for k in self.table._fields.namespace.keys()),
            tuple(state[k] for k in self.table._fields.key.keys()),
            {k: state[k] for k in self.table._fields.data.keys()},
        )

    def bulk_save(self, objects):
        bulk_objects = []
        for obj in objects:
            if not isinstance(obj, self.table):
                raise TypeError("Can't save object {}. Only accepting objects of {} type".format(obj, self.table))
            state = obj._get_state()

            bulk_objects.append((
                tuple(state[k] for k in self.table._fields.namespace.keys()),
                tuple(state[k] for k in self.table._fields.key.keys()),
                {k: state[k] for k in self.table._fields.data.keys()},
            ))
        return self.store.bulk_upsert(bulk_objects)

    def delete(self, namespace, key):
        return self.store.delete(namespace, key)

    def all(self):
        return query.Query(self.table, self.store)

    def filter(self, *args, **kwargs):
        return query.Query(self.table, self.store).filter(*args, **kwargs)

    def exclude(self, *args, **kwargs):
        return query.Query(self.table, self.store).exclude(*args, **kwargs)

    def transaction(self):
        return self.store.transaction()
