import datetime
from six import string_types

import yt.wrapper as yt


class Schema(object):
    TYPES = {
        int: 'int64',
        float: 'double',
        bool: 'boolean',
        str: 'string',
        datetime.datetime: 'uint64',
        datetime.date: 'uint64',
        datetime.time: 'uint64',
    }

    @classmethod
    def _to_schema_type(cls, t):
        return t if isinstance(t, string_types) else cls.TYPES.get(t, 'any')

    def __init__(self, table_fields):
        self._schema = []
        keys = True

        for name in table_fields:
            field = table_fields[name]
            t = field.get_type()
            field_schema = {'name': name, 'type': self._to_schema_type(t)}
            if field.key or field.namespace:
                if not keys:
                    raise ValueError('Key column {} going after not key column'.format(name))
                field_schema['sort_order'] = 'ascending'
                if field.key:
                    field_schema['required'] = True
            else:
                keys = False
            self._schema.append(field_schema)

    @property
    def simple(self):
        return [
            {'name': el['name'], 'type': el['type']} for el in self._schema
        ]

    @property
    def schema(self):
        return self._schema

    @property
    def yt_attributes(self):
        yt_schema = yt.yson.YsonList()
        yt_schema.extend(self._schema)
        yt_schema.attributes['unique_keys'] = True
        yt_schema.attributes['strict'] = True

        return {
            'schema': yt_schema,
            'dynamic': True
        }

    @property
    def sort_keys(self):
        return [el['name'] for el in self._schema if 'sort_order' in el]
