#include "broadcast_action.h"

#include <library/cpp/string_utils/quote/quote.h>
#include <util/string/vector.h>

namespace NDaemonController {
    NJson::TJsonValue TBroadcastAction::DoSerializeToJson() const {
        NJson::TJsonValue result;
        result.InsertValue("command", Command);
        result.InsertValue("ctype", CType);
        result.InsertValue("service", Service);
        result.InsertValue("service_type", ServiceType);
        result.InsertValue("result", Result);
        result.InsertValue("ignore_unknown_path", IgnoreUnknowPath);
        result.InsertValue("search_map", SearchMap);
        result.InsertValue("timeout", Timeout.ToString());
        result.InsertValue("multiple_map", IsMultipleMap);
        result.InsertValue("with_sd", WithSd);
        for (const auto& filter : Filter)
            result["filter"].AppendValue(filter);
        return result;
    }

    void TBroadcastAction::DoDeserializeFromJson(const NJson::TJsonValue& json) {
        Command = json["command"].GetStringRobust();
        CType = json["ctype"].GetStringRobust();
        Service = json["service"].GetStringRobust();
        ServiceType = json["service_type"].GetStringRobust();
        IgnoreUnknowPath = json["ignore_unknown_path"].GetBooleanRobust();
        SearchMap = json["search_map"].GetStringRobust();
        Timeout = TDuration::Parse(json["timeout"].GetStringRobust());
        IsMultipleMap = json["multiple_map"].GetBooleanRobust();
        WithSd = json["with_sd"].GetBooleanRobust();
        Filter.clear();
        if (json.Has("filter"))
            for (const auto& f : json["filter"].GetArray())
                Filter.insert(f.GetStringRobust());
        Result = json["result"];
    }

    TString TBroadcastAction::DoBuildCommand() const {
        const TString filter = JoinStrings(Filter.cbegin(), Filter.cend(), ",");
        return "command=" + Command
            + "&ctype=" + CType
            + "&service=" + Service
            + "&service_type=" + ServiceType
            + "&filter=" + filter
            + "&ignore_unknown_path=" + ToString(IgnoreUnknowPath)
            + "&multiple_sm=" + ToString(IsMultipleMap)
            + "&with_sd=" + ToString(WithSd);
    }

    void TBroadcastAction::DoInterpretResult(const TString& result) {
        TStringInput si(result);
        if (!NJson::ReadJsonTree(&si, &Result))
            Fail(result);
        else
            Success("");
    }

    TBroadcastAction::TBroadcastAction(const TString& command, const TString& ctype, const TString& service, const TString& serviceType,
        bool ignoreUnknowPath /*= false*/, const TString& searchMap /*= TString()*/, TDuration timeout
    )
        : Command(command)
        , CType(ctype)
        , Service(service)
        , ServiceType(serviceType)
        , IgnoreUnknowPath(ignoreUnknowPath)
        , SearchMap(searchMap)
        , Timeout(timeout)
    {}


    TBroadcastAction::TFactory::TRegistrator<TBroadcastAction> TBroadcastAction::Registrator(BROADCAST_ACTION_NAME);
}
