#include "grouping_common_param.h"

#include "plainer_richers.h"

#include <library/cpp/testing/unittest/registar.h>

using namespace NPrivate;

Y_UNIT_TEST_SUITE(TGrouingCommonParamTest) {
    Y_UNIT_TEST(MinGroupingTest) {
        TGroupingCommonParam::TPtr minGroupingPtr = TGroupingCommonParam::Create("docs_count_min=docs_count@min");
        NJson::TJsonValue param;
        NJson::TJsonValue item;
        UNIT_ASSERT(!minGroupingPtr->UpdateParamValueFromElement(param, item));
        UNIT_ASSERT(!param.IsDefined());
        item["docs_count"] = 0.1;
        UNIT_ASSERT(!minGroupingPtr->UpdateParamValueFromElement(param, item));
        UNIT_ASSERT(param.IsDouble());
        UNIT_ASSERT_DOUBLES_EQUAL(param.GetDouble(), 0.1, 1e-5);
        item["docs_count"] = 0.2;
        UNIT_ASSERT(!minGroupingPtr->UpdateParamValueFromElement(param, item));
        UNIT_ASSERT(param.IsDouble());
        UNIT_ASSERT_DOUBLES_EQUAL(param.GetDouble(), 0.1, 1e-5);
        item["docs_count"] = 0.0;
        UNIT_ASSERT(!minGroupingPtr->UpdateParamValueFromElement(param, item));
        UNIT_ASSERT(param.IsDouble());
        UNIT_ASSERT_DOUBLES_EQUAL(param.GetDouble(), 0.0, 1e-5);
        minGroupingPtr.Drop();
    }

   Y_UNIT_TEST(MinGroupingTestInt) {
        TGroupingCommonParam::TPtr minGroupingPtr = TGroupingCommonParam::Create("docs_count_min=docs_count@min");
        NJson::TJsonValue param;
        NJson::TJsonValue item;
        UNIT_ASSERT(!minGroupingPtr->UpdateParamValueFromElement(param, item));
        UNIT_ASSERT(!param.IsDefined());
        item["docs_count"] = 1;
        UNIT_ASSERT(!minGroupingPtr->UpdateParamValueFromElement(param, item));
        UNIT_ASSERT(param.IsUInteger());
        UNIT_ASSERT_VALUES_EQUAL(param.GetUInteger(), 1);
        item["docs_count"] = 2;
        UNIT_ASSERT(!minGroupingPtr->UpdateParamValueFromElement(param, item));
        UNIT_ASSERT(param.IsUInteger());
        UNIT_ASSERT_VALUES_EQUAL(param.GetUInteger(), 1);
        item["docs_count"] = 0;
        UNIT_ASSERT(!minGroupingPtr->UpdateParamValueFromElement(param, item));
        UNIT_ASSERT(param.IsUInteger());
        UNIT_ASSERT_VALUES_EQUAL(param.GetUInteger(), 0);
        minGroupingPtr.Drop();
    }

    Y_UNIT_TEST(MinFacetFlatGroupingTest) {
        TGroupingCommonParam::TPtr minGroupingPtr = TGroupingCommonParam::Create("dcfail@facet_flat_min");
        {
            NJson::TJsonValue param;
            NJson::TJsonValue item;
            UNIT_ASSERT(!minGroupingPtr->UpdateParamValueFromElement(param, item));
            UNIT_ASSERT(!param.IsDefined());
        }
        {
            NJson::TJsonValue param;
            NJson::TJsonValue item;
            item["dcfail"]["ALL"] = 1;
            UNIT_ASSERT(!minGroupingPtr->UpdateParamValueFromElement(param, item));
            UNIT_ASSERT(param.IsUInteger());
            UNIT_ASSERT_VALUES_EQUAL(param.GetUInteger(), 1);
        }
        {
            NJson::TJsonValue param;
            NJson::TJsonValue item;
            item["dcfail"]["VLA"] = 3;
            item["dcfail"]["SAS"] = 2;
            item["dcfail"]["MAN"] = 4;
            UNIT_ASSERT(!minGroupingPtr->UpdateParamValueFromElement(param, item));
            UNIT_ASSERT(param.IsUInteger());
            UNIT_ASSERT_VALUES_EQUAL(param.GetUInteger(), 2);
        }
        minGroupingPtr.Drop();
    }

    Y_UNIT_TEST(SimpleFacetTest) {
        TGroupingCommonParam::TPtr grouping = TGroupingCommonParam::Create("dcfail@facet");

        NJson::TJsonValue param;
        NJson::TJsonValue item;
        UNIT_ASSERT(!grouping->UpdateParamValueFromElement(param, item));
        UNIT_ASSERT(!param.IsDefined());

        item["dcfail"] = "MAN";
        UNIT_ASSERT(!grouping->UpdateParamValueFromElement(param, item));
        UNIT_ASSERT(param.IsMap());
        UNIT_ASSERT_VALUES_EQUAL(param.GetMapSafe().size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(param.GetMapSafe().at("MAN").GetIntegerSafe(), 1);

        UNIT_ASSERT(!grouping->UpdateParamValueFromElement(param, item));
        UNIT_ASSERT_VALUES_EQUAL(param.GetMapSafe().size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(param.GetMapSafe().at("MAN").GetIntegerSafe(), 2);

        item["dcfail"] = "VLA";
        UNIT_ASSERT(!grouping->UpdateParamValueFromElement(param, item));
        UNIT_ASSERT_VALUES_EQUAL(param.GetMapSafe().size(), 2);
        UNIT_ASSERT_VALUES_EQUAL(param.GetMapSafe().at("MAN").GetIntegerSafe(), 2);
        UNIT_ASSERT_VALUES_EQUAL(param.GetMapSafe().at("VLA").GetIntegerSafe(), 1);
    }

    Y_UNIT_TEST(MapFacetTest) {
        TGroupingCommonParam::TPtr grouping = TGroupingCommonParam::Create("dcfail@facet");

        NJson::TJsonValue param;
        NJson::TJsonValue item;
        UNIT_ASSERT(!grouping->UpdateParamValueFromElement(param, item));
        UNIT_ASSERT(!param.IsDefined());

        item["dcfail"]["MAN"] = 0;
        UNIT_ASSERT(!grouping->UpdateParamValueFromElement(param, item));
        UNIT_ASSERT(param.IsMap());
        UNIT_ASSERT_VALUES_EQUAL(param.GetMapSafe().size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(param.GetMapSafe().at("MAN").GetIntegerSafe(), 0);

        item["dcfail"]["VLA"] = 1;
        UNIT_ASSERT(!grouping->UpdateParamValueFromElement(param, item));
        UNIT_ASSERT_VALUES_EQUAL(param.GetMapSafe().size(), 2);
        UNIT_ASSERT_VALUES_EQUAL(param.GetMapSafe().at("MAN").GetIntegerSafe(), 0);
        UNIT_ASSERT_VALUES_EQUAL(param.GetMapSafe().at("VLA").GetIntegerSafe(), 1);

        item["dcfail"]["MAN"] = 1;
        item["dcfail"]["VLA"] = 0;
        UNIT_ASSERT(!grouping->UpdateParamValueFromElement(param, item));
        UNIT_ASSERT_VALUES_EQUAL(param.GetMapSafe().size(), 2);
        UNIT_ASSERT_VALUES_EQUAL(param.GetMapSafe().at("MAN").GetIntegerSafe(), 1);
        UNIT_ASSERT_VALUES_EQUAL(param.GetMapSafe().at("VLA").GetIntegerSafe(), 1);
    }

    Y_UNIT_TEST(JsonFacetTest) {
        TGroupingCommonParam::TPtr grouping = TGroupingCommonParam::Create("dcfail@facet_json");

        NJson::TJsonValue param;
        NJson::TJsonValue item;
        NJson::TJsonValue tmp;
        UNIT_ASSERT(!grouping->UpdateParamValueFromElement(param, item));
        UNIT_ASSERT(!param.IsDefined());

        tmp["MAN"] = 0;
        item["dcfail"] = JsonToString(tmp);
        UNIT_ASSERT(!grouping->UpdateParamValueFromElement(param, item));
        UNIT_ASSERT(param.IsMap());
        UNIT_ASSERT_VALUES_EQUAL(param.GetMapSafe().size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(param.GetMapSafe().at("MAN").GetIntegerSafe(), 0);

        tmp["VLA"] = 1;
        item["dcfail"] = JsonToString(tmp);
        UNIT_ASSERT(!grouping->UpdateParamValueFromElement(param, item));
        UNIT_ASSERT_VALUES_EQUAL(param.GetMapSafe().size(), 2);
        UNIT_ASSERT_VALUES_EQUAL(param.GetMapSafe().at("MAN").GetIntegerSafe(), 0);
        UNIT_ASSERT_VALUES_EQUAL(param.GetMapSafe().at("VLA").GetIntegerSafe(), 1);

        tmp["MAN"] = 1;
        tmp["VLA"] = 0;
        item["dcfail"] = JsonToString(tmp);
        UNIT_ASSERT(!grouping->UpdateParamValueFromElement(param, item));
        UNIT_ASSERT_VALUES_EQUAL(param.GetMapSafe().size(), 2);
        UNIT_ASSERT_VALUES_EQUAL(param.GetMapSafe().at("MAN").GetIntegerSafe(), 1);
        UNIT_ASSERT_VALUES_EQUAL(param.GetMapSafe().at("VLA").GetIntegerSafe(), 1);
    }
}
