#include "base_http_client.h"
#include "messages.h"

#include <saas/library/daemon_base/unistat_signals/signals.h>

#include <library/cpp/json/json_writer.h>

#include <util/string/strip.h>
#include <util/string/type.h>

bool TCommonHttpClientFeatures::ProcessSpecialRequest() {
    const TBaseServerRequestData& rd = GetBaseRequestData();
    const TCgiParameters& cgi = GetCgi();
    const auto iter = cgi.Find(TStringBuf("info_server"));
    if (iter != cgi.end() && IsTrue(iter->second)) {
        ProcessInfoServer();
        return true;
    } else if ("/ping"sv == rd.ScriptName()) {
        ProcessPing();
        return true;
    } else if ("/status"sv == rd.ScriptName()) {
        ProcessServerStatus();
        return true;
    } else if ("/tass"sv == rd.ScriptName()) {
        ProcessTass(Output());
        return true;
    } else if ("/metric"sv == rd.ScriptName()) {
        GetMetrics(Output());
        return true;
    } else if ("/supermind"sv == rd.ScriptName()) {
        ProcessSuperMind(Output());
        return true;
    }
    return false;
}

void TCommonHttpClientFeatures::ProcessInfoServer() {
    TServerInfo info = GetServerInfo(/* isHumanReadable = */ false);
    const TStringBuf origin = GetBaseRequestData().HeaderInOrEmpty("Origin");
    const TString originHeader(!!origin ? TString::Join("Access-Control-Allow-Origin:", origin, "\r\nAccess-Control-Allow-Credentials:true\r\n") : TString());
    Output() << "HTTP/1.1 200 Ok\r\n"sv
        << originHeader
        << "Content-Type: application/json\r\n\r\n"sv;
    NJson::TJsonWriterConfig jsonFormat;
    jsonFormat.FormatOutput = true;
    jsonFormat.SortKeys = true;
    NJson::WriteJson(&Output(), &info, jsonFormat);
}

void TCommonHttpClientFeatures::ProcessPing() {
    const TStringBuf origin = GetBaseRequestData().HeaderInOrEmpty("Origin");
    const TString originHeader(!!origin ? TString::Join("Access-Control-Allow-Origin:", origin, "\r\nAccess-Control-Allow-Credentials:true\r\n") : TString());
    Output() << "HTTP/1.1 200 Ok\r\n"sv << originHeader << "\r\n1"sv << Endl;
}

TServerInfo TCommonHttpClientFeatures::GetServerInfo(bool isHumanReadable) const {
    TCollectServerInfo info(isHumanReadable);
    return CollectServerInfo(info);
}

void TCommonHttpClientFeatures::ProcessServerStatus() {
    const TStringBuf origin = GetBaseRequestData().HeaderInOrEmpty("Origin");
    const TString originHeader(!!origin ? TString::Join("Access-Control-Allow-Origin:", origin, "\r\nAccess-Control-Allow-Credentials:true\r\n") : TString());
    Output() << "HTTP/1.1 200 Ok\r\n"sv
        << originHeader
        << "Content-Type: text/plain\r\n\r\n"sv
        << "Started";
}

void TCommonHttpClientFeatures::GetMetrics(IOutputStream& out) const {
    out << "HTTP/1.1 404 Not Found\r\n\r\n"sv << Endl;
}

void TCommonHttpClientFeatures::ProcessSuperMind(IOutputStream& out) const {
    out << "HTTP/1.1 404 Not Found\r\n\r\n"sv << Endl;
}

void TCommonHttpClientFeatures::ProcessTass(IOutputStream& out) const {
    const TStringBuf origin = GetBaseRequestData().HeaderInOrEmpty("Origin");
    const TString originHeader(!!origin ? TString::Join("Access-Control-Allow-Origin:", origin, "\r\nAccess-Control-Allow-Credentials:true\r\n") : TString());
    out << "HTTP/1.1 200 Ok\r\n"sv
        << originHeader
        << "Content-Type: application/json\r\n\r\n"sv;
    ReportUnistatSignals(out);
}
