#include "cypress.h"

#include <library/cpp/testing/unittest/registar.h>

TFsPath path = "./rtyt_test";

void Cleanup() {
    TVector<TFsPath> children;
    if (path.Exists()) {
        path.List(children);
        for (const auto& child : children) {
            child.ForceDelete();
        }
    }
}

Y_UNIT_TEST_SUITE(RTYT_CYPRESS) {
    Y_UNIT_TEST(SIMPLE_CREATING_TEST) {
        Cleanup();
        path.MkDir();
        NFs::SetCurrentWorkingDirectory(path);
        NRTYT::TCypressClient client;
        UNIT_ASSERT(!client.Exists("//test1"));
        client.Create("//test1", NYT::NT_MAP);
        UNIT_ASSERT(client.Exists("//test1"));
        client.Create("//test1/test2", NYT::NT_MAP);
        client.Create("//test2", NYT::NT_FILE);
        UNIT_ASSERT(client.Exists("//test1"));
        UNIT_ASSERT(client.Exists("//test1/test2"));
        UNIT_ASSERT(!client.Exists("//test1/test3"));
        UNIT_ASSERT(!client.Exists("//test3"));
        UNIT_ASSERT(!client.Exists("//test2/test3"));

        NFs::SetCurrentWorkingDirectory("..");
    }
    Y_UNIT_TEST(SIMPLE_COPY_TEST) {
        Cleanup();
        path.MkDir();
        NFs::SetCurrentWorkingDirectory(path.GetPath());
        NRTYT::TCypressClient client;
        client.Create("//test1", NYT::NT_MAP);
        client.Create("//test1/test2", NYT::NT_MAP);
        client.Create("//test3", NYT::NT_MAP);
        client.Copy("//test1", "//test3/test4");
        UNIT_ASSERT(client.Exists("//test1"));
        UNIT_ASSERT(client.Exists("//test3/test4"));
        UNIT_ASSERT(client.Exists("//test3/test4/test2"));
        UNIT_ASSERT(client.Exists("//test1/test2"));
        UNIT_ASSERT(!client.Exists("//test1/test3"));
        NFs::SetCurrentWorkingDirectory("..");
    }

    Y_UNIT_TEST(SIMPLE_MOVE_TEST) {
        Cleanup();
        path.MkDir();
        NFs::SetCurrentWorkingDirectory(path.GetPath());
        NRTYT::TCypressClient client;
        client.Create("//test1", NYT::NT_MAP);
        client.Create("//test1/test2", NYT::NT_MAP);
        client.Move("//test1", "//test3", NYT::TMoveOptions().Recursive(true));
        UNIT_ASSERT(!client.Exists("//test1"));
        UNIT_ASSERT(client.Exists("//test3"));
        UNIT_ASSERT(client.Exists("//test3/test2"));
        UNIT_ASSERT(!client.Exists("//test1/test2"));
        UNIT_ASSERT(!client.Exists("//test1/test3"));
        NFs::SetCurrentWorkingDirectory("..");
    }
    Y_UNIT_TEST(SIMPLE_SAVE_LOAD_TEST) {
        Cleanup();
        path.MkDir();
        NFs::SetCurrentWorkingDirectory(path.GetPath());
        TFsPath inner = "./cypress1";
        {
            NRTYT::TCypressClient client(inner);
            client.Create("//test1", NYT::NT_MAP);
            client.Create("//test1/test2", NYT::NT_MAP);
            client.Create("//test3", NYT::NT_MAP);
            client.Copy("//test1", "//test3/test4");
            client.Sync();
        }
        {
            NRTYT::TCypressClient client;
            client.Mount(inner, "//cypress1");

            UNIT_ASSERT(client.Exists("//cypress1/test1"));
            UNIT_ASSERT(client.Exists("//cypress1/test3/test4"));
            UNIT_ASSERT(client.Exists("//cypress1/test3/test4/test2"));
            UNIT_ASSERT(client.Exists("//cypress1/test1/test2"));
            UNIT_ASSERT(!client.Exists("//cypress1/test1/test3"));
        }
        NFs::SetCurrentWorkingDirectory("..");
    }

    Y_UNIT_TEST(SIMPLE_MOUNT_TEST) {
        Cleanup();
        path.MkDir();
        NFs::SetCurrentWorkingDirectory(path.GetPath());
        NRTYT::TCypressClient client;
        TFsPath inner = "./cypress1";
        client.Mount(inner, "//cypress2");
        client.Create("//cypress2/test1", NYT::NT_MAP);
        client.Create("//cypress2/test1/test2", NYT::NT_MAP);
        client.Create("//cypress2/test3", NYT::NT_MAP);
        client.Copy("//cypress2/test1", "//cypress2/test3/test4");

        UNIT_ASSERT(client.Exists("//cypress2/test1"));
        UNIT_ASSERT(client.Exists("//cypress2/test3/test4"));
        UNIT_ASSERT(client.Exists("//cypress2/test3/test4/test2"));
        UNIT_ASSERT(client.Exists("//cypress2/test1/test2"));
        UNIT_ASSERT(!client.Exists("//cypress2/test1/test3"));
        NFs::SetCurrentWorkingDirectory("..");
    }
}
