#pragma once

#include <saas/library/hash_to_docid/doc_hash.h>

#include <library/cpp/logger/global/global.h>

#include <util/generic/string.h>

struct TDocIdCandidate {
    using THash = NSaas::TDocHash;

    const TString& GetUrl() const {
        VERIFY_WITH_LOG(!!Url, "url has not set");
        return Url;
    }

    ui64 GetRelevance() const {
        VERIFY_WITH_LOG(Relevance, "relevance has not set");
        return Relevance;
    }

    void SetRelevance(ui64 relevance) {
        Relevance = relevance;
    }

    bool UrlInitialized() const {
        return !!Url;
    }

    ui32 GetDocId() const {
        VERIFY_WITH_LOG(DocId != Max<ui32>(), "docid has not set");
        return DocId;
    }

    bool IsDocIdSet() const {
        return DocId != Max<ui32>();
    }

    void SetDocId(ui32 docid) {
        DocId = docid;
    }

    void UnsetDocId() {
        DocId = Max<ui32>();
    }

    ui64 GetKeyPrefix() const {
        VERIFY_WITH_LOG(KeyPrefix != Max<ui64>(), "kps has not set");
        return KeyPrefix;
    }

    TString GetFullDocId() const {
        VERIFY_WITH_LOG(!!FullDocId, "fullDocId has not set");
        return FullDocId;
    }

    THash GetHash() const;

    const TString GetMSHash() const {
        return GetHash().Quote();
    }

    TDocIdCandidate(const TString& url, ui64 keyprefix)
        : Url(url)
        , DocId(Max<ui32>())
        , KeyPrefix(keyprefix)
        , Verified(false)
    {}

    TDocIdCandidate(const TString& url, ui32 docId, ui64 keyprefix)
        : Url(url)
        , DocId(docId)
        , KeyPrefix(keyprefix)
        , Verified(false)
    {}

    TDocIdCandidate(const TString& url, ui32 docId, ui64 keyprefix, const TString& fullDocId)
        : Url(url)
        , DocId(docId)
        , KeyPrefix(keyprefix)
        , FullDocId(fullDocId)
        , Verified(false)
    {}

    bool IsVerified() const {
        return Verified;
    }

    void SetVerified(bool verified) {
        Verified = verified;
    }

    bool operator<(const TDocIdCandidate& other) const {
        if (!IsDocIdSet()) {
            return GetHash() < other.GetHash();
        }
        if (!other.IsDocIdSet()) {
            return false;
        }
        return GetDocId() < other.GetDocId();
    }

protected:
    TString Url;
    ui32 DocId;
    ui64 KeyPrefix;
    TString FullDocId;
    ui64 Relevance;
    bool Verified;
};
