#include "search_size_chooser.h"
#include "interval_value.h"
#include "allowed_fp_mappings.h"

#include <library/cpp/testing/unittest/registar.h>

using namespace NRTYServer;

Y_UNIT_TEST_SUITE(IntervalValue) {
    Y_UNIT_TEST(NotInited) {
        TIntervalValue interval;
        interval.ParseFromConfigStr("some error string");
        UNIT_ASSERT(interval.GetParseErrorString().Defined());
    }

    Y_UNIT_TEST(ErrorConfigInfo) {
        TIntervalValue interval;
        interval.ParseFromConfigStr("");
        UNIT_ASSERT(!interval.GetParseErrorString());
        UNIT_ASSERT_VALUES_EQUAL(interval.Clamp(1), 1);
        UNIT_ASSERT_VALUES_EQUAL(interval.Clamp(2), 2);
        UNIT_ASSERT_VALUES_EQUAL(interval.Clamp(-1), -1);
        UNIT_ASSERT_VALUES_EQUAL(interval.Clamp(-2), -2);
        UNIT_ASSERT_VALUES_EQUAL(interval.Clamp(0), 0);
        UNIT_ASSERT_VALUES_EQUAL(interval.Clamp(0.9), 0.9f);
        UNIT_ASSERT_VALUES_EQUAL(interval.Clamp(-0.9), -0.9f);
    }

    Y_UNIT_TEST(Parsing1) {
        TIntervalValue interval;
        interval.ParseFromConfigStr("[-1;1]");
        UNIT_ASSERT(!interval.GetParseErrorString());
        TStringStream str;
        interval.DoToString(str);
        UNIT_ASSERT_NO_DIFF(str.Str(), "[-1;1]");
        UNIT_ASSERT_VALUES_EQUAL(interval.Clamp(1), 1);
        UNIT_ASSERT_VALUES_EQUAL(interval.Clamp(2), 1);
        UNIT_ASSERT_VALUES_EQUAL(interval.Clamp(-1), -1);
        UNIT_ASSERT_VALUES_EQUAL(interval.Clamp(-2), -1);
        UNIT_ASSERT_VALUES_EQUAL(interval.Clamp(0), 0);
        UNIT_ASSERT_VALUES_EQUAL(interval.Clamp(0.9), 0.9f);
        UNIT_ASSERT_VALUES_EQUAL(interval.Clamp(-0.9), -0.9f);
    }
}

Y_UNIT_TEST_SUITE(SearchSizeChooser) {
    Y_UNIT_TEST(NoConfigInfo) {
        THnswSearchSizeChooser sizeChooser;
        UNIT_ASSERT(!sizeChooser.GetParseErrorString());
        UNIT_ASSERT_VALUES_EQUAL(sizeChooser.GetRecommendedSearchSize(100) , 50);

        TStringStream str;
        sizeChooser.DoToString(str);
        UNIT_ASSERT_NO_DIFF(str.Str(), "Default=50");
    }

    Y_UNIT_TEST(ErrorConfigInfo) {
        THnswSearchSizeChooser sizeChooser;
        sizeChooser.ParseFromConfigStr("some error string");
        UNIT_ASSERT(sizeChooser.GetParseErrorString());
    }

    Y_UNIT_TEST(Parsing1) {
        THnswSearchSizeChooser sizeChooser;
        sizeChooser.ParseFromConfigStr("Default=100");
        UNIT_ASSERT(!sizeChooser.GetParseErrorString());
        TStringStream str;
        sizeChooser.DoToString(str);
        UNIT_ASSERT_NO_DIFF(str.Str(), "Default=100");
        UNIT_ASSERT_VALUES_EQUAL(sizeChooser.GetRecommendedSearchSize(100), 100);
    }

    Y_UNIT_TEST(Parsing2) {
        THnswSearchSizeChooser sizeChooser;
        sizeChooser.ParseFromConfigStr("Default=100 [0;10)=15");
        UNIT_ASSERT(!sizeChooser.GetParseErrorString());
        TStringStream str;
        sizeChooser.DoToString(str);
        UNIT_ASSERT_NO_DIFF(str.Str(), "Default=100 [0;10)=15");
        UNIT_ASSERT_VALUES_EQUAL(sizeChooser.GetRecommendedSearchSize(5), 15);
        UNIT_ASSERT_VALUES_EQUAL(sizeChooser.GetRecommendedSearchSize(0), 15);
        UNIT_ASSERT_VALUES_EQUAL(sizeChooser.GetRecommendedSearchSize(10), 100);
    }


    Y_UNIT_TEST(Parsing3) {
        THnswSearchSizeChooser sizeChooser;
        sizeChooser.ParseFromConfigStr("Default=100 [0;10)=15 [10;20)=25    [10;20)=25   [10;13)=45  [40;46)=45");
        UNIT_ASSERT(!sizeChooser.GetParseErrorString());
        TStringStream str;
        sizeChooser.DoToString(str);
        UNIT_ASSERT_NO_DIFF(str.Str(), "Default=100 [0;10)=15 [10;20)=25 [10;20)=25 [10;13)=45 [40;46)=45");
        UNIT_ASSERT_VALUES_EQUAL(sizeChooser.GetRecommendedSearchSize(5), 15);
        UNIT_ASSERT_VALUES_EQUAL(sizeChooser.GetRecommendedSearchSize(0), 15);
        UNIT_ASSERT_VALUES_EQUAL(sizeChooser.GetRecommendedSearchSize(10), 45);
        UNIT_ASSERT_VALUES_EQUAL(sizeChooser.GetRecommendedSearchSize(13), 25);
        UNIT_ASSERT_VALUES_EQUAL(sizeChooser.GetRecommendedSearchSize(20), 100);
        UNIT_ASSERT_VALUES_EQUAL(sizeChooser.GetRecommendedSearchSize(43), 45);
    }
}


Y_UNIT_TEST_SUITE(AllowedFingerPrints) {
    Y_UNIT_TEST(NoConfigInfo) {
        TAllowedFingerprintMappings allowedMappings;
        UNIT_ASSERT(!allowedMappings.GetParseErrorString());
        TStringStream str;
        allowedMappings.DoToString(str);
        UNIT_ASSERT_NO_DIFF(str.Str(), "");

        UNIT_ASSERT(allowedMappings.IsCompitable("x", "x"));
        UNIT_ASSERT(!allowedMappings.IsCompitable("x", "y"));
    }

    Y_UNIT_TEST(ErrorConfigInfo) {
        TAllowedFingerprintMappings allowedMappings;
        allowedMappings.ParseFromConfigStr("some error string");
        UNIT_ASSERT(allowedMappings.GetParseErrorString());
    }

    Y_UNIT_TEST(Parsing1) {
        TAllowedFingerprintMappings allowedMappings;
        allowedMappings.ParseFromConfigStr("x>y a>b x>z ");
        UNIT_ASSERT(!allowedMappings.GetParseErrorString());
        TStringStream str;
        allowedMappings.DoToString(str);
        UNIT_ASSERT_NO_DIFF(str.Str(), "a>b x>z x>y ");
        UNIT_ASSERT(allowedMappings.IsCompitable("x", "x"));
        UNIT_ASSERT(allowedMappings.IsCompitable("x", "y"));
        UNIT_ASSERT(allowedMappings.IsCompitable("x", "z"));
        UNIT_ASSERT(!allowedMappings.IsCompitable("x", "a"));
        UNIT_ASSERT(!allowedMappings.IsCompitable("y", "x"));
        UNIT_ASSERT(allowedMappings.IsCompitable("a", "b"));
        UNIT_ASSERT(!allowedMappings.IsCompitable("b", "a"));
    }
}

