#include "misspellcorrectorconfig.h"

#include <saas/library/daemon_base/config/daemon_config.h>

#include <library/cpp/testing/unittest/registar.h>

namespace {
    TMisspellCorrectorConfig ParseConfig(const char* text) {
        TAnyYandexConfig config;
        if (!config.ParseMemory(text)) {
            TString s;
            config.PrintErrors(s);
            UNIT_ASSERT_C(false, s);
        }
        auto children = config.GetRootSection()->GetAllChildren();
        auto it = children.find("Misspell");
        UNIT_ASSERT(it != children.end());
        return TMisspellCorrectorConfig(it->second->GetDirectives());
    }
}

Y_UNIT_TEST_SUITE(MisspellCorrectorConfig) {

    Y_UNIT_TEST(TestEmpty) {
        UNIT_ASSERT_EXCEPTION(ParseConfig(
            "<Misspell>\n"
            "</Misspell>\n"
        ), yexception);
    }

    Y_UNIT_TEST(TestDefaults) {
        auto conf = ParseConfig(
            "<Misspell>\n"
            "    Host: host.test\n"
            "    Port: 12345\n"
            "</Misspell>\n"
        );
        UNIT_ASSERT_STRINGS_EQUAL(conf.Connection.Host, "host.test");
        UNIT_ASSERT_EQUAL(conf.Connection.Port, 12345);

        UNIT_ASSERT_EQUAL(conf.EnablePorno, false);
        UNIT_ASSERT_STRINGS_EQUAL(conf.RawResposeOptions, "fix=1");
        UNIT_ASSERT_STRINGS_EQUAL(conf.CgiParametersStr, "fix=1");

        // used in misspellcorrectorfactory.cpp, to be replaced with an explicit flag
        UNIT_ASSERT_STRINGS_EQUAL(conf.RawResposeOptions, TMisspellCorrectorConfig::DefaultRawResposeOptions);
    }

    Y_UNIT_TEST(TestDisablePorno) {
        auto conf = ParseConfig(
            "<Misspell>\n"
            "    Host: host.test\n"
            "    Port: 12345\n"
            "    EnablePorno: false\n"
            "</Misspell>\n"
        );
        UNIT_ASSERT_STRINGS_EQUAL(conf.Connection.Host, "host.test");
        UNIT_ASSERT_EQUAL(conf.Connection.Port, 12345);

        UNIT_ASSERT_EQUAL(conf.EnablePorno, false);
    }

    Y_UNIT_TEST(TestEnablePorno) {
        auto conf = ParseConfig(
            "<Misspell>\n"
            "    Host: host.test\n"
            "    Port: 12345\n"
            "    EnablePorno: true\n"
            "</Misspell>\n"
        );
        UNIT_ASSERT_EQUAL(conf.EnablePorno, true);
    }

    Y_UNIT_TEST(TestRawResposeOptions) {
        auto conf = ParseConfig(
            "<Misspell>\n"
            "    Host: host.test\n"
            "    Port: 12345\n"
            "    RawResposeOptions: xxx=yyy\n"
            "</Misspell>\n"
        );
        UNIT_ASSERT_STRINGS_EQUAL(conf.RawResposeOptions, "xxx=yyy");

        // used in misspellcorrectorfactory.cpp, to be replaced with an explicit flag
        UNIT_ASSERT_STRINGS_UNEQUAL(conf.RawResposeOptions, TMisspellCorrectorConfig::DefaultRawResposeOptions);
    }

    Y_UNIT_TEST(TestEmptyRawResposeOptions) {
        auto conf = ParseConfig(
            "<Misspell>\n"
            "    Host: host.test\n"
            "    Port: 12345\n"
            "    RawResposeOptions: EMPTY\n"
            "</Misspell>\n"
        );
        UNIT_ASSERT_STRINGS_EQUAL(conf.RawResposeOptions, "");

        // used in misspellcorrectorfactory.cpp, to be replaced with an explicit flag
        UNIT_ASSERT_STRINGS_UNEQUAL(conf.RawResposeOptions, TMisspellCorrectorConfig::DefaultRawResposeOptions);
    }

    Y_UNIT_TEST(TestLog) {
        auto conf = ParseConfig(
            "<Misspell>\n"
            "    Host: host.test\n"
            "    Port: 12345\n"
            "    Log: mylogfile\n"
            "</Misspell>\n"
        );
        UNIT_ASSERT_STRINGS_EQUAL(conf.Log, "mylogfile");
    }

    Y_UNIT_TEST(TestCgiParameters) {
        auto conf = ParseConfig(
            "<Misspell>\n"
            "    Host: host.test\n"
            "    Port: 12345\n"
            "    CgiParameters: key=value%20value\n"
            "</Misspell>\n"
        );
        UNIT_ASSERT_STRINGS_EQUAL(conf.CgiParametersStr, "fix=1&key=value+value");
    }
    Y_UNIT_TEST(TestTimeout) {
        auto conf = ParseConfig(
            "<Misspell>\n"
            "    Host: host.test\n"
            "    Port: 12345\n"
            "    ConnectionTimeout: 50\n"
            "    InteractionTimeout: 200\n"
            "</Misspell>\n"
        );
        UNIT_ASSERT_VALUES_EQUAL(conf.Connection.ConnectionTimeout.MilliSeconds(), 50);
        UNIT_ASSERT_VALUES_EQUAL(conf.Connection.InteractionTimeout.MilliSeconds(), 200);
    }
}
