# coding: utf-8

from __future__ import absolute_import
from __future__ import division
from __future__ import unicode_literals

import math


LOG_SIZE_PER_REQUEST = 5 * 1024  # 5 kilobytes
LOG_TTL = 4 * 60 * 60  # 6 hours in seconds
DATA_VOLUME_MULTIPLIER = 2.5
BYTES_IN_GB = 2**30



def calculate_gencfg_volumes(index_size, memory_limit, target_rps, shards=1, replicas=1):
    """
    :param index_size: Service index size in bytes
    :param memory_limit: Gencfg group memory limit in bytes
    :param target_rps: target_rps for location
    :param shards: service shards count
    :param replicas: service replicas count per location
    :return: Dict with gencfg volumes properties
    :type index_size: int
    :type memory_limit: int
    :type target_rps: int
    :type shards: int
    :type replicas: int
    :rtype: List[Dict[str:str]]
    """
    shard_size = (index_size/shards)/BYTES_IN_GB
    log_size = (target_rps * LOG_SIZE_PER_REQUEST * LOG_TTL / replicas)/BYTES_IN_GB
    shard_size = shard_size if shard_size > 20 else 20
    cores_size = memory_limit * 2 / BYTES_IN_GB

    if log_size > 100:
        log_size = 100
    elif log_size < 20:
        log_size = 20

    data_volume_size = roundup(shard_size * DATA_VOLUME_MULTIPLIER, 10)
    logs_volume_size = roundup(log_size, 10)
    cores_volume_size = roundup(cores_size, 1)
    result = [
        {'guest_mp': '',                        'host_mp_root': '/place',   'quota': '10.0 Gb',                         'symlinks': []},
        {'guest_mp': '/',                       'host_mp_root': '/place',   'quota': '10.0 Gb',                         'symlinks': []},
        {'guest_mp': '/db/bsconfig/webstate',   'host_mp_root': '/place',   'quota': '1.0 Gb',                          'symlinks': ['/state']},
        {'guest_mp': '/cores',                  'host_mp_root': '/place',   'quota': '{} Gb'.format(cores_volume_size), 'symlinks': []},
        {'guest_mp': '/ssd',                    'host_mp_root': '/ssd',     'quota': '{} Gb'.format(data_volume_size),  'symlinks': ['/data']},
        {'guest_mp': '/logs',                   'host_mp_root': '/place',   'quota': '{} Gb'.format(logs_volume_size),  'symlinks': ['/usr/local/www/logs']},
    ]
    return result
