# -*- coding: utf-8 -*-

import atexit
import sys
import os
import time
import signal


class Daemon(object):
    """
    A generic daemon class.
    Usage: subclass the daemon class and override the run() method.
    """

    def __init__(self, pidfile, name='daemon'):
        self.__pidfile = pidfile
        self.__name = name

    def _daemonize(self):
        """
        UNIX double fork mechanism.
        """
        try:
            pid = os.fork()
            if pid > 0:
                # exit first parent
                sys.exit(0)
        except OSError as err:
            sys.stderr.write('fork #1 failed: {0}\n'.format(err))
            sys.exit(1)

        # decouple from parent environment
        os.chdir('/')
        os.setsid()
        os.umask(0)

        # do second fork
        try:
            pid = os.fork()
            if pid > 0:
                # exit from second parent
                sys.exit(0)
        except OSError as err:
            sys.stderr.write('fork #2 failed: {0}\n'.format(err))
            sys.exit(1)

        # redirect standard file descriptors
        sys.stdout.flush()
        sys.stderr.flush()
        si = open(os.devnull, 'r')
        so = open(os.devnull, 'a+')
        se = open(os.devnull, 'a+')

        os.dup2(si.fileno(), sys.stdin.fileno())
        os.dup2(so.fileno(), sys.stdout.fileno())
        os.dup2(se.fileno(), sys.stderr.fileno())

        # write pidfile
        atexit.register(self._removepid)

        pid = str(os.getpid())
        with open(self.__pidfile, 'w+') as f:
            f.write(pid + '\n')

    def _checkpid(self, pid):
        # check pid exists
        try:
            os.kill(pid, 0)
        except OSError:
            return False
        else:
            return True

    def _removepid(self):
        os.remove(self.__pidfile)

    def _getpid(self):
        """
        Get the pid from the pidfile
        """
        try:
            with open(self.__pidfile, 'r') as pf:
                pid = int(pf.read().strip())
        except IOError:
            pid = None
        return pid

    def start(self):
        """
        Start the daemon.
        """

        # Check for a pidfile to see if the daemon already runs
        sys.stderr.write("Starting {0}\n".format(self.__name))
        pid = self._getpid()
        if pid and self._checkpid(pid):
            message = "pidfile {0} already exist. {1} already running?\n"
            sys.stderr.write(message.format(self.__pidfile, self.__name))
            sys.exit(1)

        # Start the daemon
        self._daemonize()
        self.run()

    def status(self):
        """
        Check running status of the daemon
        """
        pid = self._getpid()
        try:
            os.kill(pid, 0)
        except (OSError, TypeError):
            sys.stderr.write("{0} is not running\n".format(self.__name))
        else:
            sys.stderr.write("{0} is running now. (pid {1})\n".format(self.__name, pid))

    def stop(self):
        """
        Stop the daemon
        """
        sys.stderr.write("Stopping {0}\n".format(self.__name))
        pid = self._getpid()

        if not pid:
            message = "pidfile {0} does not exist. Daemon not running?\n"
            sys.stderr.write(message.format(self.__pidfile))
            return  # not an error in a restart

        # Try killing the daemon process
        try:
            while 1:
                os.kill(pid, signal.SIGTERM)
                time.sleep(0.1)
        except OSError as err:
            e = str(err.args)
            if e.find("No such process") > 0:
                if os.path.exists(self.__pidfile):
                    os.remove(self.__pidfile)
            else:
                print(str(err.args))
                sys.exit(1)

    def restart(self):
        """
        Restart the daemon.
        """
        self.stop()
        self.start()

    def run(self):
        """
        You should override this method when you subclass Daemon.

        It will be called after the process has been daemonized by
        start() or restart().
        """
