#!/skynet/python/bin/python

"""
Script to reshare resources, which are present in local AgentR's cache but absent in skybone's database.
Works only on `NEW_LAYOUT` hosts.
"""

from __future__ import print_function

import os
import sys
import signal
import Queue as queue
import threading as th
import subprocess as sp

WORKERS = 30

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc


from sandbox.common import log as common_log
from sandbox.common import console as common_console
from sandbox.common import statistics as common_statistics

import sandbox.agentr.db
import sandbox.agentr.config
import sandbox.agentr.client


def worker(logger, agentr, q):
    this = th.current_thread().name
    logger.info("Worker %s started.", this)
    while True:
        rid = q.get()
        if not rid:
            break
        try:
            agentr.reshare(rid)
        except:
            logger.exception("Error sharing resource #%s", rid)
    logger.info("Worker %s stopped.", this)


def main():
    if "SANDBOX_CONFIG" not in os.environ:
        samogon_dir = None
        for samogon_key in range(4):
            samogon_dir = "/samogon/{}".format(samogon_key)
            if os.path.exists(samogon_dir):
                break
        if samogon_dir is None:
            raise OSError("Samogon directory not found")
        config_dir = samogon_dir + "/active/user/agentr/internal"

        config_path = None
        for f in os.listdir(config_dir):
            if f.endswith(".cfg"):
                config_path = os.path.join(config_dir, f)

        os.environ.update({"SANDBOX_CONFIG": config_path})

    config = sandbox.agentr.config.Registry()
    cz = common_console.AnsiColorizer()
    assert ctc.Tag.NEW_LAYOUT in config.client.tags, cz.red("The host is not NEW_LAYOUT'ed")

    logfile = os.path.basename(__file__) + '.log'
    common_statistics.Signaler(common_statistics.SignalHandlerForAgentR(), component=ctm.Component.AGENTR)
    logger = common_log.setup_log(logfile, 'DEBUG')
    print(cz.black("Logging to '{}'".format(logfile)), file=sys.stderr)
    db = sandbox.agentr.db.Database(
        config.agentr.daemon.db.path, logger,
        mmap=config.agentr.daemon.db.mmap,
        temp=config.agentr.daemon.db.temp,
    )
    with common_console.LongOperation("Querying the database") as op:
        db.open(check=False)
        resources = dict(db.query("""SELECT "skynet_id", "id" FROM "resource" WHERE "skynet_id" IS NOT NULL"""))
        op.intermediate("Found {} resources, which should be shared with copier.".format(len(resources)))

    with common_console.LongOperation("Querying skybone") as op:
        cmd = ["/skynet/tools/skybone-ctl", "check-resources", '-']
        proc = sp.Popen(cmd, stdout=sp.PIPE, stderr=sp.PIPE, stdin=sp.PIPE)
        stdout, stderr = proc.communicate('\n'.join(resources.iterkeys()))
        if proc.returncode:
            print(
                "`{}` returned {} error code. STDERR follows:\n{}".format(cmd[0], proc.returncode, stderr),
                file=sys.stderr
            )
            return -1
        unknown = set(filter(None, map(resources.get, filter(len, stdout.split('\n')))))
        op.intermediate("Detected {} resources, which are not known by skybone".format(len(unknown)))

    q = queue.Queue(WORKERS)
    c = sandbox.agentr.client.Service(logger)
    threads = [th.Thread(target=worker, args=(logger, c, q), name='#{}'.format(i)) for i in xrange(WORKERS)]
    map(th.Thread.start, threads)

    def finish():
        with common_console.LongOperation("Waiting for worker to be finished..."):
            map(q.put, [None] * WORKERS)
            map(th.Thread.join, threads)

    def sighandler(signum, _):
        print(cz.red("Caught signal #{}. Terminating".format(signum)), file=sys.stderr)
        finish()
        sys.exit(signum)

    signal.signal(signal.SIGINT, sighandler)
    signal.signal(signal.SIGHUP, sighandler)
    signal.signal(signal.SIGTERM, sighandler)

    pbar = common_console.ProgressBar("Reshare resources", len(unknown))
    for rid in unknown:
        q.put(rid)
        pbar.add(1)
    pbar.finish()

    finish()
    return 0


if __name__ == "__main__":
    sys.exit(main())
