#!/home/zomb-sandbox/venv/bin/python

from __future__ import absolute_import, print_function

import os
import sys
import json
import stat
import shutil
import argparse
import subprocess as sp

import distutils.util

import py

from kernel.util import console

SANDBOX_DIR = py.path.local(__file__).join(*("..",) * 3).strpath
sys.path = [os.path.dirname(SANDBOX_DIR), SANDBOX_DIR] + sys.path

from sandbox.common import os as common_os
from sandbox.common import fs as common_fs
from sandbox.common.mds import stream as mds_stream
from sandbox.common import threading as common_threading
from sandbox.common import itertools as common_itertools
from sandbox.common import platform as common_platform

import sandbox.agentr.types as ar_types
import sandbox.agentr.config as ar_config
import sandbox.agentr.errors as ar_errors
import sandbox.agentr.utils


def _ch_basedirs(owner, taskdir, node, rw=False, base=None):
    tasks_basedir = base or taskdir.join("..", "..", "..")
    while node >= tasks_basedir:
        try:
            if rw is not None:
                node.chmod(
                    node.lstat().mode | stat.S_IWUSR | stat.S_IWGRP | stat.S_IWOTH
                    if rw else
                    node.lstat().mode & ~stat.S_IWUSR & ~stat.S_IWGRP & ~stat.S_IWOTH
                )
            node.chown(owner.uid, owner.gid)
        except py.error.ENOENT:
            if not rw:
                raise
        node = node.dirpath()


def _tidy_up(src, owner, ro=True, rm_empty_files=False):
    src_path = py.path.local(src)
    for node in (src_path.visit(rec=lambda x: x.check(link=0)) if src_path.isdir() else [src_path]):
        if rm_empty_files and node.isfile() and node.size() == 0:
            try:
                node.remove()
            except py.error.EACCES:
                pass
            continue
        if ro:
            mode = node.lstat().mode & ~stat.S_IWUSR & ~stat.S_IWGRP & ~stat.S_IWOTH
            mode |= stat.S_IRUSR | stat.S_IRGRP | stat.S_IROTH
            if mode & stat.S_IFDIR:
                mode |= stat.S_IXUSR | stat.S_IXGRP | stat.S_IXOTH
            if not stat.S_ISLNK(mode):
                node.chmod(mode)
        os.lchown(node.strpath, owner.uid, owner.gid)


def move(args):
    rc = 0
    owner = common_os.User(args.owner)
    inloc, filoc = map(py.path.local, (args.src, args.dst))
    infile, fifile = inloc.join(py.path.local(args.fname).basename), filoc.join(args.fname)
    if fifile.check():
        fifile.remove(rec=1)
    fidir = fifile.dirpath()
    if not common_os.User.has_root:
        _ch_basedirs(owner, filoc, fidir, True)
    fidir.ensure(dir=1)
    shutil.move(str(infile), str(fifile))
    try:
        if args.skybone:
            skybone_admin = os.path.join(
                os.path.dirname(os.path.dirname(os.readlink("/skynet"))),
                "skycore/ns/skynet/skybone/bin/skybone-admin"
            )
            sp.check_call([
                skybone_admin, "file-move", "--quiet", "--after", infile.strpath, fifile.strpath
            ])
    except sp.CalledProcessError:
        rc = 42  # The constant is declared at `agentr.fetcher.Fetcher.WORKER_SPECIAL_EXIT_CODE`
    if args.recursive and fifile.check(dir=1):
        _tidy_up(fifile, owner, args.readonly, rm_empty_files=args.rm_empty_files)
    _ch_basedirs(owner, filoc, fifile, not args.readonly and None)
    return rc


def tidy_up(args):
    src = py.path.local(args.src)
    if src.exists():
        _tidy_up(src, common_os.User(args.owner), rm_empty_files=args.rm_empty_files)
    return 0


def ensure(args):
    owner = common_os.User(args.owner)
    taskdir = py.path.local(args.workdir)
    logdir = args.logdir and taskdir.join(args.logdir)
    logfile = args.logfile and logdir.join(args.logfile)
    if not common_os.User.has_root:
        _ch_basedirs(owner, taskdir, logfile or taskdir.dirpath(), True)
    else:
        for _ in filter(None, (taskdir, logdir, logfile)):
            if _.check():
                _.chmod(_.lstat().mode | stat.S_IWUSR | stat.S_IWGRP | stat.S_IWOTH)
            else:
                break

    if logfile:
        for path in (logfile, logdir.join(ar_types.STDERR_FILENAME), logdir.join(ar_types.STDOUT_FILENAME)):
            path.ensure(file=1)
            if common_os.User.has_root:
                path.chmod(path.lstat().mode | stat.S_IWUSR | stat.S_IWGRP | stat.S_IWOTH)
                path.chown(owner.uid, owner.gid)
        _ch_basedirs(owner, logdir, logfile, True, base=taskdir)
    else:
        taskdir.ensure(dir=1)
    _ch_basedirs(owner, taskdir, taskdir.dirpath())
    return 0


def empty(args):
    node = py.path.local(args.src)
    isfile = node.isfile()
    node.remove()
    if args.drop or isfile:
        return 0
    node.ensure(dir=1)
    if common_os.User.has_root:
        owner = common_os.User(args.owner)
        node.chown(owner.uid, owner.gid)
    return 0


def allocate(args):
    owner = common_os.User(args.owner)
    filename = py.path.local(args.filename)

    common_fs.allocate_file(filename.strpath, args.filesize)

    if common_os.User.has_root:
        filename.chown(owner.uid, owner.gid)
    return 0


def disk_usage(args):
    owner = common_os.User(args.owner)
    filename = py.path.local(args.filename)

    sandbox.agentr.utils.dump_dir_disk_usage_scandir(args.srcdir, filename.strpath)

    if common_os.User.has_root:
        filename.chown(owner.uid, owner.gid)
    return 0


def cut_log(args):
    owner = common_os.User(args.owner)
    filename = py.path.local(args.filename)

    common_fs.cut_log(args.logname, filename.strpath, args.offset, args.inode)

    if common_os.User.has_root:
        filename.chown(owner.uid, owner.gid)
    return 0


def _is_porto(container):
    return container and "sb/" in container


def _container_mount_path(container, path):
    if _is_porto(container):
        cmd = ["/usr/sbin/portoctl", "get", os.path.dirname(container), "root"]
        pctl = sp.Popen(cmd, stderr=sp.PIPE, stdout=sp.PIPE)
        out, err = pctl.communicate()
        if pctl.returncode:
            if "ContainerDoesNotExist" in err:
                raise ar_errors.ContainerNotFound("Container %s not found", container)
            raise sp.CalledProcessError(pctl.returncode, cmd, output=err)
        path = os.path.join(out.strip(), path.lstrip("/"))
    return path


def _cmd(container, args):
    return ["/usr/bin/lxc-attach", "-n", container, "--"] + args if container and not _is_porto(container) else args


def mount_image(args):
    target = py.path.local(_container_mount_path(args.container, args.target))
    if _is_porto(args.container):
        cmd = [
            "/usr/sbin/portoctl", "vcreate", str(target),
            "layers={}".format(args.source), "backend=squash", "read_only=true"
        ]
    else:
        cmd = ["/bin/mount", "-o", "loop,ro", args.source, str(target)]
    target.ensure(dir=1)
    p = sp.Popen(_cmd(args.container, cmd), stderr=sp.PIPE, stdout=sp.PIPE)
    out, err = p.communicate()
    # 12 for portod is VolumeAlreadyExists err
    if not p.returncode or p.returncode == 12:
        return 0
    print("cmd {} failed with code {}. Out: {}, err: {}".format(_cmd(args.container, cmd), p.returncode, out, err))
    return p.returncode


def mount_bind_tmp(args):
    is_porto = _is_porto(args.container)
    source = py.path.local(args.source)
    target = py.path.local(args.target)

    cmd = ["/bin/mount", "-o", "bind", str(source), str(target)]
    if args.container and not is_porto and not args.devbox:
        py.path.local(args.rootfs).join(args.target).ensure(dir=1)
    elif args.container and is_porto:
        target = py.path.local(_container_mount_path(args.container, args.target))
        target.ensure(dir=1)
        cmd = [
            "/usr/sbin/portoctl", "vcreate", str(target),
            "storage={}".format(str(source)), "backend=bind"
        ]
    else:
        target.ensure(dir=1)
    source.ensure(dir=1)
    cmd = _cmd(args.container, cmd)

    def run(cmd):
        p = sp.Popen(cmd, stderr=sp.PIPE, stdout=sp.PIPE)
        out, err = p.communicate()
        # 12 for portod is VolumeAlreadyExists err
        if not p.returncode or p.returncode == 12:
            return True
        if "Failed to get init pid" in err:
            return False
        raise sp.CalledProcessError(p.returncode, cmd, output=err)

    ret, _ = common_itertools.progressive_waiter(5, 10, 20, lambda: run(cmd), sleep_first=False)
    return int(not ret)


def mount_overlay(args):
    mount_point = py.path.local(args.mount_point)
    mount_point.ensure(dir=1)

    sp_args = ["/bin/mount", "-t", "overlay", "-o", args.options, "none", str(mount_point)]
    sp.check_call(_cmd(args.container, sp_args))
    return 0


def umount(args):
    try:
        target = py.path.local(_container_mount_path(args.container, args.target))
    except ar_errors.ContainerNotFound:
        return 0

    if _is_porto(args.container):
        cmd = ["/usr/sbin/portoctl", "vunlink", str(target)]
    elif common_platform.on_osx():
        cmd = ["diskutil", "unmount", "force", str(target)]
    else:
        cmd = ["/bin/umount", "-ld", str(target)]
    p = sp.Popen(_cmd(args.container, cmd), stderr=sp.PIPE)
    _, umount_err = p.communicate()
    return_code = p.returncode
    # umount on any error returns code 32 and there is no way to distinguish concrete cause
    if (
        return_code and
        "no mount point specified" not in umount_err and
        "not found" not in umount_err and
        "not mounted" not in umount_err and
        "VolumeNotFound" not in umount_err
    ):
        print(umount_err, file=sys.stderr)

        if common_platform.on_osx():
            # errors happen on osx during unmount, just pass message to stderr
            return 0

        lsof_cmd = _cmd(args.container, ["/usr/bin/lsof", "+D", str(target)])
        p = sp.Popen(lsof_cmd, stdout=sp.PIPE, stderr=sp.STDOUT)
        lsof_out, _ = p.communicate()
        print(
            "Unmounting of {target} failed with code {rc}\n"
            "{lsof_cmd} finished with code {lsof_rc} and printed:\n{lsof_out}".format(
                target=target, rc=return_code, lsof_cmd=lsof_cmd, lsof_rc=p.returncode, lsof_out=lsof_out,
            )
        )
        sys.exit(return_code)

    if args.remove_parent or args.remove:
        if args.rootfs and not _is_porto(args.container):
            target = py.path.local(args.rootfs).join(target)
        if args.remove_parent:
            target = target.join("..")
        if target.check():
            target.remove(rec=1)

    return 0


def chown(args):
    owner = common_os.User(args.owner)
    sp.check_call(
        _cmd(
            args.container,
            ["/bin/chown", "-hR", "{}:{}".format(owner.login, owner.group),
             _container_mount_path(args.container, args.path)]
        )
    )


def _clear_tmp_git_pack(repo_path):
    try:
        with common_threading.RFLock(repo_path + ".lock"):
            tmp_path = os.path.join(repo_path, "objects", "pack")
            if not os.path.exists(tmp_path) or not os.path.isdir(tmp_path):
                return
            for filename in os.listdir(tmp_path):
                if not filename.startswith("tmp_pack_"):
                    continue
                full_filepath = os.path.join(tmp_path, filename)
                try:
                    os.remove(full_filepath)
                except Exception as ex:
                    print("Error on removing file {}: {}".format(full_filepath, ex))
    except Exception as ex:
        print("Unexpected exception on cleaning tmp git files: {}".format(ex))


def git_gc(args):
    _clear_tmp_git_pack(args.repo)

    with common_os.User.Privileges(common_os.User.service_users.unprivileged.login):
        with common_threading.RFLock(args.repo + ".lock"):
            cmd = ["git", "gc", "--prune=3.days.ago"]
            pctl = sp.Popen(cmd, stderr=sp.PIPE, stdout=sp.PIPE, cwd=args.repo)
            out, err = pctl.communicate()
            if pctl.returncode:
                _clear_tmp_git_pack(args.repo)
                print("Process {} exited with {}. Out:\n{}\nErr:\n{}".format(cmd, pctl.returncode, out, err))
    sys.exit(0)


def download(args):
    resource = json.loads(args.resource)
    resource_path = args.resource_path
    config = ar_config.Registry()
    downloaded_resource = mds_stream.read_from_mds(resource, resource_path, config=config)
    if downloaded_resource is None:
        return ar_types.MDS_DOWNLOAD_NOT_ALLOWED
    print(downloaded_resource)
    return 0


def main():
    console.setProcTitle(ar_types.PROCESS_TITLE_PREFIX + " - worker")

    parser = argparse.ArgumentParser(
        formatter_class=argparse.RawTextHelpFormatter,
        description="Postprocess just downloaded resource data"
    )
    subparsers = parser.add_subparsers(help="operational mode")

    subparser = subparsers.add_parser(
        "move",
        help="relocate files from download directory to permanent place"
    )
    subparser.add_argument("owner", metavar="OWNER", type=str, help="User name to own files")
    subparser.add_argument("src", metavar="SRC", type=str, help="Source directory")
    subparser.add_argument("dst", metavar="DST", type=str, help="Destination directory")
    subparser.add_argument("fname", metavar="NAME", type=str, help="File/directory name")
    subparser.add_argument(
        "--skybone", type=distutils.util.strtobool, default=True,
        help="Notify skybone about data relocation"
    )
    subparser.add_argument(
        "--readonly", type=distutils.util.strtobool, default=True,
        help="Make the destination readonly or change owner only"
    )
    subparser.add_argument(
        "--recursive", action="store_true",
        help="Change owner and permissions recursively"
    )
    subparser.add_argument("--rm_empty_files", action="store_true", help="Remove empty files")
    subparser.set_defaults(func=move)

    subparser = subparsers.add_parser(
        "tidy_up",
        help="take ownership on the files recursively, optionally remove empty files"
    )
    subparser.add_argument("--rm_empty_files", action="store_true", help="Remove empty files")
    subparser.add_argument("owner", metavar="OWNER", type=str, help="User name to own files")
    subparser.add_argument("src", metavar="SRC", type=str, help="Source directory")
    subparser.set_defaults(func=tidy_up)

    subparser = subparsers.add_parser(
        "ensure",
        help="ensure task's logging sub-directory existance and logfile is writable"
    )
    subparser.add_argument("owner", metavar="OWNER", type=str, help="User name to own files")
    subparser.add_argument("workdir", metavar="WORKDIR", type=str, help="Task working directory path")
    subparser.add_argument("logdir", metavar="LOGDIR", nargs="?", type=str, help="Sub-directory name")
    subparser.add_argument("logfile", metavar="LOGFILE", nargs="?", type=str, help="Log file name")
    subparser.set_defaults(func=ensure)

    subparser = subparsers.add_parser(
        "empty",
        help="empty resource's directory (on old layout, it may also be a single file)"
    )
    subparser.add_argument("owner", metavar="OWNER", type=str, help="User name to own files")
    subparser.add_argument("src", metavar="SRC", type=str, help="Source directory")
    subparser.add_argument("-D", "--drop", action="store_true", help="Drop the directory itself")
    subparser.set_defaults(func=empty)

    subparser = subparsers.add_parser(
        "allocate",
        help="allocate a file of specified size",
    )
    subparser.add_argument("owner", metavar="OWNER", type=str, help="User name to own files")
    subparser.add_argument("filename", metavar="FILENAME", type=str, help="File name")
    subparser.add_argument("filesize", metavar="FILESIZE", type=int, help="File size")
    subparser.set_defaults(func=allocate)

    subparser = subparsers.add_parser(
        "disk_usage",
        help="create a file containing disk usage information in a source directory",
    )
    subparser.add_argument("owner", metavar="OWNER", type=str, help="User name to own files")
    subparser.add_argument("filename", metavar="FILENAME", type=str, help="File name")
    subparser.add_argument("srcdir", metavar="SRCDIR", type=str, help="Source directory")
    subparser.set_defaults(func=disk_usage)

    subparser = subparsers.add_parser(
        "cut_log",
        help="dump a log starting from specified offset",
    )
    subparser.add_argument("owner", metavar="OWNER", type=str, help="User name to own files")
    subparser.add_argument("filename", metavar="FILENAME", type=str, help="Output file name")
    subparser.add_argument("logname", metavar="LOGNAME", type=str, help="Input log name")
    subparser.add_argument("offset", metavar="OFFSET", type=int, help="Log offset")
    subparser.add_argument("inode", metavar="INODE", type=int, help="Log inode")
    subparser.set_defaults(func=cut_log)

    subparser = subparsers.add_parser(
        "mount_image",
        help="mount given SquashFS image at the location specified",
    )
    subparser.add_argument("source", type=str, help="Image path")
    subparser.add_argument("target", type=str, help="Mount point")
    subparser.add_argument("--container", type=str, help="Container to mount image in")
    subparser.set_defaults(func=mount_image)

    subparser = subparsers.add_parser(
        "mount_bind_tmp",
        help="bind-mount source directory to target"
    )
    subparser.add_argument("source", type=str, help="What we should bind")
    subparser.add_argument("target", type=str, help="To where we should bind source")
    subparser.add_argument("--container", type=str, help="Container to mount image in")
    subparser.add_argument("--rootfs", type=str, help="Path to container rootfs in dom0")
    subparser.add_argument("--devbox", action="store_true", help="Used in devbox installation")
    subparser.set_defaults(func=mount_bind_tmp)

    subparser = subparsers.add_parser(
        "mount_overlay",
        help="mount overlay at the location specified",
    )
    subparser.add_argument("--mount-point", metavar="PATH", type=str, help="Mount point")
    subparser.add_argument("--options", type=str, help="Mount options")
    subparser.add_argument("--container", type=str, help="LXC container to mount overlay in")
    subparser.set_defaults(func=mount_overlay)

    subparser = subparsers.add_parser(
        "umount",
        help="unmount the location specified",
    )
    subparser.add_argument("target", type=str, help="Mount point")
    subparser.add_argument("--container", type=str, help="Container to unmount image in")
    subparser.add_argument("--rootfs", type=str, help="Path to container rootfs in dom0")
    subparser.add_argument("--remove", action="store_true", help="Remove the target after unmounting")
    subparser.add_argument("--remove-parent", action="store_true", help="Remove parent directory after unmounting")
    subparser.set_defaults(func=umount)

    subparser = subparsers.add_parser(
        "chown",
        help="change owner of files from host system or from given LXC (for example, it may be helpful for tmpfs)",
    )
    subparser.add_argument("owner", metavar="OWNER", type=str, help="User name to own files")
    subparser.add_argument("path", metavar="SRC", type=str, help="Directory path")
    subparser.add_argument("--container", type=str, help="LXC container where files are located")
    subparser.set_defaults(func=chown)

    subparser = subparsers.add_parser(
        "git_gc",
        help="Start git gc in repo in path param"
    )
    subparser.add_argument("repo", type=str, help="Repo path")
    subparser.set_defaults(func=git_gc)

    subparser = subparsers.add_parser(
        "download",
        help="Download resource via http from mds"
    )
    subparser.add_argument("resource", metavar="RESOURCE", type=str, help="Resource json")
    subparser.add_argument("resource_path", metavar="RESOURCE_PATH", type=str, help="Resource directory")
    subparser.set_defaults(func=download)

    # Parse the args and call whatever function was selected.
    args = parser.parse_args()
    console.setProcTitle(ar_types.PROCESS_TITLE_PREFIX + " - worker - " + args.func.__name__)
    return args.func(args)


if __name__ == "__main__":
    sys.exit(main())
