import gevent
import logging
import random

import sandbox.agentr.storage as storage
import sandbox.agentr.utils as utils
import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
import sandbox.common.itertools as common_itertools


class Cleaner(object):
    def __init__(
        self, db, rest, config, users, buckets, tasks,
        interval_seconds, chunk_size, rest_drop_batch_size
    ):
        self.__logger = logging.getLogger("cleaner")
        self.__db = db
        self.__rest = rest
        self.__config = config
        self.__users = users
        self.__buckets = buckets
        self.__tasks = tasks
        self.__interval_seconds = interval_seconds
        self.__chunk_size = chunk_size
        self.__rest_drop_batch_size = rest_drop_batch_size
        self.__greenlet = None
        self.__running = None

    def start(self):
        """Start loop with cleaning"""
        self.__greenlet = gevent.spawn(self.__loop)
        self.__running = True

    def stop(self):
        self.__running = False

    def cleanup(self, chunk_size, job):
        """Cleanup can be called manually via agent client, e.g. when there's not enough space"""
        return self._cleanup(chunk_size, job)

    def __loop(self):
        delay = random.randint(self.__interval_seconds / 10, self.__interval_seconds)
        self.__logger.info("Sleeping (random) %d minutes before starting cleaner greenlet...", delay / 60)
        gevent.sleep(delay)

        self.__logger.info("Started cleaner greenlet")
        while self.__running:
            self.__logger.info("Looping...")
            chunk_size = self.__chunk_size * 3 if ctc.Tag.STORAGE in self.__config.client.tags else self.__chunk_size
            cleanup_result = None
            try:
                kinds = {ctc.RemovableResources.DELETED, ctc.RemovableResources.REPLICATED}
                cleanup_result = self._cleanup(chunk_size, kinds=kinds)
            except Exception as ex:
                self.__logger.exception("Exception occurred while looping _cleanup", ex)
            if not cleanup_result:
                self.__logger.debug("Going to sleep for %d seconds", self.__interval_seconds)
                gevent.sleep(self.__interval_seconds)

    def _cleanup(self, chunk_size, job=None, kinds=None):
        """Cleaning itself"""
        rest_client = self.__rest.client[self.__config.this.id]

        removable = set()
        for kind in ctc.RemovableResources:
            if kinds is not None and kind not in kinds:
                continue

            removable = self.__get_removable(rest_client, kind, chunk_size)
            if removable:
                break

        if not removable:
            self.__logger.info("No resources can be removed on the host.")
            return

        if not self.__running and job is None:
            return

        self.__logger.info("Dropping remote server records about dropped resources.")
        for chunk in common_itertools.chunker(list(removable), self.__rest_drop_batch_size):
            self.__logger.debug("Dropping remote server records chunk of length %d.", len(chunk))
            rest_client.service.resources.drop(chunk)

        self.__logger.info("Checking local DB records.")
        cache = self.__buckets.local_resources(removable)
        self.__logger.info("%d local DB records found. Dropping records.", len(cache))

        task_id = None
        tee = None
        if job:
            ses = self.__tasks.service(job.connection.id)
            tee = utils.LogTee(job.log, ses.logger(job)) if ses and ses.id else job.log
            task_id = ses.id
        fetcher = storage.Fetcher(tee, self.__logger, task_id, self.__rest, self.__config, self.__users, self.__buckets)

        if not cache or len(cache) < len(removable) / 2 and not fetcher.new_layout:
            # FIXME: Special crutch for old-layout hosts with SDK1-registered resources
            meta = (self.__rest << self.__rest.HEADERS({ctm.HTTPHeader.NO_LINKS: "true"})).resource.read(
                id=",".join(map(str, removable)), limit=len(removable)
            )
            cache = [(None, _["id"], _) for _ in meta["items"]]
        self.__buckets.drop_records(removable)

        self.__logger.info("Removing disk data.")
        fetcher.dropdirs(cache)
        return kind, len(removable), len(cache)

    def __get_removable(self, rest_client, kind, chunk_size):
        all_locked = self.__tasks.locked

        removable = set(rest_client.service.resources.read(kind=kind, limit=chunk_size))
        self.__logger.info(
            "Server reported %d %s resource which can be dropped from the host: %r",
            len(removable), kind.upper(), sorted(removable)
        )

        if not removable:
            return set()

        locked = removable & all_locked
        if locked:
            self.__logger.info(
                "%d resources are locked right now and cannot be removed: %r",
                len(locked), sorted(locked)
            )
            removable -= locked

        if kind == ctc.RemovableResources.REPLICATED:
            all_expired = self.__tasks.cache_expired()
            expired = removable & all_expired
            self.__logger.info("Only %d of them expired locally and are allowed to be removed", len(expired))
            removable = expired

        return removable
