from __future__ import absolute_import

import os
import sys
import json
import threading as th

from sandbox.common import format
from sandbox.common import patterns
from sandbox.common import itertools

import sandbox.common.vcs.cache
import sandbox.common.types.client as ctc
import sandbox.common.joint.client as jclient
import sandbox.common.joint.errors as jerrors

from sandbox.common.windows import wsl

from .. import types
from .. import config


class Service(jclient.BaseServiceClient):
    """ Wrapper for the AgentR RPC client. """

    SERVICE_APPEARANCE_WAIT = 300  # Discussion here: SAMOGON-613

    def __init__(self, logger):
        """
        Constructor. Do not establishes the actual connection to the AgentR service immediately -
        it will be established on demand.

        param logger:       Logger to be used for logging
        """
        super(Service, self).__init__(logger.getChild("agentr"))
        self._config = cfg = config.Registry().agentr.daemon
        host, port = (
            (cfg.server.host, cfg.server.port)
            if cfg.server.unix is None else
            (cfg.server.unix, None)
        )
        self._srv = jclient.RPCClient(cfg=cfg.rpc, host=host, port=port)

    @staticmethod
    def local_path(path):
        if path and sys.platform == "win32":
            return wsl.FS.win_path(path, rootfs=config.Registry().agentr.wsl_rootfs)
        return path

    @staticmethod
    def remote_path(path):
        if path and sys.platform == "win32":
            return wsl.FS.wsl_path(path)
        return path

    ###########################################################################
    # Proxy for server-side methods
    ###########################################################################
    def reset(self, hard=False):
        return self("reset", hard)

    def freeze(self):
        return self("freeze")

    def ease(self):
        return self("ease")

    def ban(self, no):
        return types.DiskSpaceInfo(*self("ban", no))

    def unban(self, no):
        return types.DiskSpaceInfo(*self("unban", no))

    def banned(self):
        return {no: types.DiskSpaceInfo(*df) for no, df in self("banned").iteritems()}

    def df(self):
        return types.DiskSpaceInfo(*self("df"))

    def cleanup(self, chunk=None):
        res = self("cleanup", chunk)
        return types.CleanupResult(*res) if res else types.CleanupResult(None, 0, 0)

    def maintain(self, dry=True, **limits):
        return types.MaintainLimits(*self("maintain", dry, **limits))

    def restore_links(self):
        return self("restore_links")

    def backup(self):
        return self("backup")

    def patch_info(self, pullrequest_id, iteration=None, published=True, pr_info=None):
        return self("patch_info", pullrequest_id, iteration, published, pr_info)

    def reshare(self, rid):
        return self("reshare", rid)

    def bucket_cache_info(self, bid):
        return types.CacheInfo(*self("bucket_cache_info", bid))

    def erase_bucket(self, bid):
        return self("erase_bucket", bid)

    def jobs(self):
        return {key: json.loads(value) for key, value in self("get_jobs").iteritems()}

    @patterns.singleton_property
    def fileserver_meta(self):
        # noinspection PyMethodParameters
        class Wrapper(object):
            def __getitem__(__, tid):
                res = self("fileserver_meta_getter", tid)
                return res if res is None else types.FileServerMeta(**res)

            def __setitem__(__, tid, commands):
                raise NotImplementedError("You can only save fileserver_meta for task via agentr.client.Session")

        return Wrapper()

    @patterns.singleton_property
    def progress_meta(self):
        # noinspection PyMethodParameters
        class Wrapper(object):
            def __check_caller(__, message="You can only operate on progress meter data via agentr.client.Session"):
                if not isinstance(self, Session):
                    raise NotImplementedError(message)

            def by_task(inner_self, tid=None):
                """
                :rtype: list of sandbox.agentr.types.Action
                """

                if tid is None:
                    inner_self.__check_caller("When working outside of task session, you must specify task identifier")
                value = self("progress_meta_getter", tid)
                return value and [types.Action(*item) for item in value]

            def insert(inner_self, meter_id, data):
                """
                :type data: sandbox.agentr.types.Action
                """

                inner_self.__check_caller()
                self("progress_meta_setter", meter_id, tuple(data))

            def delete(inner_self, meter_id):
                inner_self.__check_caller()
                return self("progress_meta_deleter", meter_id=meter_id)

        return Wrapper()

    @property
    def stopped(self):
        value = self("pragma_getter", "stopped")
        return bool(value)

    @stopped.setter
    def stopped(self, value):
        self("pragma_setter", "stopped", "1" if value else "")

    @property
    def getajob_token(self):
        return self("pragma_getter", "getajob_token")

    @getajob_token.setter
    def getajob_token(self, value):
        self("pragma_setter", "getajob_token", value, __censored=True)

    @property
    def dropajobs_info(self):
        value = self("pragma_getter", "dropajobs_info")
        return json.loads(value) if value else {}

    @dropajobs_info.setter
    def dropajobs_info(self, value):
        self("pragma_setter", "dropajobs_info", json.dumps(value) if value else None, __censored=True)

    @property
    def iproutes(self):
        value = self("pragma_getter", "iproutes")
        return json.loads(value) if value else None

    @iproutes.setter
    def iproutes(self, value):
        self("pragma_setter", "iproutes", json.dumps(value) if value else None)

    def register_coredump(self, pid, filename, rpid, core_limit=None, base_filename=None):
        return self("register_coredump", pid, filename, rpid, core_limit=core_limit, base_filename=None)

    def cleanup_work_directory(self):
        return self("cleanup_work_directory")


class Session(Service):
    """
    The class is a wrapper for the AgentR RPC client. It remembers the task session token
    and restores the connection implicitly.
    """

    def __init__(self, token, iteration, reserved, logger, local=False):
        """
        Constructor. Do not establishes the actual connection to the AgentR service immediately -
        it will be established on demand.

        param token:        Task session token to associate with the connectiton.
        param iteration:    Task execution iteration (0 for first run).
        param reserved:     Reserved amount of disk space for the session.
        param logger:       Logger to be used for logging
        param local:        Do not merge different instances into a single connection (so far as single task session).
        """

        super(Session, self).__init__(logger)
        self.__token = token
        self.__iteration = iteration
        self.__reserved = reserved
        self.__logdir = None
        self.__lxc = None
        self.__lock = th.RLock()
        if local:
            self._srv._reactor_key += hex(id(self))[1:].replace("x", ":")

    @classmethod
    def service(cls, logger):
        """ Initiate service session. """
        return cls(ctc.ServiceTokens.SERVICE_TOKEN, 1, 0, logger)

    @classmethod
    def taskbox(cls, logger):
        return cls(ctc.ServiceTokens.TASKBOX_TOKEN, 1, 0, logger)

    @property
    def token(self):
        return self.__token

    @property
    def iteration(self):
        return self.__iteration

    @property
    def logdir(self):
        if self.__logdir:
            return self.local_path(self.__logdir)
        with self.__lock:
            if self.__logdir:
                return self.local_path(self.__logdir)
            r = self._reactor
            self.logger.debug(
                "{%s} Registering task session %r %r on the connection",
                self._sid, format.obfuscate_token(self.__token), self.__iteration
            )
            c = self._srv.call("task_session", self.__token, self.__iteration, self.__reserved)
            self.__logdir = c.wait()
            self.logger.debug(
                "{%s} Task session %r associated with the connection",
                self._sid, format.obfuscate_token(self.__token)
            )
            # update stored sid that late so other threads retrieve it only after task_session finishes
            self._sid = r.sid
        return self.local_path(self.__logdir)

    def __call__(self, method, *args, **kwargs):
        reactor = self._reactor
        try:
            if self.__token and reactor.sid != self._sid:
                self.__logdir = None
                assert self.logdir
            if method is None:  # Client asked to just establish a connection.
                return
            censored = kwargs.pop("__censored", None)
            c = self._srv.call(method, *args, **kwargs)
            if censored:
                self.logger.debug("{%s:%s} Calling remote method %s(...)", c.sid, c.jid, method)
            else:
                self.logger.debug("{%s:%s} Calling remote method %s(%r, %r)", c.sid, c.jid, method, args, kwargs)
            return c.wait()
        except jerrors.Reconnect:
            self._sid = None
            self.logger.info("Server asked to reconnect it.")
        self._srv.disconnect()
        return self(method, *args, **kwargs)

    ###########################################################################
    # Proxy for server-side methods
    ###########################################################################
    @property
    def meta(self):
        return self("task_meta")

    @meta.setter
    def meta(self, data):
        self("task_meta_setter", data, __censored=True)

    @property
    def lxc(self):
        return self.__lxc

    @lxc.setter
    def lxc(self, value):
        self.__lxc = value
        self("task_session", self.__token, self.__iteration, self.__reserved, lxc=self.__lxc, __censored=True)

    @property
    def log_resource(self):
        return self("task_logs_meta")

    def resource_sync(self, rid, fastbone=True, restore=False):
        return self.local_path(self("resource_sync", rid, fastbone=fastbone, restore=restore))

    def resource_meta(self, rid):
        return self("resource_meta", rid)

    def resource_register(
        self, path, type, name, arch, attrs, share=True, service=False, for_parent=False, depth=None,
        resource_meta=None, system_attributes=None
    ):
        return self(
            "resource_register", self.remote_path(path), type, name, arch, attrs, share, service, for_parent, depth,
            resource_meta=resource_meta, system_attributes=system_attributes
        )

    def resource_register_meta(self, path, meta, share=True, service=False, depth=None, add_source=False):
        return self(
            "resource_register_meta",
            self.remote_path(path), meta, share=share, service=service, depth=depth, add_source=add_source
        )

    def resource_update(self, resource_id, path=None, name=None, arch=None, attrs=None, depth=None):
        return self("resource_update", resource_id, self.remote_path(path), name, arch, attrs, depth)

    def resource_complete(self, rid, broken=False):
        return self("resource_complete", rid, broken=broken)

    def resource_delete(self, rid):
        return self("resource_delete", rid)

    def finished(self, drop_session=True, mark_as_ready=True):
        return self("task_finished", drop_session=drop_session, mark_as_ready=mark_as_ready)

    def monitoring_setup(self):
        return self("monitoring_setup")

    def monitoring_finish(self):
        return self("monitoring_finish")

    def monitoring_status(self):
        return self("monitoring_status")

    def monitoring_add_log(self, path, label):
        return self("monitoring_add_log", path, label)

    def register_atop(self, pid):
        return self("register_atop", pid)

    def register_tcpdump(self, pid):
        return self("register_tcpdump", pid)

    def hard_link(self, source, target=""):
        return self("hard_link", source, target)

    def register_log(self, file_name, log_type=types.LogType.TASK):
        return self("register_log", file_name, log_type=log_type)

    def register_process(self, pid):
        return self("register_process", pid)

    def git_repos_gc(self):
        return self("git_repos_gc")

    @property
    def coredumps(self):
        return self("coredumps")

    def arcadia_hg_clone(self, rev="default"):
        from sandbox.sdk2 import environments
        environments.SandboxHgEnvironment().prepare()
        hg_env = os.environ.copy()
        cache_dir, task_id = self("arcadia_hg_clone", hg_env, rev=rev)
        cache = sandbox.common.vcs.cache.CacheableHg()
        cache.add_cache_metadata(
            cache_dir, task_id, is_permanent=True,
        )
        cache.add_cache_metadata(
            cache_dir + ".check", task_id, is_permanent=True,
        )
        return cache_dir

    @property
    def state(self):
        value = self("job_state_getter")
        if value is not None:
            return json.loads(value)

    @state.setter
    def state(self, value):
        self("job_state_setter", json.dumps(value), __censored=True)

    @property
    def fileserver_meta(self):
        res = self("fileserver_meta_getter")
        return res if res is None else types.FileServerMeta(**res)

    # noinspection PyMethodOverriding
    @fileserver_meta.setter
    def fileserver_meta(self, fs_meta):
        self("fileserver_meta_setter", fs_meta._asdict())

    @property
    def executor_pid(self):
        return self("executor_pid_getter")

    def register_executor(self, pid=None):
        self("register_executor", pid)

    def dropper(self, files, workers=None):
        self("dropper", list(files), workers=workers, __censored=True)

    def mount_image(self, image, dirname=None):
        return self("mount_image", str(image), dirname=dirname)

    def prepare_tmp(self):
        return self("prepare_tmp")

    def mount_overlay(self, mount_point, lower_dirs, upper_dir="", work_dir=""):
        lower_dirs = list(itertools.chain(lower_dirs))
        upper_dir = upper_dir and str(upper_dir)
        work_dir = work_dir and str(work_dir)
        return self("mount_overlay", str(mount_point), lower_dirs, upper_dir, work_dir)

    def umount(self, mount_point):
        return self("umount", str(mount_point))

    def turbo_boost(self, enable):
        return self("turbo_boost", bool(enable))

    def prepare_xcode(self, version):
        return self("prepare_xcode", version)

    def prepare_xcode_simulator(self, version):
        return self("prepare_xcode_simulator", version)

    def prepare_host_for_multixcode(self):
        return self("prepare_host_for_multixcode")

    def dependant_resources(self):
        return self("dependant_resources")

    def remove_dependant_resource(self, rid):
        return self("remove_dependant_resource", rid)

    def check_task_files(self, task_id, task_resources):
        return self("check_task_files", task_id, task_resources)
