import sandbox.agentr.deblock

import gevent.hub
import gevent.lock
import gevent.event
import gevent.monkey
import gevent.timeout
import gevent.subprocess as sp

import os
import sys
import glob
import time
import json
import errno
import shutil
import socket
import logging
import httplib
import calendar
import datetime as dt
import functools as ft

import py
import yaml
import apsw
import aniso8601

import sandbox.common.platform as scp
import sandbox.common.types.misc as ctm
import sandbox.common.types.user as ctu
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr
import sandbox.common.types.statistics as cts

from sandbox.common import os as common_os
from sandbox.common import fs as common_fs
from sandbox.common import mds as common_mds
from sandbox.common import tvm as common_tvm
from sandbox.common import auth as common_auth
from sandbox.common import enum as common_enum
from sandbox.common import rest as common_rest
from sandbox.common import config as common_config
from sandbox.common import format as common_format
from sandbox.common import system as common_system
from sandbox.common import encoding as common_encoding
from sandbox.common import platform as common_platform
from sandbox.common import patterns as common_patterns
from sandbox.common import itertools as common_itertools
from sandbox.common import statistics as common_statistics

from sandbox.common.windows import wsl
from sandbox.common.vcs import cache as vcs_cache

import sandbox.common.joint.server as jserver

import sandbox.agentr.db
import sandbox.agentr.utils
import sandbox.agentr.types
import sandbox.agentr.config
import sandbox.agentr.errors
import sandbox.agentr.deblock
import sandbox.agentr.session
import sandbox.agentr.storage
import sandbox.agentr.cleaner as cleaner


class Daemon(jserver.RPC):
    """ AgentR daemon """
    PROC_TITLE = "AgentR"
    COREDUMP_TIMEOUT = 300  # time to wait core dump complete, in seconds
    MAX_RESOURCES_IN_A_REQUEST = 10000  # amount of resource IDs to be operated in a single REST API request
    DB_MIGRATION_PREFIX = "sandbox/agentr/share/db/"

    NO_TURBO_PATH = "/sys/devices/system/cpu/intel_pstate/no_turbo"

    Fetcher = sandbox.agentr.storage.Fetcher

    class Event(common_enum.Enum):
        STOP_HARDLY = None
        STOP_GRACEFULLY = None

    _stopping = gevent.event.AsyncResult()

    def __init__(self, config, users):
        """
        Constructor

        :param config: settings object
        """
        self._config = config or sandbox.agentr.config.Registry()
        self._logger = logging.getLogger("agentr")
        self.users = users

        ctx = sandbox.agentr.config.Context(config)
        super(Daemon, self).__init__(ctx)
        self.__server = jserver.Server(ctx)
        self.__server.register_connection_handler(self.get_connection_handler())
        self.__inprogress = {}

        self.__db_deblock = sandbox.agentr.deblock.Deblock(logger=self.log.getChild("dbdeblock"))
        self.__db = self.__db_deblock.make_proxy(self._db_factory())
        ini, fw, bw = self._load_migrations_scripts()
        self._logger.info("Updating database %r", config.agentr.daemon.db.path)
        self.__db.migrate(ini, fw, bw)
        self.__buckets = sandbox.agentr.storage.Buckets(
            self._logger, self.__db,
            config.agentr.data.buckets,
            config.agentr.data.banned,
            config.client.resources.data_dir,
            config.agentr.data.max_bucket_locks,
            config.agentr.data.bucket_semaphore,
            config.agentr.cleaner.cache_expiration_created_hours,
            config.agentr.cleaner.cache_expiration_fetched_hours,
        )
        self.__buckets.info()
        self.__buckets.cache_info()
        self.__tasks = sandbox.agentr.session.Registry(
            self.__db, self._logger, self._config, self._rest, self.users.unprivileged
        )
        self.__hardware_statistics_pusher = None
        self.__mds_uploader_delay_pusher = None
        self.__loop_device_cleaner = None
        self.__turbo_boost_watchdog_greenlet = None
        self.__cleaner = None
        self._logger.info("Restored %d task session(s): %r", len(self.__tasks), sorted([_.id for _ in self.__tasks]))

    def _load_migrations_scripts(self):
        ini, fw, bw = {}, {}, {}
        if common_system.inside_the_binary():
            import library.python.resource
            for rname, _ in library.python.resource.iteritems():
                if rname.startswith(self.DB_MIGRATION_PREFIX):
                    name = rname[len(self.DB_MIGRATION_PREFIX):]
                    t, no = name.split("_", 2)[:2]
                    (fw if t == "fw" else (bw if t == "bw" else ini)).setdefault(int(no), []).append(rname)
        else:
            for _ in sorted(py.path.local(__file__).join("..", "share", "db").listdir()):
                if _.ext in (".sql", ".py"):
                    t, no = _.basename.split("_", 2)[:2]
                    (fw if t == "fw" else (bw if t == "bw" else ini)).setdefault(int(no), []).append(_)
        return ini, fw, bw

    def _db_factory(self, lite=False):
        db = sandbox.agentr.db.Database(
            self._config.agentr.daemon.db.path, self.log,
            mmap=self._config.agentr.daemon.db.mmap,
            temp=self._config.agentr.daemon.db.temp,
        )
        db.set_debug()
        db.open(force=common_platform.on_osx() or common_platform.on_wsl(), lite=lite)
        return db

    def _on_fork(self):
        self.__server.on_fork()
        common_statistics.Signaler().reset()
        del common_statistics.Signaler.instance

    def start(self):
        self._logger.info("Starting")
        common_mds.S3.set_s3_logging_level(logging.INFO)
        socket.getaddrinfo("localhost", None)  # ensure network initialization before any forking
        sandbox.agentr.storage.Quota.start_dropper(self.__db, self._config, self._logger)
        sandbox.agentr.storage.MdsUploader.start(
            self._logger, self.__db, self.__buckets, self._config, self._rest, self._on_fork
        )
        self.__cleaner = cleaner.Cleaner(
            self.__db, self._rest, self._config, self.users, self.__buckets, self.__tasks,
            self._config.agentr.cleaner.interval_seconds,
            self._config.agentr.cleaner.chunk_size,
            self._config.agentr.cleaner.rest_drop_batch_size,
        )
        self.__cleaner.start()
        if common_os.CGroup.mounted:
            self.__hardware_statistics_pusher = gevent.spawn(self.__tasks.hardware_metrics_pusher)
        self.__mds_uploader_delay_pusher = gevent.spawn(self.__tasks.mds_uploader_delay_pusher)
        if (
            ctc.Tag.LXC in self._config.client.tags and
            self._config.common.installation not in ctm.Installation.LOCAL
        ):
            self.__loop_device_cleaner = gevent.spawn(self.__tasks.loop_device_cleaner)
        if os.path.exists(self.NO_TURBO_PATH) and ctc.Tag.MULTISLOT not in self._config.client.tags:
            self.__turbo_boost_watchdog_greenlet = gevent.spawn(self.__turbo_boost_watchdog, self.__turbo_boost)

        super(Daemon, self).start()
        self.__server.start()
        self._logger.info("Started")
        return self

    def loop(self):
        self._logger.info("Entering main loop.")
        minutes = 0
        while True:
            try:
                ev = self._stopping.get(timeout=60)
                minutes += 1
            except gevent.timeout.Timeout:
                # Some service activity can be performed here
                continue
            except gevent.hub.LoopExit:
                self._logger.info("Gevent's loop finished.")
                break
            self._logger.info("Got an event %r", ev)
            if ev == self.Event.STOP_GRACEFULLY:
                sandbox.agentr.storage.MdsUploader.stop(True)
                self._stopping = gevent.event.AsyncResult()
                self.stop(True)
                continue
            break
        self._logger.info("The main loop finished.")
        self.stop()

    def stop(self, graceful=False):
        st = self.counters
        self._logger.info("Stopping %s. Current connections: %r", "gracefully" if graceful else "hardly", st)
        self.__server.stop()
        if graceful and st:
            return super(Daemon, self).stop(graceful)
        self.stop = lambda *_: self
        self._stopping.set(self.Event.STOP_HARDLY)
        if self.__hardware_statistics_pusher is not None:
            self.__hardware_statistics_pusher.kill()
        super(Daemon, self).stop()
        super(Daemon, self).join()
        sandbox.agentr.storage.MdsUploader.stop()
        self.__db.close()
        self.__db_deblock.stop()
        self._logger.info("Stopped")

    @common_patterns.singleton_property
    def _push_client_config(self):
        try:
            with open(self._config.agentr.statbox_push_client.config) as f:
                return yaml.load(f.read())
        except Exception as ex:
            raise sandbox.agentr.errors.PushClientUnavailable(str(ex))

    def drop_connection(self, sid):
        self.__tasks.disconnected(sid, self._logger)
        return super(Daemon, self).drop_connection(sid)

    @jserver.RPC.simple()
    def shutdown(self):
        """ Shutdown the daemon. """
        if self._config.common.installation == ctm.Installation.TEST:
            sandbox.agentr.storage.MdsUploader.stop()
        self._stopping.set(self.Event.STOP_HARDLY)
        if self._config.common.installation == ctm.Installation.TEST:
            sandbox.agentr.storage.MdsUploader.stop()

    @jserver.RPC.simple()
    def ping(self, ret=True):
        """
        Just returns `ret` parameter's value.
        param ret:  Value to return
        """
        self._logger.info("Pinged with %r value", ret)
        return ret

    @jserver.RPC.simple()
    def freeze(self):
        """
        Prepare to task execution - stop any service activities in a given timeout.
        Raises an exception if the operation not possible in the timeout specified.
        """
        self._logger.info("Switched to FREEZE mode")
        return True

    @jserver.RPC.simple()
    def ease(self):
        """ Allow any service activities. Raises an exception if some tasks are still executing. """
        if len(self.__tasks):
            self._logger.error(
                "There are %r connections associated with %r tasks",
                self.__tasks.connections, [_.id for _ in self.__tasks]
            )
            raise sandbox.agentr.errors.TaskInProgress
        self._logger.info("Switched to EASE mode")
        return True

    @property
    def arcanum_client(self):
        if self._config.common.installation in ctm.Installation.Group.LOCAL:
            try:
                oauth_token = common_fs.read_settings_value_from_file(
                    self._config.server.auth.oauth.token
                )
                auth = common_auth.OAuth(oauth_token)
            except Exception as ex:
                self._logger.error("Error in reading oauth token for arcanum.", exc_info=ex)
                raise
        else:
            try:
                auth = common_auth.TVMSession(service_ticket=common_tvm.TVM.get_service_ticket(["arcanum"])["arcanum"])
            except common_tvm.TVM.Error as ex:
                self._logger.error("Error in tvm while getting service ticket.", exc_info=ex)
                raise

        return common_rest.Client(common_config.Registry().client.sdk.arc_api_url, auth=auth)

    @jserver.RPC.simple()
    def reset(self, hard=False):
        """
        Resets internal state. All registered but not ready resources will be marked as BROKEN,
        all tasks sessions will be dropped. Internal state will be switched to "ease".
        """
        (self._logger.warning if len(self.__tasks) else self._logger.info)(
            "RESETTING INTERNAL STATE hard={}".format(hard)
        )
        self.__tasks.reset()
        self.__db.query("""DELETE FROM "client_state" """)
        if hard:
            self.__db.query("""DELETE FROM "resource" """)

    @jserver.RPC.full()
    def df(self, job):
        """
        Calculate overall disk usage and that for registered task sessions

        :return tuple of totally disk free, locked and total space for the whole host
            (should be serializable to `sandbox.agentr.types.DiskSpaceInfo`)
        """
        ret = self.__buckets.info(job.log)
        space_reserved = sum(_.reserved_space for _ in self.__tasks)
        du = self._disk_space(self._config.client.tasks.data_dir)
        for session, quota_usage in zip(self.__tasks, sandbox.agentr.storage.Quota.usage(self.__tasks)):
            self.__tasks.update_disk_usage(session, du, quota_usage)

        job.log.debug(
            "%d sessions reserved totally %s space, %s free left",
            len(self.__tasks),
            common_format.size2str(space_reserved),
            common_format.size2str(ret.free - space_reserved)
        )
        return ret.free, ret.locked + space_reserved, ret.total

    @staticmethod
    def _disk_space(path):
        """ We have to replace implementation in case of test installation, because AgentR is a binary within tests """
        if common_config.Registry().common.installation == ctm.Installation.TEST:
            return sandbox.agentr.utils.get_disk_usage(path)[1]

        res = common_system.get_disk_space(path)
        return res.total - res.free

    def __associate_registered_session(self, job, token, lxc):
        with self.__tasks:
            s = self.__tasks.get(token)
            if not s:
                return
            s.logger(job).info("Associating already registered task #%s session with connection %r", s.id, job.sid)
            self.__tasks.associate(s, job.connection.id)
            if not s.lxc and lxc:
                self.__tasks.set_lxc(s, lxc)
            return os.path.dirname(s.logfile)

    @jserver.RPC.full()
    def task_session(self, job, token, iteration, reserved_space=0, lxc=None):
        """
        Associate current connection with the task session provided.
        Connection drop DOES NOT release the internal task session - it should be dropped by "task_finished" or by
        "reset" call.
        Starts logging into the directory specified (this will be replaced with automatic log path detection
        by first registered resource for the task, which should be `TASK_LOGS`.
        Will raise an exception, if the daemon in "ease" mode currently.

        :param job:             RPC job object, provided by the library.
        :param token:           Task's session token
        :param iteration:       Task's execution iteration number
        :param lxc:             LXC container name the task executed in (if any)
        :param reserved_space:  Disk space required for task execution in bytes
        :return:                Path to task's log files directory
        """

        logdir = self.__associate_registered_session(job, token, lxc)
        if logdir:
            return logdir

        with sandbox.agentr.utils.Synchronized(
            job.log, self.__inprogress, token,
            "Session %r is currently registering by another thread. Waiting for it.", obfuscate=True,
        ):
            logdir = self.__associate_registered_session(job, token, lxc)
            if logdir:
                return logdir
            job.log.debug(
                "Registering task session %r with iteration %r. Currently %d active connections for %d task sessions.",
                common_format.obfuscate_token(token), iteration, self.__tasks.connections, len(self.__tasks)
            )
            du = self._disk_space(self._config.client.tasks.data_dir)

            s = self.__tasks.add(job, token, iteration, lxc, reserved_space, du, self.ctx.token)
            self.__tasks.associate(s, job.connection.id)
            logger = s.logger(job)
            job.log.info(
                "Registered new task #%s session with token %r, reserved %s of disk space "
                "executing in %r LXC container. The task has declared %d dependencies: %r. Logging to %r",
                s.id, common_format.obfuscate_token(s.token), common_format.size2str(reserved_space), s.lxc,
                len(s.dependencies), s.dependencies, s.logfile
            )
            logger.info("Task #%s session associated with the connection %r", s.id, job.sid)
        self.__buckets.cache_info(sandbox.agentr.utils.LogTee(job.log, logger))
        return s.logdir.strpath

    @jserver.RPC.full()
    def task_meta(self, job):
        """ Returns current session task's metadata (server's API response) """
        return self.__tasks.meta(self.__tasks.associated(job.connection.id).id)

    @jserver.RPC.full()
    def task_meta_setter(self, job, data):
        """ Updates current session task's metadata (server's API response) """
        return self.__tasks.meta_update(self.__tasks.associated(job.connection.id).id, data)

    def __resource_completer(self, s, f, rid, res, failed_resources, mark_as_ready):
        """ A worker to complete a resource in parallel with others. """
        try:
            if mark_as_ready or res.service:
                f.complete(res)
                return
        except (Exception, common_rest.Client.SessionExpired) as ex:
            (f.tee.warning if res.service else f.tee.error)("Can not mark resource #%s as ready: %s", rid, ex)
            if not (res.service or mark_as_ready is None):
                failed_resources.append("#{} exception: {}".format(rid, str(ex)))
        if mark_as_ready is None:
            return

        # Mark missing resources as broken in the following cases:
        # 1. task finished successfully (mark_as_ready is True)
        # 2. resource belongs to task
        # 3. resource was created by a child task for its parent task
        # If resource was originally created by parent task, it is not marked as broken.
        own_res = s.id == res.meta["task"]["id"] or s.id == int(res.meta["attributes"].get("from_task", -1))
        if mark_as_ready or own_res:
            try:
                f.complete(res, broken=True)
            except (Exception, common_rest.Client.SessionExpired) as ex:
                f.tee.exception("Can not mark resource #%s as broken", rid)
        else:
            f.tee.warning("Skip marking resource #%s as broken, belongs to the parent task.", rid)

    @jserver.RPC.full()
    def task_finished(self, job, drop_session=True, mark_as_ready=True):
        """
        Finish task session, complete all registered resources.

        :param job:             RPC job object, provided by the library.
        :param drop_session:    drop task session.
        :param mark_as_ready:   Mark non-services resources as "READY" if `True`, "BROKEN" if `False` and skip if `None`
        """
        s = self.__tasks.associated(job.connection.id, False)
        if not s:
            if drop_session and mark_as_ready is None:
                return
            raise sandbox.agentr.errors.NoTaskSession

        if drop_session and s.token in ctc.ServiceTokens:
            self.__tasks.remove(s, job.connection.id, job.log)
            return

        logger = s.logger(job)
        resources = self.__tasks.task_resources(s.id, done=True)
        tee = sandbox.agentr.utils.LogTee(job.log, logger)

        quota_usage = sandbox.agentr.storage.Quota.usage([s])[0]
        self.__tasks.update_disk_usage(s, self._disk_space(self._config.client.tasks.data_dir), quota_usage)
        tee.info(
            "Reporting disk usage of the task #%s:"
            " disk_usage=(max=%s, last=%s) work_disk_usage=(max=%s, last=%s, resources=%s)",
            s.id, s.disk_usage.max, s.disk_usage.last,
            s.disk_usage.work_max, s.disk_usage.work_last, s.disk_usage.resources
        )
        try:
            s.rest.task.current.execution.update(dict(
                disk_usage=dict(
                    max=s.disk_usage.max, last=s.disk_usage.last
                ),
                work_disk_usage=dict(
                    max=s.disk_usage.work_max, last=s.disk_usage.work_last, resources=s.disk_usage.resources
                )
            ))
        except (
            common_rest.Client.HTTPError,
            sandbox.agentr.errors.NoTaskSession,
            sandbox.agentr.errors.UserDismissed
        ):
            tee.exception("Failed to push disk usage for task %r to server", s.id)

        if drop_session:
            tee.debug("Finalizing task #%s session.", s.id)
            if s.quota:
                sandbox.agentr.storage.Quota.destroy(s.quota.path)
            with self.__tasks:
                logs2push = self.__tasks.remove(s, job.connection.id, job.log)
            if logs2push:
                tee.info("Completing registered logs for sending to LogBroker")
                try:
                    log_dirs = {item["log_type"]: item["int_dir"] for item in self._push_client_config["files"]}
                    self.Fetcher.complete_registered_logs(s, logs2push, log_dirs, tee)
                except Exception:
                    tee.exception("Error occurred while completing logs")
            if self._config.common.installation not in ctm.Installation.Group.LOCAL:
                oldest_update_time = time.time()
                for task_id, meta in self.__tasks.all_meta().iteritems():
                    if task_id == s.id:
                        continue
                    update_time = meta.get("time", {}).get("updated")
                    if not update_time:
                        continue
                    oldest_update_time = min(
                        oldest_update_time,
                        calendar.timegm(aniso8601.parse_datetime(update_time).timetuple())
                    )
                tee.info("Cleanup coredumps older than %s", dt.datetime.fromtimestamp(oldest_update_time))
                coredumps_dir = self._config.client.tasks.coredumps_dir
                for coredump_file in os.listdir(coredumps_dir):
                    coredump_path = os.path.join(coredumps_dir, coredump_file)
                    try:
                        st = os.lstat(coredump_path)
                    except OSError as ex:
                        tee.info("Coredump %s removing failed.", coredump_path, exc_info=ex)
                        if not os.path.exists(coredump_path):
                            tee.info("Skip %s.", coredump_path)
                            continue
                        else:
                            raise
                    if st.st_mtime <= oldest_update_time:
                        tee.info("Removing %s", coredump_path)
                        with open(os.devnull, "w") as devnull:
                            p = sp.Popen(
                                ["/bin/rm", "-rf", coredump_path],
                                preexec_fn=common_os.User.Privileges().__enter__,
                                close_fds=True, stdout=devnull, stderr=devnull
                            )
                            if p.wait():
                                tee.error("Subprocess finished with exit code %d", p.returncode)

            tee.info("Task #%s session finalized.", s.id)

        if resources:
            if mark_as_ready is not None:
                msg_suffix = " as {}".format(ctr.State.READY if mark_as_ready else ctr.State.BROKEN)
            else:
                msg_suffix = ""
            tee.debug("Completing task #%s resources %r%s.", s.id, sorted(resources), msg_suffix)

            f = self.Fetcher(tee, logger, s.id, s.rest, self._config, self.users, self.__buckets)

            local_resources = {rid for (_, rid, _) in self.__buckets.local_resources(resources.keys())}
            failed_resources = []
            workers = []
            while resources:
                rid, res = resources.popitem()
                if res.meta["state"] == ctr.State.READY or rid in local_resources:
                    continue  # leave READY resources alone regardless of planned action
                workers.append(gevent.spawn(self.__resource_completer, s, f, rid, res, failed_resources, mark_as_ready))

            tee.debug("Waiting %d workers to complete.", len(workers))
            gevent.joinall(workers)
            if failed_resources:
                raise sandbox.agentr.errors.InvalidResource("; ".join(failed_resources))

        if not self.__tasks.connections:
            self.__buckets.reset()

        if s.hardware_metrics:
            s.hardware_metrics.pid = None

        if s.lxc != "privileged":
            tee.info("Unmounting all registered mount points for task #%s", s.id)
            task_mounts = self.__tasks.mounts(s.id)
            task_mounts.extend(self.__other_mount_points(task_mounts, s, tee))

            for mount in task_mounts:
                self.Fetcher.umount(tee, mount, s.lxc, self.__container_rootfs(s), s.id)

    def __other_mount_points(self, task_mounts, task_session, tee):
        if not common_platform.on_osx():
            return

        task_dir = str(self.Fetcher.task_dir(task_session.id))
        for fs in sp.check_output(["df"]).splitlines()[1:]:
            mounted_on = fs.split(" ")[-1]
            if mounted_on.startswith(task_dir) and not any([known == mounted_on for known in task_mounts]):
                tee.debug("Going to unmount not registered mount point %s for task #%s", mounted_on, task_session.id)
                yield sandbox.agentr.session.Mount(target=mounted_on)

    @jserver.RPC.full()
    def resource_sync(self, job, resource_id, fastbone=True, restore=False):
        """
        Synchronize a resource by given ID.

        :param job:         RPC job object, provided by the library.
        :param resource_id: Resource ID to be synchronized.
        :param fastbone:    Use fastbone network instead of backbone.
        :param restore:     Try to restore `BROKEN` resource if possible.
        :returns:           Path to the resource's data location.
        """
        ses = self.__tasks.associated(job.connection.id)
        rest = self._rest if ses.token in ctc.ServiceTokens else ses.rest
        logger = ses.logger(job)
        tee = sandbox.agentr.utils.LogTee(job.log, logger)

        with sandbox.agentr.utils.Synchronized(
            tee, self.__inprogress, resource_id,
            "Resource #%d is currently synchronizing by another thread. Waiting for it."
        ):
            self.__buckets.info(logger)  # KORUM: Temporary debug
            sync_info = self.Fetcher(
                tee, logger, ses.id, rest, self._config, self.users, self.__buckets, fastbone=fastbone
            )(resource_id, restore=restore)
            if not self.__tasks.account(ses.id, resource_id, sync_info.downloaded):
                ses.disk_usage.resources += sync_info.size
            return sync_info.path

    @jserver.RPC.full()
    def resource_meta(self, job, resource_id):
        """
        Returns resource's metadata by given ID from the local cache or `None` in case of no resource in the cache.

        :param job:         RPC job object, provided by the library.
        :param resource_id: Resource ID to be synchronized.
        :returns:           Resource metadata or `None`.
        """
        ses = self.__tasks.associated(job.connection.id)
        ses.logger(job).debug("Checking local cache for resource's #%s metadata", resource_id)
        _, res = self.__buckets.check(resource_id)
        if not res:
            # No READY resource found in the local cache. But may be it was registered by the task?
            res = self.__tasks.check_resource(ses.id, resource_id)
        return res and res.meta

    def _normalize_path(self, ses, path, depth):
        try:
            path.decode(encoding="ascii", errors="strict")
        except UnicodeError:
            raise sandbox.agentr.errors.InvalidData(
                u"Non-ASCII characters are not allowed in resource path: {}"
                .format(common_encoding.force_unicode_safe(path))
            )

        taskdir = self.Fetcher.task_dir(ses.id)
        path = self.Fetcher.normalize_resource_path(taskdir, path)
        node = path
        if depth is None:
            node = taskdir
        else:
            for _ in xrange(depth + 1):
                node = node.join("..")
                if not node >= taskdir:
                    raise ValueError("Depth {!r} unacceptable for path {!r}".format(depth, path))
        return node, path.relto(node) or node.basename

    @jserver.RPC.full()
    def resource_register(
        self, job, path, type_, name, arch, attrs,
        share=True, service=False, for_parent=False, depth=None,
        full_data=None, resource_meta=None, system_attributes=None
    ):
        """
        Register a `NOT_READY` resource at the path specified.

        :param job:         RPC job object, provided by the library.
        :param path:        Resource's data location.
        :param type_:       Resource type.
        :param name:        Resource name.
        :param arch:        Resource architecture (platform actually).
        :param attrs:       Resource attributes.
        :param share:       Share resource data.
        :param service:     Service resource.
        :param for_parent:  Create resource for parent of current task
        :param depth:       Which depth of resource path should be registered as resource.
                            For example, in case of "a/b/c" passed as `path` following path will be shared
                            in accordance with `depth` parameter:
                            - `None`: "a/b/c"
                            - `0`: "c"
                            - "1": "b/c"
                            - "2": "a/b/c"
                            - "3": will cause `ValueError` exception
        :param resource_meta: Resource class meta information
        :param system_attributes: Resource system attributes
        :returns:           Registered resource data. The resource object can be re-used
                            in case of there's already registered `NOT_READY` with the same metadata
        """
        ses = self.__tasks.associated(job.connection.id)
        logger = ses.logger(job)
        tee = sandbox.agentr.utils.LogTee(job.log, logger)

        basepath, relpath = self._normalize_path(ses, path, depth)
        tee.info(
            "Creating resource with type: %s, arch: %s, attrs: %r, at path %r (depth %r, base path: %r, meta path: %r)."
            "Resource meta: %s",
            type_, arch, attrs, path, depth, str(basepath), relpath, resource_meta
        )

        try:
            res = ses.rest.resource({
                "type": type_,
                "arch": arch or ctm.OSFamily.ANY,
                "description": name,
                "attributes": attrs,
                "file_name": relpath,
                "for_parent": for_parent,
                "resource_meta": resource_meta,
                "system_attributes": system_attributes,
            })
        except common_rest.Client.HTTPError as ex:
            if ex.status == httplib.BAD_REQUEST:
                raise sandbox.agentr.errors.InvalidResource(str(ex))
            raise
        self.__tasks.new_resource(ses, res, basepath, share, service)
        return res

    @jserver.RPC.full()
    def resource_register_meta(
        self, job, path, meta,
        share=True, service=False, for_parent=False, depth=None, add_source=False
    ):
        """
        Register a resource at the path specified.

        :param job:         RPC job object, provided by the library.
        :param path:        Resource's data location.
        :param meta:        Resource metadata, as returned from server.
        :param share:       Share resource data.
        :param service:     Service resource.
        :param for_parent:  Create resource for parent of current task
        :param depth:       Which depth of resource path should be registered as resource.
        :param add_source:  Assign source.
        """
        ses = self.__tasks.associated(job.connection.id)
        logger = ses.logger(job)
        tee = sandbox.agentr.utils.LogTee(job.log, logger)

        basepath, relpath = self._normalize_path(ses, path, depth)
        tee.info(
            "Registering resource %s with type %s at path %r (depth %r, base path: %r, meta path: %r)",
            meta["id"], meta["type"], path, depth, str(basepath), relpath
        )
        meta["file_name"] = relpath
        self.__tasks.new_resource(ses, meta, basepath, share, service)

        if add_source:
            ses.rest.resource[meta["id"]].source()

    @jserver.RPC.full()
    def resource_update(self, job, resource_id, path=None, name=None, arch=None, attrs=None, depth=None):
        """
        Update `NOT_READY` resource.

        :param job:         RPC job object, provided by the library.
        :param resource_id: Resource ID to update.
        :param path:        Resource's data location.
        :param name:        Resource name.
        :param arch:        Resource architecture (platform actually).
        :param attrs:       Resource attributes.
        :param depth:       Which depth of resource path should be registered as resource.
        :returns:           Registered resource data. The resource object can be re-used
                            in case of there's already registered `NOT_READY` with the same metadata
        """
        ses = self.__tasks.associated(job.connection.id)
        logger = ses.logger(job)
        tee = sandbox.agentr.utils.LogTee(job.log, logger)
        res = self.__tasks.check_resource(ses.id, resource_id)
        update = {}
        basepath = None
        if path is not None:
            basepath, relpath = self._normalize_path(ses, path, depth)
            update["file_name"] = relpath
        if name is not None:
            update["description"] = name
        if arch is not None:
            update["arch"] = arch
        if attrs is not None:
            update["attributes"] = attrs

        if update:
            tee.info("Updating resource %s with meta %r", resource_id, update)
            ses.rest.resource[res.meta["id"]] = update
            res.meta.update(update)
            self.__tasks.new_resource(ses, res.meta, basepath or res.data, res.share, res.service)
        return res.meta

    @jserver.RPC.full()
    def resource_complete(self, job, resource_id, broken=False):
        """
        Marks the resource with given ID as READY or BROKEN on this host

        :param job:         RPC job object, provided by the library.
        :param resource_id: Resource ID to be marked as READY or BROKEN
        :param broken:      Mark resource as BROKEN
        """
        ses = self.__tasks.associated(job.connection.id)
        resource = self.__tasks.check_resource(ses.id, resource_id, drop=True)
        if not resource:
            raise sandbox.agentr.errors.InvalidResource("Resource #{} wasn't registered.".format(resource_id))
        if not resource.data.check() and not broken:
            raise sandbox.agentr.errors.InvalidData(
                "cannot mark resource #{} as completed: path {} not found".format(resource_id, resource.data.strpath)
            )

        logger = ses.logger(job)
        tee = sandbox.agentr.utils.LogTee(job.log, logger)
        self.Fetcher(
            tee, logger, ses.id, ses.rest, self._config, self.users, self.__buckets
        ).complete(resource, broken=broken)
        return resource.meta

    @jserver.RPC.full()
    def resource_delete(self, job, resource_id):
        """
        Drops resource from task resources if it is already registered, then sends a request to delete it

        :param job:         RPC job object, provided by the library.
        :param resource_id: Resource ID to be deleted

        :returns: sandbox.web.api.v1.schemas.BatchResult (as dict) for deleted resource
        """
        ses = self.__tasks.associated(job.connection.id)
        self.__tasks.check_resource(ses.id, resource_id, drop=True)
        res = ses.rest.batch.resources["delete"].update([resource_id])
        return res[0]

    @jserver.RPC.full()
    def task_logs_meta(self, job):
        """
        Returns metadata of task logs resource.

        :param job:         RPC job object, provided by the library.
        :returns: dict
        """
        ses = self.__tasks.associated(job.connection.id)
        return self.__tasks.find_log_resource(ses)

    @jserver.RPC.full()
    def patch_info(self, job, pullrequest_id, iteration=None, published=True, pr_info=None):
        ses = self.__tasks.associated(job.connection.id)
        logger = sandbox.agentr.utils.LogTee(job.log, ses.logger(job))
        if pr_info is None:
            pr_info = self.arcanum_client.pullrequest[pullrequest_id].read()
        patch_info = filter(
            lambda p: (
                (published is None or p["published"] == published) and
                (iteration is None or p["revision"] == int(iteration))
            ),
            pr_info["patches"]
        )
        if not patch_info:
            logger.info("No patch information for iteration %s.", iteration)
            return None
        patch_info = max(patch_info, key=lambda p: p["revision"])
        return patch_info

    @jserver.RPC.full()
    def reshare(self, job, resource_id, drop_failed=True):
        """
        Reshare given resource by its ID

        :param job:         RPC job object, provided by the library.
        :param resource_id: Resource ID to be marked reshared
        :param drop_failed: Drop the resource's data and metadata on resharing error
        """
        ses = self.__tasks.service(job.connection.id)
        fetcher = self.Fetcher(job.log, self._logger, ses.id, ses.rest, self._config, self.users, self.__buckets)

        try:
            return fetcher.reshare(resource_id)
        except sandbox.agentr.errors.ARException:
            if not drop_failed:
                raise
            cache = self.__buckets.local_resources([resource_id])
            if cache:
                self.__buckets.drop_records([resource_id])
                job.log.info("Removing disk data.")
                fetcher.dropdirs(cache)
            job.log.info("Dropping remote server record about broken resource.")
            self._rest.client[self._config.this.id].service.resources.drop([resource_id])
            raise

    @jserver.RPC.full()
    def monitoring_setup(self, job):
        """
        Set up monitoring for the task.

        :param job:             RPC job object, provided by the library.
        """
        ses = self.__tasks.associated(job.connection.id)
        mon = self.__tasks.monitoring(ses)
        if mon:
            return

        if ses.dump_disk_usage:
            self.Fetcher.allocate_disk_usage(
                ses.logger(job), ses.logdir, self.users.unprivileged
            )

        self.__tasks.monitoring_init(ses)

        syslog_path = common_os.system_log_path()
        if syslog_path and os.path.exists(syslog_path):
            fstat = os.stat(syslog_path)
            self.__tasks.monitoring_add_log(ses, syslog_path, "system.log", fstat.st_ino, fstat.st_size)

    @jserver.RPC.full()
    def monitoring_add_log(self, job, path, label):
        """
        Register atop process for the task.

        :param job:             RPC job object, provided by the library.
        :param pid:             atop process PID
        """
        ses = self.__tasks.associated(job.connection.id)
        mon = self.__tasks.monitoring(ses)
        if not mon:
            return

        if os.path.exists(path):
            fstat = os.stat(path)
            self.__tasks.monitoring_add_log(ses, path, label, fstat.st_ino, fstat.st_size)

    @staticmethod
    def _real_pid(pid, caller_pid):
        if not caller_pid or common_config.Registry().common.installation == ctm.Installation.TEST:
            return pid
        caller_ns_fname = "/proc/{}/ns/pid".format(caller_pid)
        if not os.path.lexists(caller_ns_fname):
            return pid
        cap = common_os.Capabilities(common_os.Capabilities.Cap.Bits.CAP_SYS_PTRACE)
        with cap:
            caller_ns = os.readlink(caller_ns_fname)
        for pid_fname in glob.glob("/proc/*/ns/pid"):
            try:
                with cap:
                    if os.readlink(pid_fname) != caller_ns:
                        continue
                fname = os.path.join(os.path.dirname(os.path.dirname(pid_fname)), "status")
                with open(fname) as f:
                    nspid = next(iter(filter(lambda _: _.startswith("NSpid:"), f.readlines())), None)
                    if nspid:
                        nspid = nspid.strip().split()
                        if int(nspid[-1]) == pid:
                            return int(nspid[:2][-1])
            except (IOError, OSError) as ex:
                if ex.errno not in (errno.ENOENT, errno.ESRCH, errno.EACCES):
                    raise
        return pid

    @jserver.RPC.full()
    def register_atop(self, job, pid):
        """
        Register atop process for the task.

        :param job:             RPC job object, provided by the library.
        :param pid:             atop process PID
        """
        ses = self.__tasks.associated(job.connection.id)
        mon = self.__tasks.monitoring(ses)
        if not mon or mon.atop_pid:
            return

        self.__tasks.register_atop(ses, self._real_pid(pid, job.connection.peerid.pid))

    @jserver.RPC.full()
    def register_tcpdump(self, job, pid):
        """
        Register tcpdump process for the task.

        :param job:             RPC job object, provided by the library.
        :param pid:             tcpdump process PID
        """
        ses = self.__tasks.associated(job.connection.id)
        mon = self.__tasks.monitoring(ses)
        if not mon or mon.tcpdump_pid:
            return

        self.__tasks.register_tcpdump(ses, self._real_pid(pid, job.connection.peerid.pid))

    @jserver.RPC.full()
    def monitoring_finish(self, job):
        """
        Finish monitoring for the task.

        :param job:             RPC job object, provided by the library.
        """
        ses = self.__tasks.associated(job.connection.id, False)
        if not ses:
            self._logger.warning("No task session: %s", job.connection.id)
            return
        mon = self.__tasks.monitoring(ses, with_logs=True)
        if not mon:
            return
        logger = sandbox.agentr.utils.LogTee(job.log, ses.logger(job))
        if ses.peak_disk_usage_thread:
            logger.debug("Waiting for peak disk usage thread")
            ses.peak_disk_usage_thread.join()
            ses.peak_disk_usage_thread = None
        self.Fetcher.dump_monitoring_logs(
            logger,
            ses.taskdir,
            ses.logdir,
            mon.logs,
            self.users.unprivileged,
            dump_disk_usage=ses.dump_disk_usage,
            peak_dumped=mon.peak_dumped
        )
        if mon.atop_pid:
            self.Fetcher.stop_process(logger, "atop", mon.atop_pid)
        if mon.tcpdump_pid:
            self.Fetcher.stop_process(logger, "tcpdump", mon.tcpdump_pid)

        self.__tasks.monitoring_done(ses)

    @jserver.RPC.full()
    def monitoring_status(self, job):
        """
        Is task monitoring currently running?

        :param job:             RPC job object, provided by the library.
        """
        ses = self.__tasks.associated(job.connection.id)
        mon = self.__tasks.monitoring(ses)
        return bool(mon)

    @jserver.RPC.full
    def hard_link(self, job, source, target=""):
        """
        Make a hard link from /place filesystem to a task directory

        :param source: source path
        :param target: target path relative to task directory
        """
        ses = self.__tasks.associated(job.connection.id, False)
        logger = sandbox.agentr.utils.LogTee(job.log, ses.logger(job))

        source = py.path.local(source).realpath()
        target = ses.taskdir.join(target)
        if target.exists() and target.isdir():
            target = target.join(source.basename)

        logger.debug("Making hard link '%s' -> '%s'", source, target)
        try:
            sandbox.agentr.utils.ProjectQuota(
                source.strpath,
                create=False,
                popen=ft.partial(sp.Popen, preexec_fn=common_os.User.Privileges().__enter__, close_fds=True)
            ).project = sandbox.agentr.utils.ProjectQuota(
                ses.taskdir.strpath,
                create=False,
                popen=ft.partial(sp.Popen, preexec_fn=common_os.User.Privileges().__enter__, close_fds=True)
            ).project
            with common_os.User.Privileges():
                target.mklinkto(source)
            return target.strpath
        except py.error.Error:
            logger.exception("Unable to make hard link")

    @staticmethod
    def __check_limit(items, limit, msg):
        if len(items) > limit:
            raise sandbox.agentr.errors.MaintenanceLimitExceeded(msg.format(len(items), limit))

    @common_patterns.singleton_property
    def _rest(self):
        """ Creates an instance of REST client with administrative privileges. FOR SERVICE PURPOSES ONLY! """
        token = self._config.client.auth.oauth_token
        token = common_fs.read_settings_value_from_file(token) if token else None
        rest = common_rest.Client(auth=token)
        # The request can took a lot of time
        rest.DEFAULT_TIMEOUT *= 10
        rest.MAX_TIMEOUT *= 10
        rest.reset()
        return rest

    @jserver.RPC.full()
    def cleanup(self, job, chunk=None):
        """
        Asks server about resources, which can be dropped from the host.

        :param job:     RPC job object, provided by the library.
        :param chunk:   Chunk size to be used.
        :return:        Kind of resources which was dropped or `None`.
        """

        return self.__cleaner.cleanup(chunk, job)

    @jserver.RPC.full()
    def maintain(self, job, dry=True, **_limits):
        """
        Compares local database with the central one and also with the actual filesystem.
        Removes data files, which should not exists on the host. Removes database records for files, which should
        exists, but actually no.

        :param job:     RPC job object, provided by the library.
        :param dry:     Log difference only, without deleting any files and/or database records.
        :param _limits: Files removal limits (see `sandbox.agentr.types.MaintainLimits`)
        """
        rest = self._rest
        ses = self.__tasks.service(job.connection.id)
        logger = sandbox.agentr.utils.LogTee(job.log, ses.logger(job)) if ses and ses.id else job.log
        remote = set(rest.client[self._config.this.id].service.resources[:])

        logger.info("Server reported %d resources for this node", len(remote))
        local = set(_[1] for _ in self.__buckets.local_resources())
        logger.info("Local cache has %d resource records", len(local))
        actual = self.__buckets.actual_resources()
        logger.info("Actually there are %d resources on the disk", len(actual))

        extra_local = sorted(local - actual.viewkeys())
        logger.debug("There are %d database cache records without actual data: %r", len(extra_local), extra_local)
        extra_actual = sorted(actual.viewkeys() - local)
        logger.debug("There are %d data entries without cache records: %r", len(extra_actual), extra_actual)
        actual_local = actual.viewkeys() & local
        extra_actual_local = sorted(actual_local - remote)
        logger.debug(
            "There are %d local data entries which should not exist on this host: %r",
            len(extra_actual_local), extra_actual_local
        )
        extra_remote = sorted(remote - actual_local)
        logger.debug(
            "There are %d remote server records which do not exist on this host: %r",
            len(extra_remote), extra_remote
        )

        limits = sandbox.agentr.types.MaintainLimits(**self._config.agentr.daemon.maintain.limits)
        overrides = sandbox.agentr.types.MaintainLimits(**_limits)
        [limits.__setattr__(k, v) for k, v in overrides if v is not None]
        self.__check_limit(
            extra_local, limits.extra_local,
            "Extra local records {} exceeded limit of {}"
        )
        self.__check_limit(
            extra_actual_local, limits.extra_actual_local,
            "Extra local actual records {} exceeded limit of {}"
        )
        self.__check_limit(
            extra_remote, limits.extra_remote,
            "Extra server records {} exceeded limit of {}"
        )
        res = len(extra_local), len(extra_actual_local), len(extra_remote)
        if dry:
            return res

        logger.info("Dropping extra cache records (without data).")
        self.__buckets.drop_records(extra_local)
        logger.info("Dropping extra cache records (should not exist).")
        self.__buckets.drop_records(extra_actual_local)
        logger.info("Dropping extra cache records (without remote server record).")
        self.__buckets.drop_records(extra_remote)

        fetcher = self.Fetcher(job.log, logger, ses.id, ses.rest, self._config, self.users, self.__buckets)
        logger.info("Removing disk data without cache records.")
        fetcher.dropdirs([(actual[_], _, None) for _ in extra_actual])
        logger.info("Removing disk data without remote server records.")
        fetcher.dropdirs([(actual[_], _, None) for _ in extra_actual_local])
        logger.info("Dropping remote server records about nonexistent resources.")
        for chunk in common_itertools.chunker(list(extra_remote), self.MAX_RESOURCES_IN_A_REQUEST):
            logger.debug("Dropping remote server records chunk of %d length.", len(chunk))
            rest.client[self._config.this.id].service.resources.drop(chunk)

        self.__db.vacuum()
        return res

    @jserver.RPC.simple()
    def backup(self):
        return self.__db.backup()

    @jserver.RPC.simple()
    def restore_links(self):
        self.Fetcher.restore_links(self._logger, self.__buckets, self._config.client.dirs.data)

    @jserver.RPC.full
    def bucket_cache_info(self, job, bid):
        logger = job.log
        rss = self.__buckets[bid].resources()
        amount, size = len(rss), sum(_[1] for _ in rss)
        logger.info(
            "Local cache has information about %d resources totally for %s in bucket #%d",
            amount, common_format.size2str(size), bid
        )
        return amount, size

    @jserver.RPC.full
    def erase_bucket(self, job, bid):
        """
        Erases a given bucket meta-information in the local cache and also reports there are not
        resources, which are located in this bucket to the remote server so it looks as empty both
        for local and remote resource cache collections.

        DANGEROUS!!! Drops local and remote cache database records about this bucket!!!
        """
        logger = job.log
        rest = self._rest
        logger.warning("Erasing a bucket #%r", bid)
        ids = self.Fetcher.erase_bucket(logger, self.__buckets[bid], self._config.client.dirs.data)
        logger.info("Dropping remote server records about %d bucket resources: %r", len(ids), ids)
        for chunk in common_itertools.chunker(ids, self.MAX_RESOURCES_IN_A_REQUEST):
            logger.debug("Dropping remote server records chunk of %d length.", len(chunk))
            rest.client[self._config.this.id].service.resources.drop(chunk)

    @jserver.RPC.full()
    def register_log(self, job, file_name, log_type=sandbox.agentr.types.LogType.TASK):
        """
        Register log file for to LogBroker by push client

        TODO: remove after DMP-175

        :param job: RPC job object, provided by the library.
        :param file_name: log file
        :param log_type: log type defined in push client's config
        """
        ses = self.__tasks.associated(job.connection.id)
        assert self._push_client_config
        self.__tasks.new_log2push(ses, file_name, log_type)

    @jserver.RPC.full()
    def register_process(self, job, pid):
        """
        Register process for coredump tracking

        :param job: RPC job object, provided by the library.
        :param pid: process id
        """
        ses = self.__tasks.associated(job.connection.id)
        logger = sandbox.agentr.utils.LogTee(job.log, ses.logger(job))
        rpid = pid
        if sys.platform.startswith("linux"):
            with self.Fetcher.UserPrivileges():
                ns = common_os.Namespace.from_pid(job.connection.peerid.pid, common_os.Namespace.Type.PID)
                if ns:
                    rpid = common_os.real_pid(pid, ns) or pid
        logger.info("Registering process with pid %s (%s) for task #%s", pid, rpid, ses.id)
        try:
            self.__tasks.new_task_process(ses, rpid)
        except apsw.ConstraintError:
            logger.warning("Replacing previously registered process with rpid %s for task #%s", rpid, ses.id)
            self.__tasks.delete_task_process(ses, rpid)
            self.__tasks.new_task_process(ses, rpid)
        return rpid

    @jserver.RPC.full()
    def register_coredump(self, job, pid, filename, rpid, core_limit=None, base_filename=None):
        """
        Register coredump for the process

        :param job: RPC job object, provided by the library.
        :param pid: process id
        :param filename: name of the coredump file
        :param rpid: real process id in host namespace
        :return: coredump full path
        """
        ses = self.__tasks.session_by_pid(rpid)
        if base_filename is None:
            base_filename = filename
        # protect against too big values
        if core_limit is not None:
            core_limit = min(int(core_limit), sys.maxint)
        coredump_statistic = {
            "type": cts.SignalType.COREDUMPS,
            "timestamp": dt.datetime.utcnow(),
            "pid": pid,
            "rpid": rpid,
            "core_limit": core_limit,
            "client_id": self._config.this.id,
            "client_tags": self._config.client.tags,
            "path": base_filename
        }
        if not ses:
            coredump_path = os.path.join(self._config.client.tasks.coredumps_dir, filename)
            job.log.info("Registering coredump '%s' for process %s (%s)", coredump_path, pid, rpid)
            common_statistics.Signaler().push(coredump_statistic)
            return coredump_path
        logger = sandbox.agentr.utils.LogTee(job.log, ses.logger(job))
        coredump_path = ses.taskdir.join(filename)
        coredump_statistic["task_id"] = ses.id
        common_statistics.Signaler().push(coredump_statistic)
        if core_limit == 0:
            return coredump_path.strpath
        logger.info("Registering coredump '%s' for process %s (%s) of task #%s", coredump_path, pid, rpid, ses.id)
        self.__tasks.add_coredump(ses.id, rpid, coredump_path)
        return coredump_path.strpath

    @jserver.RPC.full()
    def coredumps(self, job):
        """
        Coredumps for the task's processes, waits files to complete for self.COREDUMP_TIMEOUT seconds

        :param job: RPC job object, provided by the library.
        :return: dict with pid as key and coredump path as value
        """
        ses = self.__tasks.associated(job.connection.id)
        logger = sandbox.agentr.utils.LogTee(job.log, ses.logger(job))
        coredumps = self.__tasks.task_coredumps(ses)
        logger.info("Waiting for coredumps %s to complete", coredumps.values())
        for pid, coredump_path in coredumps.items():
            if coredump_path and common_itertools.progressive_waiter(
                0, 1, self.COREDUMP_TIMEOUT, lambda: os.path.exists(coredump_path)
            )[0]:
                logger.info(
                    "Coredump file %s for process with real pid %d is completed (%s)",
                    coredump_path, pid, common_format.size2str(os.path.getsize(coredump_path))
                )
            else:
                if coredump_path:
                    logger.error("Coredump file %s for process with real pid %d is not completed", coredump_path, pid)
                coredumps.pop(pid)
        return coredumps

    @jserver.RPC.simple()
    def test_deblock(self, amount):
        """
        This method is for testing purpose only - it checks that
        `sandbox.agentr.deblock.Deblock` class __really__ deblocks
        operations with the database.
        """
        stop_checker = gevent.event.Event()

        class Sleeper(object):
            def __init__(self, logger):
                self.logger = logger

            def sleep(self, amount):
                self.logger.info("Sleeping for %d seconds", amount)
                gevent.monkey.get_original("time", "sleep")(amount)

        def checker():
            import time
            while not stop_checker.is_set():
                now = time.time()
                stop_checker.wait(0.1)
                self._logger.debug("Deblock checker loop")
                if time.time() - now > 0.2:
                    self._logger.warning("Too big sleep gap detected!")
            self._logger.debug("Deblock checker loop finished!")

        checker_loop = gevent.Greenlet(checker)
        checker_loop.start()
        self._logger.debug("Deblock checker - yield control")
        gevent.sleep()
        deblock = sandbox.agentr.deblock.Deblock(logger=self.log.getChild("deblock"))
        sleeper = deblock.make_proxy(Sleeper(self._logger))
        self._logger.debug("Deblock checker - yield control2")
        gevent.sleep()
        sleeper.sleep(amount)
        stop_checker.set()
        checker_loop.join()
        deblock.stop()

    @jserver.RPC.full
    def arcadia_hg_clone(self, job, hg_env, rev="default"):
        """
        Ensure a clone of Arcadia Hg repository is present on the host

        :param job: RPC job object, provided by the library.
        :param hg_env: OS environment where Mercurial "hg" command is present
        :param rev: branch/tag/revision name (hg update REV parameter)
        """

        ses = self.__tasks.associated(job.connection.id)
        logger = ses.logger(job)
        tee = sandbox.agentr.utils.LogTee(job.log, logger)

        cache = vcs_cache.CacheableHg()
        cache_check_path = py.path.local(cache.base_cache_dir).join("arcadia.hg.check")
        cache_path = py.path.local(cache.base_cache_dir).join("arcadia.hg")

        setup_reason = None
        if not cache_check_path.exists():
            setup_reason = "Hg cache not found, trying to build it."
        elif cache_path.join(".hg/store/journal").exists():
            setup_reason = "Found unfinished transaction, rebuilding hg cache."

        if setup_reason:
            tee.info(setup_reason)
            resources = ses.rest.resource.read(type=cache.RESOURCE_TYPE, state=ctr.State.READY, limit=1)["items"]
            if not resources:
                raise sandbox.agentr.errors.ResourceNotAvailable(
                    "Cannot find any suitable {} resources".format(cache.RESOURCE_TYPE)
                )
            fetcher = self.Fetcher(tee, logger, ses.id, ses.rest, self._config, self.users, self.__buckets)
            sync_info = fetcher(resources[0]["id"])
            fetcher.setup_hg_cache(
                tee, cache_path, cache_check_path, py.path.local(sync_info.path), self.users,
                check_space=int(resources[0]["attributes"].get("full_size", 0))
            )

        self.Fetcher.update_hg_cache(tee, hg_env, cache_path, rev=rev)

        cache_dir = cache_path.strpath
        tee.info("Hg cache at %s", cache_dir)
        return cache_dir, ses.id

    @jserver.RPC.full
    def git_repos_gc(self, job):
        ses = self.__tasks.associated(job.connection.id)
        logger = ses.logger(job)
        tee = sandbox.agentr.utils.LogTee(job.log, logger)

        cache_dirs = vcs_cache.CacheableGit.get_cache_dirs()
        tee.info("Got request for git gc in %s", cache_dirs)
        self.Fetcher.git_repos_gc(tee, cache_dirs)

    @jserver.RPC.full
    def job_state_getter(self, job):
        ses = self.__tasks.associated(job.connection.id)
        return self.__tasks.job_state_getter(ses)

    @jserver.RPC.full
    def job_state_setter(self, job, state):
        ses = self.__tasks.associated(job.connection.id)
        logger = ses.logger(job)
        tee = sandbox.agentr.utils.LogTee(job.log, logger)

        tee.info("Switching state to %s for session %s", state, common_format.obfuscate_token(ses.token))
        self.__tasks.job_state_setter(ses, state)

    @jserver.RPC.simple()
    def get_jobs(self):
        return self.__tasks.jobs()

    @jserver.RPC.simple()
    def pragma_setter(self, pragma, value):
        self.__db.query(
            """REPLACE INTO "client_state" ("key", "value") VALUES (?, ?)""",
            (pragma, value), log=sandbox.agentr.db.Log.STATEMENT
        )

    @jserver.RPC.simple()
    def pragma_getter(self, pragma):
        result = self.__db.query_one(
            """SELECT "value" FROM "client_state" WHERE "key" = ?""", (pragma,)
        )
        return result[0] if result else None

    @jserver.RPC.full
    def fileserver_meta_getter(self, job, tid=None):
        if tid is None:  # called from Session
            try:
                tid = self.__tasks.associated(job.connection.id).id
            except sandbox.agentr.errors.NoTaskSession:
                return None
        return self.__tasks.fileserver_meta_getter(tid)

    @jserver.RPC.full
    def fileserver_meta_setter(self, job, fs_meta):
        tid = self.__tasks.associated(job.connection.id).id
        return self.__tasks.fileserver_meta_setter(tid, fs_meta)

    @jserver.RPC.full
    def executor_pid_getter(self, job):
        tid = self.__tasks.associated(job.connection.id).id
        return self.__tasks.executor_pid_getter(tid)

    @jserver.RPC.full
    def register_executor(self, job, pid=None):
        session = self.__tasks.associated(job.connection.id)
        tid = session.id
        pid = pid or job.connection.peerid.pid
        self.__tasks.executor_pid_setter(tid, pid)
        # noinspection PyBroadException
        try:
            self.__tasks.update_hardware_metrics(session)
        except Exception:
            pass

    @jserver.RPC.full
    def progress_meta_getter(self, job, tid=None):
        if tid is None:
            try:
                tid = self.__tasks.associated(job.connection.id).id
            except sandbox.agentr.errors.NoTaskSession:
                return None
        return self.__tasks.progress_meta_getter(tid)

    @jserver.RPC.full
    def progress_meta_setter(self, job, meter_id, progress_meta):
        tid = self.__tasks.associated(job.connection.id).id
        return self.__tasks.progress_meta_setter(meter_id, tid, progress_meta)

    @jserver.RPC.simple
    def progress_meta_deleter(self, meter_id):
        return self.__tasks.progress_meta_deleter(meter_id)

    @jserver.RPC.simple()
    def ban(self, no):
        """ Ban specified bucket """
        bucket = self.__buckets[no]
        ret = (bucket.free, bucket.locked, bucket.total)
        bucket.ban()
        return ret

    @jserver.RPC.simple()
    def unban(self, no):
        """ Unban specified bucket """
        bucket = self.__buckets[no]
        bucket.unban()
        return bucket.free, bucket.locked, bucket.total

    @jserver.RPC.simple()
    def banned(self):
        """ List banned buckets """
        ret = {}
        for no in xrange(len(self.__buckets)):
            bucket = self.__buckets[no]
            if bucket.banned:
                total, free = common_system.get_disk_space(bucket.path)
                ret[no] = (free, bucket.locked, total)
        return ret

    @jserver.RPC.full
    def dropper(self, job, files, workers=None):
        """
        Hack method specially for CLEANUP task on old layout hosts.

        :param job: RPC job object, provided by the framework
        :param files:   Files list to remove
        :param workers: Override amount of workers to be used for files removal
        """
        if common_platform.on_wsl():
            files = [os.path.normpath(wsl.FS.wsl_path(f)) for f in files]
        ses = self.__tasks.associated(job.connection.id)
        cu = ses.rest.user.current[:]
        if cu["role"] != ctu.Role.ADMINISTRATOR:
            raise ValueError("'{}' is not an administrator".format(cu["login"]))
        logger = sandbox.agentr.utils.LogTee(job.log, ses.logger(job)) if ses and ses.id else job.log
        fetcher = self.Fetcher(job.log, logger, ses.id, ses.rest, self._config, self.users, self.__buckets)
        if fetcher.new_layout:
            raise ValueError("The method is not allowed on NEW_LAYOUT")
        base = self._config.client.tasks.data_dir
        if any(not _.startswith(base) for _ in files):
            raise ValueError("Some file(s) ({}) not starts with '{}'".format(files, base))
        fetcher.dropdirs([(None, None, _) for _ in files], workers)

    @jserver.RPC.full()
    def mount_image(self, job, image, dirname=None):
        """
        Mounts given SquashFS image.

        :param image: Path to SquashFS image.
        :param dirname: target subdirectory name, if None, will be uses unique number
        :return: path to mounted image.
        """
        ses = self.__tasks.associated(job.connection.id)
        if not ses.lxc:
            raise ValueError("Image mounting is supported on LXC hosts only")

        max_mounts = self._config.client.tasks.max_image_mounts
        if len(list(_ for _ in self.__tasks.mounts(ses.id) if _.type == ctm.FilesystemType.SQUASHFS)) >= max_mounts:
            raise ValueError("It's forbidden to mount more than {} squashfs images".format(max_mounts))

        logger = sandbox.agentr.utils.LogTee(job.log, ses.logger(job))
        if ses.mount_lock is None:
            ses.mount_lock = gevent.lock.RLock()
        with ses.mount_lock:
            target = (
                dirname
                if dirname and dirname.startswith("/") else
                self.Fetcher.next_task_mount_point(ses.id, name=dirname)
            )
        self.Fetcher.mount_image(logger, image, target, ses.lxc)
        self.__tasks.mount(ses.id, image, target, ctm.FilesystemType.SQUASHFS)
        return str(target)

    def __container_rootfs(self, ses):
        return str(py.path.local(self._config.client.lxc.rootfs.basedir).join(
            str(ses.lxc),
            self._config.client.lxc.rootfs.base
        )) if ses.lxc else ""

    def __task_tmpdirs(self, tid):
        mounted_tmp = [str(mount.target) for mount in self.__tasks.mounts(tid) if mount.type == ctm.FilesystemType.BIND]
        return mounted_tmp + [str(self.Fetcher.task_dir(tid).join("tmp"))]

    @jserver.RPC.full()
    def prepare_tmp(self, job):
        """
        Mount task tmp directory to /tmp/<task_id>
        :param job: AgentR job
        :return: path to tmp dir
        """
        ses = self.__tasks.associated(job.connection.id)
        tid = str(ses.id)
        logger = sandbox.agentr.utils.LogTee(job.log, ses.logger(job))
        bind_src = str(self.Fetcher.task_dir(tid).join("tmp"))
        bind_dst = str(py.path.local("/tmp").join(tid))
        devbox = common_config.Registry().common.installation == ctm.Installation.LOCAL

        if scp.on_wsl() or scp.on_windows():
            return wsl.FS.win_path(bind_src)
        if not scp.on_linux() or common_config.Registry().common.installation == ctm.Installation.TEST:
            return bind_src
        if bind_dst in [mount.target for mount in self.__tasks.mounts(ses.id)]:
            return bind_dst

        if ses.mount_lock is None:
            ses.mount_lock = gevent.lock.RLock()
        with ses.mount_lock:
            self.Fetcher.mount_bind_tmp(logger, bind_src, bind_dst, self.__container_rootfs(ses), ses.lxc, devbox)
            self.__tasks.mount(ses.id, bind_src, bind_dst, ctm.FilesystemType.BIND)
        return bind_dst

    @jserver.RPC.full()
    def mount_overlay(self, job, mount_point, lower_dirs, upper_dir=None, work_dir=None):
        """
        Mounts OverlayFS to mount_point using lower_dirs, upper_dir and work_dir as options of `mount`-command.
        upper_dir and work_dir have to be in task's working directory or in mounted tmpfs on the same device.

        :param mount_point: Directory for merged filesystem.
        :param lower_dirs: List of paths of readonly layers or singular path.
        :param upper_dir: Path to directory where the changes will be saved.
        :param work_dir: Empty directory on the same filesystem as upper_dir. It's required for proper upper_dir work.
        :return: path to mounted filesystem.
        """
        ses = self.__tasks.associated(job.connection.id)
        if not ses.lxc:
            raise ValueError("Mounting overlays is supported on LXC hosts only")

        logger = sandbox.agentr.utils.LogTee(job.log, ses.logger(job))

        if not lower_dirs:
            raise ValueError("'lower_dirs' is not defined")
        if bool(upper_dir) != bool(work_dir):
            raise ValueError("'work_dir' and 'upper_dir' have to be both set or omitted")

        lower_dirs = map(py.path.local, common_itertools.chain(lower_dirs))
        mount_point = py.path.local(mount_point)
        upper_dir = upper_dir and py.path.local(upper_dir)
        work_dir = work_dir and py.path.local(work_dir)

        w_dirs = [
            str(getter(ses.id))
            for getter in
            (self.Fetcher.task_dir, self.Fetcher.task_ramdrive_dir)
        ] + self.__task_tmpdirs(ses.id)
        r_dirs = w_dirs + [self._config.client.dirs.data]

        for paths, allowed_dirs in (
            ((mount_point, upper_dir, work_dir), w_dirs),
            (lower_dirs, r_dirs),
        ):
            for path in paths:
                if path and not any(path.relto(_) for _ in allowed_dirs):
                    raise ValueError("Path '{}' is located outside of allowed dirs: {}".format(str(path), allowed_dirs))

        options = self.Fetcher.overlay_options(lower_dirs, upper_dir, work_dir)
        self.Fetcher.mount_overlay(logger, options, mount_point, ses.lxc)
        self.__tasks.mount(ses.id, options, mount_point, ctm.FilesystemType.OVERLAY)
        return str(mount_point)

    @jserver.RPC.full()
    def umount(self, job, mount_point):
        """
        Unmounts mounted earlier mount_point.

        :param mount_point: Path to point to unmount.
        """
        ses = self.__tasks.associated(job.connection.id)
        logger = sandbox.agentr.utils.LogTee(job.log, ses.logger(job))
        mount_point = str(py.path.local(mount_point).realpath())
        mount = next((mount for mount in self.__tasks.mounts(ses.id) if mount.target == mount_point), None)
        if not mount:
            raise ValueError("Mount point not found: {}".format(mount_point))
        self.Fetcher.umount(logger, mount, ses.lxc, self.__container_rootfs(ses), ses.id)
        self.__tasks.umount(mount)

    @property
    def __turbo_boost(self):
        with open(self.NO_TURBO_PATH) as fh:
            return not int(fh.read())

    @__turbo_boost.setter
    def __turbo_boost(self, enable):
        cap = common_os.Capabilities(
            common_os.Capabilities.Cap.Bits.CAP_DAC_OVERRIDE |
            common_os.Capabilities.Cap.Bits.CAP_SYS_ADMIN
        )
        with cap, open(self.NO_TURBO_PATH, "w") as fh:
            fh.write("{:d}".format(not enable))

    def __set_turbo_boost(self, enable):
        enabled = self.__turbo_boost
        if enabled != enable:
            self.__turbo_boost = enable
        return enabled

    def __turbo_boost_watchdog(self, enable):
        self._logger.info("Starting turbo boost watchdog for value %s", enable)
        try:
            while True:
                try:
                    self._stopping.get(timeout=1)
                except gevent.timeout.Timeout:
                    enabled = self.__set_turbo_boost(enable)
                    if enabled != enable:
                        self._logger.warning(
                            "External modification of turbo boost detected: %s -> %s, former value restored",
                            enable, enabled
                        )
                    continue
                except gevent.hub.LoopExit:
                    break
                break
        finally:
            self._logger.info("Turbo boost watchdog for value %s finished", enable)

    @jserver.RPC.full()
    def turbo_boost(self, job, enable):
        """
        Enable or disable turbo boost.

        :param enable: enable if True
        """
        enable = bool(enable)
        client_tags = self._config.client.tags
        if not enable and ctc.Tag.MULTISLOT in client_tags:
            raise sandbox.agentr.errors.InvalidPlatform("It is forbidden to disable turbo boost on multislot hosts")
        ses = self.__tasks.associated(job.connection.id)
        logger = sandbox.agentr.utils.LogTee(job.log, ses.logger(job))

        if os.path.exists(self.NO_TURBO_PATH):
            enabled = self.__set_turbo_boost(enable)
            logger.info(
                "Turbo boost%s %sabled",
                " is already" if enabled == enable else "",
                "en" if enable else "dis"
            )
            if self.__turbo_boost_watchdog_greenlet and not self.__turbo_boost_watchdog_greenlet.dead:
                self.__turbo_boost_watchdog_greenlet.kill()
            self.__turbo_boost_watchdog_greenlet = gevent.spawn(self.__turbo_boost_watchdog, enable)
        else:
            raise sandbox.agentr.errors.InvalidPlatform("Turbo boost is not available on current platform")

    @staticmethod
    def __remove_part_of_xcode(logger, path):
        logger.info("Try to remove {}.".format(path))
        xcode_paths = [
            sandbox.agentr.types.Xcode.XCODE_ROOT,
            sandbox.agentr.types.Xcode.DEVELOPER_ROOT,
            sandbox.agentr.types.Xcode.XCODE_CACHE_ROOT,
            sandbox.agentr.types.Xcode.XCODE_SIMULATOR_CACHE_ROOT
        ]
        if not any(path.startswith(_) for _ in xcode_paths):
            raise sandbox.agentr.errors.XcodeError("Path: {} doesn't belong to xcode.".format(path))
        if os.path.lexists(path):
            with common_os.User.Privileges():
                if os.path.isdir(path) and not os.path.islink(path):
                    shutil.rmtree(path)
                else:
                    os.remove(path)
            logger.info("Successfully removed {}.".format(path))
        else:
            logger.info("{} doesn't exists".format(path))

    @staticmethod
    def __execute_macos_cmd(logger, cmd, root=False, cwd=None, shell=False):
        logger.info("Executing the command {}.".format(cmd))
        p = sp.Popen(
            cmd,
            preexec_fn=common_os.User.Privileges().__enter__ if root else None,
            close_fds=True, stdout=sp.PIPE, stderr=sp.PIPE, cwd=cwd, shell=shell
        )
        out, err = p.communicate()
        rc = p.returncode
        logger.info("Rc={}\nOut={}\nErr={}".format(rc, out, err))
        if rc != 0:
            raise sandbox.agentr.errors.XcodeError("{} failed. {}".format(cmd, err))
        logger.info("{} finished successfully.".format(cmd))
        return out

    def __execute_macos_cmd_list(self, commands, logger, shell=False):
        for command in commands:
            try:
                output = self.__execute_macos_cmd(logger, command, root=True, shell=shell)
                logger.debug("{} output: {}".format(command, output))
            except sandbox.agentr.errors.XcodeError:
                logger.info("Failed to execute {}".format(command))

    def __macos_processes_info(self, logger):
        commands = [
            ["ps", "ax"],
            ["launchctl", "list"],
            ["xcrun", "simctl", "list"],
        ]
        logger.info("Collect info about processes, services and simulators")
        self.__execute_macos_cmd_list(commands, logger)

    def __cleanup_simulators(self, logger):
        commands = [
            ["xcrun", "simctl", "shutdown", "all"],
            ['xcrun', 'simctl', 'erase', 'all'],
            ["xcrun", "simctl", "delete", "all"],
        ]
        self.__execute_macos_cmd_list(commands, logger)

    def __kill_simulator_processes(self, logger):
        commands = [
            ["launchctl remove com.apple.CoreSimulator.CoreSimulatorService || True"],
            ["launchctl remove com.apple.CoreSimulator.SimLaunchHost-x86 || True"],
            ["launchctl remove com.apple.CoreSimulator.SimLaunchHost-arm64 || True"],
            ["launchctl remove com.apple.CoreSimulator.SimulatorTrampoline || True"],
        ]
        self.__execute_macos_cmd_list(commands, logger, shell=True)

    def __kill_security_agent(self, logger):
        # https://st.yandex-team.ru/MOBDEVTOOLS-60
        kill_attempts = 0
        while kill_attempts < 5:
            kill_attempts += 1
            try:
                self.__execute_macos_cmd(logger, ["killall", "-9", "SecurityAgent"], root=True)
            except sandbox.agentr.errors.XcodeError:
                break

    def __switch_sources(self, logger, xcode_dir, developer_dir):
        links = [
            [xcode_dir, sandbox.agentr.types.Xcode.XCODE_ROOT],
            [developer_dir, sandbox.agentr.types.Xcode.DEVELOPER_ROOT]
        ]

        self.__cleanup_simulators(logger)
        self.__kill_simulator_processes(logger)
        self.__kill_security_agent(logger)

        self.__macos_processes_info(logger)

        for source, target in links:
            logger.info("Try to link {} to {}.".format(target, source))
            self.__remove_part_of_xcode(logger, target)
            with common_os.User.Privileges():
                os.symlink(source, target)
            logger.info("Link {} to {} created successfully.".format(target, source))

        self.__execute_macos_cmd(logger, ["xcode-select", "--reset"], root=True)
        self.__execute_macos_cmd(
            logger, ["/usr/bin/xcode-select", "-s", sandbox.agentr.types.Xcode.XCODE_ROOT], root=True
        )

        self.__execute_macos_cmd(logger, ["xcodebuild", "-license", "accept"], root=True)
        self.__execute_macos_cmd(logger, ["xcodebuild", "-runFirstLaunch"], root=True)
        self.__macos_processes_info(logger)

    def __xcode_succeeded(self, logger, xcode_cache_dir, xcode_dir, developer_dir):
        try:
            self.__switch_sources(logger, xcode_dir, developer_dir)
        except sandbox.agentr.errors.XcodeError:
            logger.info("Installed version is incorrect. Remove it.")
            self.__remove_part_of_xcode(logger, xcode_cache_dir)
            return False
        logger.info("Installed version is correct. Use it.")
        return True

    def __fetch_extract_xcode(self, logger, ses, ses_logger, version, xcode_cache_dir):
        resources = ses.rest.resource.read(
            type="XCODE_ARCHIVE", state=ctr.State.READY, attrs={"version": version}, limit=1,
        )["items"]
        if not resources:
            raise sandbox.agentr.errors.ResourceNotAvailable(
                "Cannot find Xcode {} in resources.".format(version)
            )
        fetcher = self.Fetcher(logger, ses_logger, ses.id, ses.rest, self._config, self.users, self.__buckets)
        sync_info = fetcher(resources[0]["id"])
        try:
            self.__remove_part_of_xcode(logger, xcode_cache_dir)
            with common_os.User.Privileges():
                os.makedirs(xcode_cache_dir)
            self.__execute_macos_cmd(logger, ["xip", "-x", sync_info.path], root=True, cwd=xcode_cache_dir)
            self.__execute_macos_cmd(logger, ["mv -fv Xcode-beta.app Xcode.app || True"],
                                     root=True,
                                     cwd=xcode_cache_dir,
                                     shell=True,
                                     )
            logger.info("The source of the resource prepared.")
        except sandbox.agentr.errors.XcodeError:
            logger.info("Try to cleanup Xcode.app after failed unpack to xcode_cache_dir")
            self.__remove_part_of_xcode(logger, xcode_cache_dir)
            raise
        finally:
            self.__execute_macos_cmd(logger, ["rm", "-rf", sync_info.path], root=True)

    @jserver.RPC.full()
    def prepare_xcode(self, job, version):

        if not scp.on_osx():
            raise sandbox.agentr.errors.InvalidPlatform("Xcode supported only for darwin.")

        if any(not _.isalnum() for _ in version.split(".")):
            raise sandbox.agentr.errors.XcodeError("Xcode version is incorrect.")

        ses = self.__tasks.associated(job.connection.id)
        ses_logger = ses.logger(job)
        logger = sandbox.agentr.utils.LogTee(job.log, ses.logger(job))
        if not os.path.exists(sandbox.agentr.types.Xcode.XCODE_CACHE_ROOT):
            with common_os.User.Privileges():
                os.makedirs(sandbox.agentr.types.Xcode.XCODE_CACHE_ROOT)
            logger.info("Created {}".format(sandbox.agentr.types.Xcode.XCODE_CACHE_ROOT))

        xcode_cache_dir = sandbox.agentr.types.Xcode.XCODE_CACHE_DIR.format(version)
        xcode_dir = sandbox.agentr.types.Xcode.XCODE_DIR.format(version)
        developer_dir = sandbox.agentr.types.Xcode.DEVELOPER_DIR.format(version)

        if not os.path.exists(xcode_dir) or \
                not self.__xcode_succeeded(logger, xcode_cache_dir, xcode_dir, developer_dir):
            logger.info(
                "Xcode {} isn't found on host or Xcode switch has failed. Try to get from resource.".format(version))
            self.__fetch_extract_xcode(logger, ses, ses_logger, version, xcode_cache_dir)
            self.__switch_sources(logger, xcode_dir, developer_dir)

    def __is_xcode_simulator_already_installed(self, logger, version):
        try:
            runtimes = json.loads(self.__execute_macos_cmd(
                logger, ["xcrun", "simctl", "list", "runtimes", "--json", version]
            ))
            return any(elem["name"] == version for elem in runtimes["runtimes"])
        except Exception as ex:
            logger.info("Failed to get actual simulators.", exc_info=ex)
            return False

    @jserver.RPC.full()
    def prepare_xcode_simulator(self, job, version):

        if not scp.on_osx():
            raise sandbox.agentr.errors.InvalidPlatform("Xcode Simulator supported only for darwin.")

        ses = self.__tasks.associated(job.connection.id)
        ses_logger = ses.logger(job)
        logger = sandbox.agentr.utils.LogTee(job.log, ses.logger(job))

        if self.__is_xcode_simulator_already_installed(logger, version):
            logger.info("Xcode {} Simulator is already installed.".format(version))
            return

        if not os.path.exists(sandbox.agentr.types.Xcode.XCODE_SIMULATOR_CACHE_ROOT):
            with common_os.User.Privileges():
                os.makedirs(sandbox.agentr.types.Xcode.XCODE_SIMULATOR_CACHE_ROOT)
            logger.info("Created {}".format(sandbox.agentr.types.Xcode.XCODE_SIMULATOR_CACHE_ROOT))
        xcode_simulator_cache_dir = sandbox.agentr.types.Xcode.XCODE_SIMULATOR_CACHE_DIR.format(version)
        xcode_simulator_root = sandbox.agentr.types.Xcode.XCODE_SIMULATOR_ROOT.format(version)
        xcode_simulator_dir = sandbox.agentr.types.Xcode.XCODE_SIMULATOR_DIR.format(version, version)
        if not os.path.exists(xcode_simulator_dir):
            logger.info("Xcode {} Simulator isn't found on host. Try to prepare from resource.".format(version))
            resources = ses.rest.resource.read(
                type="XCODE_SIMULATOR_ARCHIVE", owner="MOBDEVTOOLS", state=ctr.State.READY, limit=1,
                attrs={"version": version, "released": "stable"},
            )["items"]
            if not resources:
                raise sandbox.agentr.errors.ResourceNotAvailable(
                    "Cannot find Xcode {} Simulator in resources.".format(version)
                )
            fetcher = self.Fetcher(logger, ses_logger, ses.id, ses.rest, self._config, self.users, self.__buckets)
            sync_info = fetcher(resources[0]["id"])
            try:
                self.__remove_part_of_xcode(logger, xcode_simulator_cache_dir)
                with common_os.User.Privileges():
                    os.makedirs(xcode_simulator_cache_dir)
                self.__execute_macos_cmd(
                    logger, ["tar", "-xvf", sync_info.path], cwd=xcode_simulator_cache_dir, root=True
                )
                logger.info("The source of the resource prepared.")
            except sandbox.agentr.errors.XcodeError as ex:
                logger.info("Try to cleanup Xcode Simulator after failed unpack to xcode_simulator_dir", exc_info=ex)
                self.__remove_part_of_xcode(logger, xcode_simulator_cache_dir)
                raise
            finally:
                self.__execute_macos_cmd(logger, ["rm", "-rf", sync_info.path], root=True)

        logger.info("Try to link {} to {}.".format(xcode_simulator_dir, xcode_simulator_root))
        self.__remove_part_of_xcode(logger, xcode_simulator_root)
        with common_os.User.Privileges():
            runtime_dir = os.path.dirname(xcode_simulator_root)
            if not os.path.exists(runtime_dir):
                os.makedirs(runtime_dir)
            os.symlink(xcode_simulator_dir, xcode_simulator_root)
        logger.info("Link {} to {} created successfully.".format(xcode_simulator_dir, xcode_simulator_root))

    @jserver.RPC.full()
    def prepare_host_for_multixcode(self, job):
        """ Prepare host for multixcode. """

        if sys.platform != "darwin":
            raise sandbox.agentr.errors.InvalidPlatform("Xcode supported only for darwin.")

        ses = self.__tasks.associated(job.connection.id)
        logger = sandbox.agentr.utils.LogTee(job.log, ses.logger(job))
        removed_paths = [
            sandbox.agentr.types.Xcode.XCODE_CACHE_ROOT,
            sandbox.agentr.types.Xcode.XCODE_ROOT,
            sandbox.agentr.types.Xcode.DEVELOPER_ROOT
        ]
        for path in removed_paths:
            self.__remove_part_of_xcode(logger, path)
        with common_os.User.Privileges():
            os.makedirs(sandbox.agentr.types.Xcode.XCODE_CACHE_ROOT)

    @jserver.RPC.full()
    def dependant_resources(self, job):
        tid = self.__tasks.associated(job.connection.id).id
        return self.__db.query(
            """SELECT "resource" FROM "task_deps" WHERE "task" = ?""", (tid,)
        ) or []

    @jserver.RPC.full()
    def remove_dependant_resource(self, job, rid):
        tid = self.__tasks.associated(job.connection.id).id
        self.__db.query_one(
            """DELETE FROM "task_deps" WHERE "task" = ? AND "resource" = ?""", (tid, rid)
        )

    @jserver.RPC.full()
    def cleanup_work_directory(self, job):
        ses = self.__tasks.associated(job.connection.id)
        task_id = ses.id
        logger = sandbox.agentr.utils.LogTee(job.log, ses.logger(job))
        resource_paths = set()
        for resource in ses.rest.resource.read(task_id=task_id, limit=3000)["items"]:
            resource_paths.add(str(ses.taskdir.join(resource["file_name"])))
        logger.debug("Cleaning work directory for task #%s, excluding paths: %s", task_id, resource_paths)
        with common_os.Subprocess(
            "[cleanup_work_directory for task #{}]".format(task_id), logger=logger, using_gevent=True, privileged=True
        ):
            common_fs.cleanup(str(ses.taskdir.join("tmp")), remove_upper_dir=True, logger=logger)
            if self._config.client.auto_cleanup.on_task_finish:
                common_fs.cleanup(
                    str(ses.taskdir), ignore_path=set(resource_paths), remove_upper_dir=False, logger=logger
                )

    @jserver.RPC.full()
    def check_task_files(self, job, task_id, task_resources):
        # TODO: remover after SANDBOX-9317
        ses = self.__tasks.associated(job.connection.id)
        cu = ses.rest.user.current[:]
        if cu["role"] != ctu.Role.ADMINISTRATOR:
            raise ValueError("'{}' is not an administrator".format(cu["login"]))

        extra, missing, ok = common_fs._check_task_files(task_id, task_resources, admin_privileges=True)
        return list(extra), list(missing), list(ok)
