import os
import sys
import time
import logging
import traceback
import functools
import contextlib

import gevent.lock
import gevent.socket
import gevent.monkey

import six

from sandbox.agentr import threading as athreading


class DeblockLock(gevent.lock.RLock):
    def __init__(self, log):
        self._lock_ts = None
        self._lock_trace_disable_reason = None
        self._log = log
        super(DeblockLock, self).__init__()

    def do_not_trace(self, reason):
        self._lock_trace_disable_reason = reason

    def acquire(self, blocking=1, guard=False):
        ret = super(DeblockLock, self).acquire(blocking)
        if ret and self._count == 1:
            if guard:
                self._lock_ts = time.time()
            self._lock_trace_disable_reason = None
        return ret

    def release(self):
        ret = super(DeblockLock, self).release()
        if not self._count and self._lock_ts is not None:
            locked_for = self.locked_for
            if locked_for > 1:
                self._log.warning('Lock was held for %0.2fs by %r', locked_for, gevent.getcurrent())
                if locked_for > 10 and not self._lock_trace_disable_reason:
                    self._log.warning(''.join(traceback.format_stack(sys._getframe(1))))
            self._lock_ts = None
        return ret

    @property
    def locked_for(self):
        if self._lock_ts is not None:
            return time.time() - self._lock_ts
        return 0


class Deblock(object):
    def __init__(self, keepalive=None, logger=None):
        self.log = logger or logging.getLogger('deblock')
        # Set real lock on logger handler
        log = self.log
        lock = athreading.RLock()
        while not log.handlers:
            log = log.parent
        for h in log.handlers:
            h.lock = lock

        # generic lock for all thread operations
        self._lock = athreading.Lock()

        # to prevent simultaneous usage
        self._ex_lock = DeblockLock(self.log.getChild('exlock'))

        self._keepalive = keepalive
        self._stopped = False

        self._rpipe = None
        self._wpipe = None

        self._job = None
        self._job_event = None
        self._job_active = None
        self._job_result = None

    def stop(self):
        self._stopped = True
        self._stopthread()

    def apply(self, meth, *args, **kwargs):
        if self._stopped:
            raise RuntimeError('Deblock object was stopped')

        if self._job_event is None:
            with self._lock:
                if self._job_event is None:
                    self._runthread()

        with self._ex_lock:
            self._job = meth, args, kwargs, getattr(gevent.getcurrent(), 'slocal', {})
            self._job_event.set()

            exc_info = None

            while 1:
                # Somebody could try to kill us here, but we
                # 100% need to wait until deblocked obj finishes
                # If we dont -- result of deblocked obj may be catched up
                # by next call, which will be an error anyway
                try:
                    gevent.socket.wait_read(self._rpipe)
                    break
                except BaseException as ex:
                    # Read request canceled
                    self.log.warning('Canceling read request (%s)', ex)
                    exc_info = sys.exc_info()

            restype = os.read(self._rpipe, 1)

            if exc_info:
                self.log.warning('Canceled, reraising exception (%r)', exc_info)
                six.reraise(*exc_info)

            if restype == 'y':
                return self._job_result
            elif restype == 'n':
                six.reraise(*self._job_result)
            else:
                raise RuntimeError('Unknown error occured (%r)!' % (restype, ))

    def make_proxy(self, obj, put_deblock=None):
        class DeblockProxy(object):
            __slots__ = [m for m in dir(obj) if not m.startswith('_')] + ['_ex_lock']

            if hasattr(obj, '__enter__'):
                @functools.wraps(obj.__enter__)
                def __enter__(self2):  # noqa
                    self._ex_lock.acquire(guard=True)
                    try:
                        self.apply(obj.__enter__)
                    except:
                        self._ex_lock.release()
                        raise

            if hasattr(obj, '__exit__'):
                @functools.wraps(obj.__exit__)
                def __exit__(self2, *args, **kwargs):  # noqa
                    try:
                        self.apply(obj.__exit__, *args, **kwargs)
                    finally:
                        self._ex_lock.release()

            if hasattr(obj, '__call__'):
                @functools.wraps(obj.__call__)
                def __call__(self2, *args, **kwargs):  # noqa
                    return self.apply(obj.__call__, *args, **kwargs)

            _ex_lock = self._ex_lock

            def __repr__(self):
                return '<DeblockProxy %r>' % (obj, )

        if put_deblock:
            setattr(DeblockProxy, put_deblock, self)

        class AttrForward(object):
            def __init__(self, obj, name):
                self.obj = obj
                self.name = name

            def __get__(self, _, owner=None):
                return getattr(self.obj, self.name)

            def __set__(self, _, value):
                setattr(self.obj, self.name, value)

            def __delete__(self, _):
                raise NotImplementedError

        proxy = DeblockProxy()

        for attr in dir(obj):
            if attr not in proxy.__slots__:
                continue

            item = getattr(obj, attr)
            if callable(item):
                setattr(proxy, attr, functools.partial(self.apply, item))
            else:
                setattr(DeblockProxy, attr, AttrForward(obj, attr))

        return proxy

    @contextlib.contextmanager
    def lock(self, reason):
        self._ex_lock.acquire(guard=True)
        self._ex_lock.do_not_trace(reason)

        try:
            yield
        finally:
            self._ex_lock.release()

    @property
    def lock_reason(self):
        return self._ex_lock._lock_trace_disable_reason

    @property
    def lock_waiters(self):
        return len(self._ex_lock._block._links)

    @property
    def locked_for(self):
        return self._ex_lock.locked_for

    @property
    def job(self):
        return self._job_active

    def _thread_loop(self):
        self.log.debug('Loop started')

        try:
            while True:
                if not self._job_event.wait(timeout=self._keepalive):
                    break
                if self._job is None:
                    break
                self._job_event.clear()
                func, args, kwargs, local = self._job_active = self._job

                try:
                    if local:
                        gevent.getcurrent().tlocal = local
                    self._job_result = func(*args, **kwargs)
                    os.write(self._wpipe, 'y')
                    try:
                        del gevent.getcurrent().tlocal
                    except AttributeError:
                        pass
                except:
                    self._job_result = sys.exc_info()
                    os.write(self._wpipe, 'n')

                self._job_active = None

            # Remove ourselves reference, to clean up resources
            with self._lock:
                self._job_event = None
                self._job = None
                os.close(self._rpipe)
                os.close(self._wpipe)
        except BaseException as ex:
            self.log.warning('Loop finished (err:%s)', str(ex))
            raise
        else:
            self.log.debug('Loop finished (ok)')

    def _runthread(self):
        self.log.debug('Run new thread')

        self._job_event = athreading.Event()
        self._rpipe, self._wpipe = os.pipe()
        gevent.monkey.get_original("thread", "start_new_thread")(self._thread_loop, ())

    def _stopthread(self):
        with self._lock:
            if self._job_event:
                self._job = None
                self._job_event.set()
            self._job_event = None
