import os
import json
import stat
import time
import logging
import datetime as dt
import subprocess as sp

import py
import gevent.lock

from sandbox.common import os as common_os
from sandbox.common import log as common_log
from sandbox.common import auth as common_auth
from sandbox.common import rest as common_rest
from sandbox.common import format as common_format
from sandbox.common import patterns as common_patterns
from sandbox.common import statistics as common_statistics

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr
import sandbox.common.types.statistics as cts

import sandbox.agentr.db
import sandbox.agentr.types
import sandbox.agentr.utils
import sandbox.agentr.errors
import sandbox.agentr.storage


class AgentRRestClient(common_rest.Client):
    """Raises AgentR-specific errors for specific situations (like expired sessions) """

    def __init__(self, *args, **kwargs):
        self.token = kwargs.get("auth")
        super(AgentRRestClient, self).__init__(*args, component=ctm.Component.AGENTR, **kwargs)

    def _request(self, *args, **kwargs):
        try:
            return super(AgentRRestClient, self)._request(*args, **kwargs)
        except common_rest.Client.SessionExpired as ex:
            raise sandbox.agentr.errors.NoTaskSession(
                "The token {!r} is expired: {}".format(self.token, ex)
            )
        except common_rest.Client.HTTPError as ex:
            if ex.response.status_code == common_rest.Client.USER_DISMISSED:
                raise sandbox.agentr.errors.UserDismissed("Unauthorized user: {}".format(ex))
            else:
                raise


class Resource(common_patterns.Abstract):
    __slots__ = ("meta", "data", "share", "service")
    __defs__ = (None,) * 4

    class Attrs(object):
        def __init__(self, meta):
            self.__attrs = meta and meta.get("attributes") or {}

        def __getattr__(self, item):
            return self.__attrs.get(item)

    @property
    def id(self):
        return self.meta["id"]

    @property
    def attrs(self):
        return self.Attrs(self.meta)


class Mount(common_patterns.Abstract):
    __slots__ = ("rowid", "source", "target", "type")
    __defs__ = (None,) * 4


class Monitoring(common_patterns.Abstract):
    __slots__ = ("atop_pid", "tcpdump_pid", "peak_dumped", "logs")
    __defs__ = (None, None, None, [])


class MonitoringLog(common_patterns.Abstract):
    __slots__ = ("path", "label", "inode", "offset")
    __defs__ = (None,) * 4


class Log2Push(common_patterns.Abstract):
    __slots__ = ("name", "type")
    __defs__ = (None, None)


class DiskUsage(common_patterns.Abstract):
    __slots__ = ("start", "last", "max", "work_start", "work_last", "work_max", "resources")
    __defs__ = (0,) * len(__slots__)


class HardwareMetrics(common_patterns.Abstract):
    __slots__ = (
        "timestamp", "pid",
        "ram", "cpu",
        "disk_read", "disk_write",
        "disk_read_ops", "disk_write_ops",
        "net_read", "net_write"
    )
    __defs__ = (0, None, 0, 0, 0, 0, 0, 0, 0, 0)


class TaskSession(common_patterns.Abstract):
    __slots__ = (
        "id", "token", "_logger", "logfile", "rest", "taskdir", "logdir", "monitoring",
        "registered_logs", "reserved_space", "dependencies", "lxc",
        "disk_usage", "dump_disk_usage", "quota", "peak_disk_usage_thread", "hardware_metrics",
        "mount_lock",
    )
    __defs__ = (None,) * 8 + ({}, 0, set()) + (None,) * 7

    def logger(self, job):
        return common_log.MessageAdapter(self._logger, fmt="{{{}:{}}} %(message)s".format(job.sid, job.id))


class Registry(object):
    """
    Active tasks' sessions registry - currently executed tasks with disk usage info and connections associations.
    """
    # Auxiliary task session tables, connected with the "task" table via foreign key.
    TASK_AUX_TABLES = (
        "task_monitoring", "task_monitoring_log",
        "task_resource", "task_deps", "task_disk",
        "task_push_logs", "task_process", "task_mount",
    )

    HARDWARE_METRICS_UPDATE_INTERVAL = 5  # in seconds
    MDS_UPLOADER_DELAY_UPDATE_INTERVAL = 60  # in seconds
    LOOP_DEVICE_CLEANER_INTERVAL = 3600  # in seconds
    ONE_BILLION = 1000000000

    def __init__(self, db, log, config, rest, unprivileged_user):
        self.__sessions = {}
        self.__connections = {}
        self.__lock = gevent.lock.RLock()

        self.__db = db
        self._config = config
        self.logger = log.getChild("registry")
        self.unprivileged_user = unprivileged_user
        server_sessions = None
        if self._config.common.installation in ctm.Installation.Group.NONLOCAL:
            try:
                sessions_response = rest.client[self._config.this.id].job.read()
                if sessions_response != rest.RESET:
                    server_sessions = {session["id"]: session for session in sessions_response}
            except rest.HTTPError as ex:
                self.logger.error("Failed to get session of client.", exc_info=ex)

        removed_sessions = []
        with self.__db:
            for row in self.__db.query(
                """ SELECT "id", "session", "iteration", "lxc", "reserved", "meta" FROM "task" """
            ):
                if server_sessions is None or row[1] in server_sessions or row[1] in ctc.ServiceTokens:
                    self.__restore(*row)
                else:
                    removed_sessions.append(row)

            for session in removed_sessions:
                self._drop(session[0])

    @staticmethod
    def __logger(logdir, tid):
        logfile = logdir.join(sandbox.agentr.storage.Fetcher.LOG_NAME)
        logger = logging.getLogger("task" + str(tid))
        logger.handlers = []
        logger.name = "agentr"
        logger.propagate = False
        if not tid:
            # for service tasks we don't need this logger to do anything
            return logger, logfile.strpath

        logfile.ensure()  # ensure file exists in case it is already removed as obsolete
        # FIXME: A hack here - since task can change resource files' privileges on its own, we have to ensure
        # FIXME: write privileges before initializing a handler.
        mode = logfile.lstat().mode
        if not (mode & stat.S_IWUSR and mode & stat.S_IWGRP):
            with common_os.User.Privileges():
                logfile.chmod(mode | stat.S_IWUSR | stat.S_IWGRP)
        logfile = logfile.strpath
        handler = logging.FileHandler(logfile)
        handler.setFormatter(logging.getLogger().handlers[0].formatter)
        handler.lock = None
        logger.addHandler(handler)
        return logger, logfile

    def __restore(self, tid, sid, iteration, lxc, reserved, meta):
        if sid in ctc.ServiceTokens:
            taskdir, logdir = sandbox.agentr.storage.Fetcher.taskdirs(self._config.client.log.root, tid, iteration)
        else:
            taskdir, logdir = sandbox.agentr.storage.Fetcher.taskdirs(
                self._config.client.tasks.data_dir, tid, iteration
            )
        logger, logfile = self.__logger(logdir, tid)
        s = self.__sessions[sid] = TaskSession(tid, sid, logger, logfile, AgentRRestClient(auth=sid))
        # No need to fetch dependencies list additionally
        s.taskdir, s.logdir = taskdir, logdir
        s.quota = sandbox.agentr.storage.Quota(taskdir, create=False)
        s.reserved_space = reserved
        s.lxc = lxc
        du = self.__db.query_one("""
            SELECT "start", "last", "max", "work_start", "work_last", "work_max", "resources"
            FROM "task_disk"
            WHERE "task" = ?
        """, (tid,))
        s.disk_usage = DiskUsage(*du) if du else DiskUsage()
        s.dump_disk_usage = json.loads(meta).get("dump_disk_usage", True)

    def __enter__(self):
        self.__lock.acquire()

    def __exit__(self, *args):
        self.__lock.release()

    def __len__(self):
        return len(self.__sessions)

    def __iter__(self):
        return iter(self.__sessions.values())

    @property
    def connections(self):
        return len(self.__connections)

    def reset(self):
        self.__sessions = {}
        self.__connections = {}
        with self.__db:
            for tbl in self.TASK_AUX_TABLES:
                self.__db.query("""DELETE FROM "{}" """.format(tbl))
            self.__db.query("""DELETE FROM "task" """)

    def associate(self, s, cid):
        self.__connections[cid] = s

    def associated(self, cid, require=True):
        """
        :return: associated task session
        :rtype: `TaskSession`
        """
        s = self.__connections.get(cid)
        if require and s is None:
            raise sandbox.agentr.errors.NoTaskSession()
        return s

    def disconnected(self, cid, logger):
        s = self.__connections.pop(cid, None)
        if s is None:
            return
        tee = sandbox.agentr.utils.LogTee(logger, s._logger)
        left = sorted(k for k, v in self.__connections.iteritems() if v.id == s.id)
        tee.info("Task #%s session connection dropped. %d active connections left: %r", s.id, len(left), left)
        logger.debug(
            "Currently %d active connections for %d task sessions left.",
            len(self.__connections), len(self.__sessions)
        )
        return s

    def get(self, token):
        return self.__sessions.get(token)

    def service(self, cid):
        """ Returns service session or creates new. """
        try:
            return self.associated(cid)
        except sandbox.agentr.errors.NoTaskSession:
            ses = self.add(None, ctc.ServiceTokens.SERVICE_TOKEN, 0, None, 0, 0, None)
            self.associate(ses, cid)
            return ses

    def taskbox(self, cid):
        """ Returns service session or creates new. """
        try:
            return self.associated(cid)
        except sandbox.agentr.errors.NoTaskSession:
            ses = self.add(None, ctc.ServiceTokens.TASKBOX_TOKEN, 0, None, 0, 0, None)
            self.associate(ses, cid)
            return ses

    @staticmethod
    def _get_current_task_info(rest, token):
        try:
            data = rest.task.current[:]
            deps_count = data["requirements"]["resources"]["count"]
            req = (rest << rest.HEADERS({ctm.HTTPHeader.NO_LINKS: "true"}))
            dependencies = req.resource.read(dependant=data["id"], limit=deps_count)["items"]
        except common_rest.Client.HTTPError as ex:
            raise sandbox.agentr.errors.NoTaskSession(
                "The token {!r} provided cannot verify the task session: {}".format(
                    common_format.obfuscate_token(token), ex
                )
            )
        return data, dependencies

    @staticmethod
    def _register_task_log_resource(rest, logger, taskdir, logdir):
        task_log_res = rest.resource({
            "type": ctr.TASK_LOG_RESOURCE_TYPE,
            "description": "Task logs",
            "file_name": logdir.relto(taskdir),
        })
        try:
            rest.resource[task_log_res["id"]].source()
        except Exception as ex:
            logger.exception("Error when adding current host to the resource #%s: %s", task_log_res["id"], ex)
        return task_log_res

    def add(self, job, token, iteration, lxc, reserved_space, disk_start, admin_token):
        if token in ctc.ServiceTokens:
            data = {"id": int(token) * -1}
            rest = common_rest.Client()
            dependencies = []
            iteration = 1
            taskdir, logdir = sandbox.agentr.storage.Fetcher.taskdirs(
                self._config.client.log.root, int(token) * -1, None
            )
        else:
            rest = AgentRRestClient(auth=token, debug=True)
            if admin_token is not None and self._config.common.installation not in ctm.Installation.Group.LOCAL:
                rest <<= rest.HEADERS(dict(common_auth.AdminOAuth(admin_token)))
            data, dependencies = self._get_current_task_info(rest, token)
            iteration = self.__get_iteration(iteration)
            taskdir, logdir = sandbox.agentr.storage.Fetcher.ensure_logdir(
                job.log, self.unprivileged_user, self._config.client.tasks.data_dir, data["id"], iteration
            )

        logger, logfile = self.__logger(logdir, data["id"])
        s = TaskSession(data["id"], token, logger, logfile, rest)
        s.dependencies = {_["id"] for _ in dependencies}
        s.taskdir, s.logdir = taskdir, logdir
        s.lxc = lxc

        task_log_res = None
        if token not in ctc.ServiceTokens:
            task_log_res = self._register_task_log_resource(rest, logger, taskdir, logdir)

        # XXX: [SANDBOX-3330] cannot use project quotas for privileged tasks (Invalid cross-device link)
        privileged = data.get("requirements", {}).get("privileged")
        if (
            token not in ctc.ServiceTokens and
            self._config.common.installation not in ctm.Installation.Group.LOCAL and
            not privileged
        ):
            s.quota = sandbox.agentr.storage.Quota(taskdir)
        s.reserved_space = reserved_space
        s.disk_usage = DiskUsage(disk_start, 0, 0, sandbox.agentr.storage.Quota.usage([s])[0])
        s.dump_disk_usage = data.get("dump_disk_usage", True)

        with self.__db:
            self._drop(data["id"])  # TODO: It should not happens, but sometimes it does..
            self.__db.query(
                """
                  INSERT INTO "task" (
                    "id", "session", "iteration", "lxc", "reserved", "meta"
                  ) VALUES (
                    ?, ?, ?, ?, ?, ?
                  )
                """,
                (data["id"], token, iteration, lxc, reserved_space, json.dumps(data)),
                log=sandbox.agentr.db.Log.STATEMENT
            )
            self.__db.query(
                """INSERT INTO "task_disk" ("task", "start", "work_start") VALUES (?, ?, ?)""",
                (data["id"], disk_start, s.disk_usage.work_start)
            )
            for rid in s.dependencies:
                self.__db.query("""INSERT INTO "task_deps" ("task", "resource") VALUES (?, ?)""", (s.id, rid))
            if task_log_res:
                self.new_resource(s, task_log_res, taskdir, False, True)

        self.__sessions[token] = s
        return s

    @staticmethod
    def __get_iteration(iteration):
        try:
            result = int(iteration)
            if result < 1:
                raise ValueError("Non-positive number")
        except (TypeError, ValueError):
            raise ValueError("{!r} is not an iteration number".format(iteration))
        return result

    def set_lxc(self, ses, lxc):
        self.__sessions[ses.token].lxc = ses.lxc = lxc
        return self.__db.query_one("""UPDATE "task" SET "lxc" = ? WHERE "id" = ?""", (lxc, ses.id))

    def meta(self, tid):
        return json.loads(self.__db.query_one("""SELECT "meta" FROM "task" WHERE "id" = ?""", (tid,))[0])

    def meta_update(self, tid, data):
        return self.__db.query_one(
            """UPDATE "task" SET "meta" = ? WHERE "id" = ?""", (json.dumps(data), tid),
            log=sandbox.agentr.db.Log.STATEMENT
        )

    def all_meta(self):
        return {tid: json.loads(meta) for tid, meta in self.__db.query("""SELECT "id", "meta" FROM "task" """)}

    def _drop(self, tid):
        with self.__db:
            for tbl in self.TASK_AUX_TABLES:
                self.__db.query("""DELETE FROM "{}" WHERE "task" = ?""".format(tbl), (tid,))
            self.__db.query("""DELETE FROM "task" WHERE "id" = ?""", (tid,))
        return tid

    def remove(self, s, cid, logger):
        logs2push = self.logs2push(s)
        self.__sessions.pop(s.token)
        self._drop(s.id)
        self.__connections.pop(cid, None)
        left = sorted(cid_ for cid_, s_ in self.__connections.iteritems() if s.id == s_.id)
        if left:
            logger.warning("%d active connection for the session found: %r", len(left), left)
            map(self.__connections.pop, left)
        return logs2push

    def account(self, tid, rid, downloaded):
        res = self.__db.query(
            """UPDATE "task_deps" SET "fetched" = ? WHERE "task" = ? AND "resource" = ?""",
            (int(bool(downloaded)) + 1, tid, rid),
            get_changed=True
        )
        if not res:
            self.__db.query(
                """INSERT INTO "task_deps" ("fetched", "task", "resource") VALUES (?, ?, ?)""",
                (int(bool(downloaded)) + 3, tid, rid),
            )
        return res

    def accounted(self, tid, fetched=sandbox.agentr.types.DependencyFetch.DECLARED_FETCHED):
        return self.__db.query_one(
            """
                SELECT SUM("resource"."size")
                FROM "task_deps" LEFT JOIN "resource" ON "resource"."id" = "task_deps"."resource"
                WHERE "task_deps"."task" = ? AND "task_deps"."fetched" = ?
            """,
            (tid, fetched)
        )[0] or 0

    @property
    def locked(self):
        return set(row[0] for row in self.__db.query("""SELECT DISTINCT "resource" FROM "task_deps" """))

    def cache_expired(self):
        """
        Return all resources which expired based on local copy creation/touch date
        """
        now = dt.datetime.utcnow()
        now_str = now.strftime(sandbox.agentr.types.DT_FMT)
        return set(row[0] for row in self.__db.query(
            """SELECT "id" FROM "resource" WHERE "cache_expires" < ?""", (now_str,))
        )

    def update_disk_usage(self, session, occupied, quota_usage):
        occupied -= self.accounted(session.id)
        session.disk_usage.last = max(0, occupied - session.disk_usage.start)
        session.disk_usage.max = max(session.disk_usage.last, session.disk_usage.max)
        session.disk_usage.work_last = max(0, quota_usage - session.disk_usage.work_start)
        session.disk_usage.work_max = max(session.disk_usage.work_last, session.disk_usage.work_max)
        self.__db.query(
            """
            UPDATE "task_disk"
            SET "last" = ?, "max" = ?, "work_last" = ?, "work_max" = ?, "resources" = ?
            WHERE "task" = ?
            """, (
                session.disk_usage.last, session.disk_usage.max,
                session.disk_usage.work_last, session.disk_usage.work_max, session.disk_usage.resources,
                session.id
            )
        )
        if 0 < session.reserved_space < session.disk_usage.last and session.dump_disk_usage:
            mon = self.monitoring(session)
            if mon and not mon.peak_dumped:
                logger = sandbox.agentr.utils.LogTee(self.logger, session._logger)
                logger.debug(
                    "Dumping peak disk usage: reserved space=%s, disk_usage=%s",
                    session.reserved_space, session.disk_usage.last
                )
                du_thread = gevent.Greenlet.spawn(
                    lambda: sandbox.agentr.storage.Fetcher.dump_disk_usage(
                        logger, ctm.DiskUsageType.PEAK, session.taskdir, session.logdir, self.unprivileged_user,
                        wait=True
                    )
                )
                session.peak_disk_usage_thread = du_thread
                self.set_peak_dumped(session)

        return occupied

    def _get_hardware_metrics(self, pid, session):
        if not os.path.exists("/proc/{}".format(pid)):
            return
        now = time.time()
        timestamp = int(now) * self.ONE_BILLION + int((now - int(now)) * self.ONE_BILLION)
        if session.lxc:
            cgroup = common_os.CGroup("/lxc/{}".format(session.lxc))
        else:
            cgroup = common_os.CGroup(pid=pid)
        cgroup_cpuacct = cgroup.cpuacct
        cgroup_memory = cgroup.memory
        cgroup_blkio = cgroup["blkio"]
        if not (
            cgroup_cpuacct.cgroup.name and cgroup_cpuacct.exists and
            cgroup_memory.cgroup.name and cgroup_memory.exists and
            cgroup_blkio.cgroup.name and cgroup_blkio.exists
        ):
            logging.debug(
                "Can't get CPU/RAM/DISK stats: cgroup files not exist (%s, %s, %s)",
                cgroup_cpuacct, cgroup_memory, cgroup_blkio
            )
            return
        logging.debug("Getting CPU stats for task #%s (process #%s) from %s", session.id, pid, cgroup_cpuacct)
        logging.debug("Getting RAM stats for task #%s (process #%s) from %s", session.id, pid, cgroup_memory)
        logging.debug("Getting disk IO stats for task #%s (process #%s) from %s", session.id, pid, cgroup_blkio)
        cpu = int(cgroup_cpuacct["usage"][0])
        ram = int(cgroup_memory["usage_in_bytes"][0])
        disk_read = 0
        disk_write = 0
        for line in cgroup_blkio["io_service_bytes"]:
            rows = line.split()
            if rows[1] == "Read":
                disk_read += int(rows[2])
            elif rows[1] == "Write":
                disk_write += int(rows[2])
        disk_read_ops = 0
        disk_write_ops = 0
        for line in cgroup_blkio["io_serviced"]:
            rows = line.split()
            if rows[1] == "Read":
                disk_read_ops += int(rows[2])
            elif rows[1] == "Write":
                disk_write_ops += int(rows[2])
        net_read = 0
        net_write = 0
        with open("/proc/{}/net/dev".format(pid)) as f:
            for line in f:
                rows = line.split()
                if not rows[0].startswith("eth"):
                    continue
                net_read = int(rows[1])
                net_write = int(rows[9])
                logging.debug("Network device for process #%s: %s", pid, rows[0])
                break
        return HardwareMetrics(
            timestamp=timestamp, pid=pid,
            ram=ram, cpu=cpu,
            disk_read=disk_read, disk_write=disk_write,
            disk_read_ops=disk_read_ops, disk_write_ops=disk_write_ops,
            net_read=net_read, net_write=net_write,
        )

    def update_hardware_metrics(self, session):
        if not session.id:
            return
        pid = self.executor_pid_getter(session.id) if session.hardware_metrics is None else session.hardware_metrics.pid
        hardware_metrics = self._get_hardware_metrics(pid, session)
        logging.debug("Hardware metrics for task #%s: %s", session.id, hardware_metrics)
        if (
            hardware_metrics is None or
            hardware_metrics.cpu is None or
            hardware_metrics.ram is None or
            hardware_metrics.disk_read is None
        ):
            return
        prev_hardware_metrics = session.hardware_metrics
        if prev_hardware_metrics is None:
            session.hardware_metrics = hardware_metrics
            return
        time_delta = hardware_metrics.timestamp - prev_hardware_metrics.timestamp
        time_delta_sec = time_delta // self.ONE_BILLION
        if time_delta_sec <= 0:
            return
        cpu = (hardware_metrics.cpu - prev_hardware_metrics.cpu) * 100 // time_delta
        ram = hardware_metrics.ram
        disk_read = (hardware_metrics.disk_read - prev_hardware_metrics.disk_read) // time_delta_sec
        disk_write = (hardware_metrics.disk_write - prev_hardware_metrics.disk_write) // time_delta_sec
        disk_read_ops = (hardware_metrics.disk_read_ops - prev_hardware_metrics.disk_read_ops) // time_delta_sec
        disk_write_ops = (hardware_metrics.disk_write_ops - prev_hardware_metrics.disk_write_ops) // time_delta_sec
        net_read = (hardware_metrics.net_read - prev_hardware_metrics.net_read) // time_delta_sec
        net_write = (hardware_metrics.net_write - prev_hardware_metrics.net_write) // time_delta_sec
        timestamp = hardware_metrics.timestamp // self.ONE_BILLION
        # sometimes these metrics temporary become less than the former, at the next measurement normal values come back
        if net_read < 0 or net_write < 0:
            return
        session.hardware_metrics = hardware_metrics
        metrics = dict(
            type=cts.SignalType.TASK_HARDWARE_METRICS,
            date=timestamp,
            timestamp=timestamp,
            client_id=self._config.this.id,
            task_id=session.id,
            cpu=cpu, ram=ram,
            disk_read=disk_read, disk_write=disk_write,
            disk_read_ops=disk_read_ops, disk_write_ops=disk_write_ops,
            net_read=net_read, net_write=net_write,
        )
        logging.debug("Sending hardware metrics for task #%s: %s", session.id, metrics)
        # FIXME: don't push wrong values to avoid big gaps in metrics
        if cpu < 0 or cpu > 8000:
            return
        common_statistics.Signaler().push(metrics)

    def hardware_metrics_pusher(self):
        self.logger.info("Hardware metrics pusher started")
        while True:
            for session in self:
                # noinspection PyBroadException
                try:
                    self.update_hardware_metrics(session)
                except (gevent.GreenletExit, gevent.hub.LoopExit):
                    self.logger.info("Hardware metrics pusher exited")
                    raise
                except Exception:
                    self.logger.exception("Unexpected error while updating metrics for task #%s", session.id)
            gevent.sleep(self.HARDWARE_METRICS_UPDATE_INTERVAL)

    def mds_uploader_delay_pusher(self):
        self.logger.info("MDS uploader delay pusher started")
        while True:
            # noinspection PyBroadException
            try:
                amount, size = self.__db.query_one(
                    """
                      SELECT count(), sum("size") FROM "upload_to_mds"
                    """
                )
                timestamp = int(time.time())
                metrics = dict(
                    type=cts.SignalType.RESOURCES_SYNC_TO_MDS_DELAY,
                    date=timestamp,
                    timestamp=timestamp,
                    client_id=self._config.this.id,
                    amount=amount,
                    size=size or 0,
                )
                common_statistics.Signaler().push(metrics)
            except (gevent.GreenletExit, gevent.hub.LoopExit):
                self.logger.info("MDS uploader delay pusher exited")
                raise
            except Exception:
                self.logger.exception("Unexpected error while updating MDS uploader delay")
            gevent.sleep(self.MDS_UPLOADER_DELAY_UPDATE_INTERVAL)

    def loop_device_cleaner(self):
        self.logger.info("Loop device cleaner started")
        while True:
            p = sp.Popen(
                ["mount | grep deleted | cut -d ' ' -f 4 | xargs -r umount"],
                preexec_fn=common_os.User.Privileges().__enter__,
                shell=True, close_fds=True, stdout=sp.PIPE, stderr=sp.PIPE
            )
            if p.wait():
                self.logger.error("Subprocess finished with exit code %d", p.returncode)

            gevent.sleep(self.LOOP_DEVICE_CLEANER_INTERVAL)

    def new_resource(self, ses, res, path, share, service):
        _ERR_MSG = (
            "Resource #{} (type: {!r}, attributes: {!r} at '{}') {} already registered resource "
            "#{} (type: {!r}, attributes: {!r} at '{}')"
        )
        # check for duplicate resources
        for row in self.__db.query(
            """SELECT "resource", "type", "path", "meta" FROM "task_resource" WHERE "task" = ? AND "resource" != ?""",
            (ses.id, res["id"])
        ):
            meta = json.loads(row[3])
            a, b = sorted([meta["file_name"], res["file_name"]])
            if a.startswith(b):
                raise sandbox.agentr.errors.InvalidResource(_ERR_MSG.format(
                    res["id"], res["type"], res.get("attributes"), res["file_name"], "has path intersection with",
                    row[0], row[1], meta.get("attributes"), meta["file_name"]
                ))
            if a != b:
                continue
            if row[1] == res["type"] and res.get("attributes") == meta.get("attributes"):
                res["id"] = row[0]
                break
            raise sandbox.agentr.errors.InvalidResource(_ERR_MSG.format(
                res["id"], res["type"], res.get("attributes"), res["file_name"], "is a duplicate of",
                row[0], row[1], meta.get("attributes"), meta["file_name"]
            ))

        if "http" not in res:
            res["http"] = {"proxy": None}
        if "sources" not in res:
            res["sources"] = []
        now = dt.datetime.utcnow().strftime(sandbox.agentr.types.DT_FMT)
        self.__db.query(
            """
              REPLACE INTO "task_resource" (
                "resource", "task", "type", "updated", "path", "meta", "share", "service"
              ) VALUES (?, ?, ?, ?, ?, ?, ?, ?)
            """,
            (res["id"], ses.id, res["type"], now, path.strpath, json.dumps(res), int(share), int(service))
        )
        return res["id"]

    def check_resource(self, tid, rid, drop=False):
        res = self.__db.query_one(
            """
              SELECT "task", "meta", "path", "share", "service" FROM "task_resource"
              WHERE "resource" = ?
            """,
            (rid,)
        )
        if not res:
            return None
        if res[0] != tid:
            raise ValueError("Resource #{} registered by task #{} instead of #{}".format(rid, res[0], tid))
        if drop:
            self.__db.query("""DELETE FROM "task_resource" WHERE "resource" = ?""", (rid,))
        return Resource(json.loads(res[1]), py.path.local(res[2]), bool(res[3]), bool(res[4]))

    def task_resources(self, tid, done=False):
        q = self.__db.query(
            """SELECT "resource", "meta", "path", "share", "service" FROM "task_resource" WHERE "task" = ?""",
            (tid,)
        )
        if done:
            self.__db.query("""DELETE FROM "task_resource" WHERE "task" = ?""", (tid,))
        return {
            _[0]: Resource(json.loads(_[1]), py.path.local(_[2]), bool(_[3]), bool(_[4]))
            for _ in q
        }

    def find_log_resource(self, ses):
        file_name = ses.logdir.relto(ses.taskdir)
        resources = self.__db.query(
            """SELECT "meta" FROM "task_resource" WHERE "task" = ?""",
            (ses.id, )
        )
        for rdata in resources:
            res = json.loads(rdata[0])
            if res.get("file_name") == file_name:
                return res

    def monitoring_init(self, ses):
        self.__db.query(
            """INSERT INTO "task_monitoring" ("task") VALUES (?)""",
            (ses.id,)
        )

    def monitoring_add_log(self, ses, path, label, inode, offset):
        self.__db.query(
            """INSERT INTO "task_monitoring_log" ("task", "path", "label", "inode", "offset") VALUES (?, ?, ?, ?, ?)""",
            (ses.id, path, label, inode, offset)
        )

    def monitoring(self, ses, with_logs=False):
        res = self.__db.query_one(
            """SELECT "atop_pid", "tcpdump_pid", "peak_dumped", "syslog_inode", "syslog_offset"
            FROM "task_monitoring" WHERE "task" = ?""",
            (ses.id,)
        )
        logs = []
        if res and with_logs:
            logs = self.__db.query(
                """SELECT "path", "label", "inode", "offset"
                FROM "task_monitoring_log" WHERE "task" = ?""",
                (ses.id,)
            )
            if not logs:
                # backward compatibility for SANDBOX-4430
                logs = [[common_os.system_log_path(), "system.log", res[3], res[4]]]
            logs = [MonitoringLog(*_) for _ in logs]

        return Monitoring(res[0], res[1], res[2], logs) if res else None

    def monitoring_done(self, ses):
        self.__db.query("""DELETE FROM "task_monitoring" WHERE "task" = ?""", (ses.id,))
        self.__db.query("""DELETE FROM "task_monitoring_log" WHERE "task" = ?""", (ses.id,))

    def register_atop(self, ses, pid):
        self.__db.query("""UPDATE "task_monitoring" SET "atop_pid" = ? WHERE "task" = ?""", (pid, ses.id))

    def register_tcpdump(self, ses, pid):
        self.__db.query("""UPDATE "task_monitoring" SET "tcpdump_pid" = ? WHERE "task" = ?""", (pid, ses.id))

    def set_peak_dumped(self, ses):
        self.__db.query("""UPDATE "task_monitoring" SET "peak_dumped" = 1 WHERE "task" = ?""", (ses.id,))

    def new_log2push(self, ses, file_name, log_type):
        self.__db.query(
            """INSERT INTO "task_push_logs" ("task", "name", "type") VALUES (?, ?, ?)""",
            (ses.id, file_name, log_type)
        )

    def logs2push(self, ses):
        res = self.__db.query(
            """SELECT "name", "type" FROM "task_push_logs" WHERE "task" = ?""",
            (ses.id,)
        )
        return [Log2Push(*_) for _ in res]

    def new_task_process(self, ses, pid):
        self.__db.query(
            """INSERT INTO "task_process" ("task", "pid") VALUES (?, ?)""",
            (ses.id, pid)
        )

    def delete_task_process(self, ses, pid):
        self.__db.query(
            """DELETE FROM "task_process" WHERE "task" = ? AND "pid" = ?""",
            (ses.id, pid)
        )

    def add_coredump(self, tid, pid, coredump_path):
        self.__db.query(
            """UPDATE "task_process" SET "coredump" = ? WHERE "task" = ? AND "pid" = ? """,
            (coredump_path.strpath, tid, pid)
        )

    def task_coredumps(self, ses):
        res = self.__db.query(
            """SELECT "pid", "coredump" FROM "task_process" WHERE "task" = ?""",
            (ses.id,)
        )
        return {pid: coredump for pid, coredump in res if coredump}

    def session_by_pid(self, pid):
        sid = self.__db.query_one(
            """
              SELECT "task"."session"
              FROM "task_process" LEFT JOIN "task" ON "task"."id" = "task_process"."task"
              WHERE "task_process"."pid" = ?
            """,
            (pid,)
        )
        if sid:
            return self.__sessions.get(sid[0])

    def job_state_getter(self, ses):
        state = self.__db.query_one("""SELECT "state" FROM "task" WHERE "session" = ?""", (ses.token, ))
        if state:
            return state[0]

    def job_state_setter(self, ses, state):
        self.__db.query(
            """UPDATE "task" SET "state" = ? WHERE "session" = ?""", (state, ses.token),
            log=sandbox.agentr.db.Log.STATEMENT
        )

    def jobs(self):
        return dict(self.__db.query("""SELECT "session", "state" FROM "task" WHERE "state" IS NOT NULL"""))

    def fileserver_meta_getter(self, tid):
        result = self.__db.query_one(
            """SELECT "fileserver_meta" FROM "task" WHERE "id" = ?""", (tid,)
        )
        return result and result[0] and json.loads(result[0])

    def fileserver_meta_setter(self, tid, fs_meta):
        self.__db.query_one(
            """UPDATE "task" SET "fileserver_meta" = ?  WHERE "id" = ?""", (json.dumps(fs_meta), tid)
        )

    def executor_pid_getter(self, tid):
        result = self.__db.query_one(
            """SELECT "executor_pid" FROM "task" WHERE "id" = ?""", (tid,)
        )
        return result and result[0]

    def executor_pid_setter(self, tid, pid):
        self.__db.query_one(
            """UPDATE "task" SET "executor_pid" = ?  WHERE "id" = ?""", (pid, tid)
        )

    def progress_meta_getter(self, tid):
        result = self.__db.query(
            """
            SELECT "message", "started", "current", "total", "percentage"
            FROM "action"
            WHERE "task" = ?
            """,
            (tid,),
        )
        return sorted(result, key=lambda x: x[1]) or None

    def progress_meta_setter(self, meter_id, tid, progress_meta):
        self.__db.query(
            """
            REPLACE INTO "action" (
                "id", "task", "message", "started", "current", "total", "percentage"
            ) VALUES (?, ?, ?, ?, ?, ?, ?)
            """,
            (meter_id, tid) + tuple(progress_meta)
        )

    def progress_meta_deleter(self, meter_id):
        self.__db.query_one(
            """DELETE FROM "action" WHERE "id" = ?""", (meter_id,)
        )

    def mount(self, tid, src, dst, fs_type):
        self.__db.query_one(
            """INSERT INTO "task_mount" ("task", "source", "target", "type") VALUES (?, ?, ?, ?)""",
            (tid, str(src), str(dst), str(fs_type))
        )

    def mounts(self, tid):
        return [Mount(*_) for _ in self.__db.query(
            """SELECT "rowid", "source", "target", "type" FROM "task_mount" WHERE "task" = ? ORDER BY rowid DESC""",
            (tid,)
        )]

    def umount(self, mount):
        self.__db.query_one(
            """DELETE FROM "task_mount" WHERE "rowid" = ?""",
            (mount.rowid,)
        )
