from __future__ import absolute_import

import collections

from sandbox.common import enum
from sandbox.common import patterns

# Main process title and sub-processes title prefix
PROCESS_TITLE_PREFIX = "[sandbox] AgentR"

# Disk space information reported by AgentR service.
DiskSpaceInfo = collections.namedtuple("DiskSpaceInfo", ("free", "locked", "total"))

# Cache information reported by AgentR service.
CacheInfo = collections.namedtuple("CacheInfo", ("amount", "size"))

# Resource synchronization result of fetcher call
ResourceSyncInfo = collections.namedtuple("ResourceSyncInfo", ("resource", "path", "downloaded", "size"))

# Task metadata for fileserver: commands for process list and web shell and executor cgroup to run the shell in
FileServerMeta = collections.namedtuple(
    "FileServerMeta", ("ps_command", "shell_command", "cgroup", "attach_command", "pid")
)

# Return code for AgentR worker if downloading resource by http from mds not allowed for resource
MDS_DOWNLOAD_NOT_ALLOWED = 3


# Current action metadata for a task (see sandbox.sdk2.helpers.misc.ProgressMeter); used in AgentR and the said helper.
class Action(
    collections.namedtuple("Action", ("message", "started", "current", "total", "percentage"))
):
    def encode(self):
        progress = None
        if self.percentage is not None:
            progress = dict(
                current=self.current,
                total=self.total,
                percentage=self.percentage,
            )
        return dict(
            message=self.message,
            started=self.started,
            progress=progress,
        )


# Cleanup results information
CleanupResult = collections.namedtuple("CleanupResult", ("kind", "removable", "removed"))

# The datetime format, which is used by SQLite to store
DT_FMT = "%Y-%m-%dT%H:%M:%S"

# Filenames for executor's stderr and stdout
STDERR_FILENAME = "executor.err"
STDOUT_FILENAME = "executor.out"

# Task's working directory's sub-directory name for ArcadiaAPI FUSE mount points
FUSE_DIRNAME = "__FUSE"


class LogType(enum.Enum):
    """
    Log types to upload by statbox push client
    """
    TASK = "sandbox-task-log"
    STAT_TASK = "sandbox-stat-task-log"  # TODO: remove after SANDBOX-4907
    STAT_TASK_EXECUTION = "task-execution-log"
    PYTHON_STATINFRA = "python-statinfra-log"
    STATBOX_DAEMON = "statbox-daemon-log"


class MaintainLimits(patterns.Abstract):
    """ Files removal limits on maintain procedure. """

    __slots__ = ("extra_local", "extra_actual_local", "extra_remote")
    __defs__ = [None] * 3


class DependencyFetch(enum.Enum):
    """ Task's resource dependency state """
    DECLARED_NOT_REQUESTED = 0      # Dependency declared but not requested
    DECLARED_FOUND_LOCALLY = 1      # Dependency declared, requested and found in the local cache
    DECLARED_FETCHED = 2            # Dependency declared and fetched from external source
    NOT_DECLARED_FOUND_LOCALLY = 3  # Dependency was NOT declared but requested and found in the local cache
    NOT_DECLARED_FETCHED = 4        # Dependency was NOT declared but requested and fetched from external source


# Describes resource fetched from local DB
LocalResource = collections.namedtuple("LocalResource", "bucket id meta")


class Xcode(object):
    XCODE_CACHE_ROOT = "/usr/local/xcode-cache"
    XCODE_CACHE_DIR = "/usr/local/xcode-cache/{}"
    XCODE_ROOT = "/Applications/Xcode.app"
    XCODE_DIR = "/usr/local/xcode-cache/{}/Xcode.app"
    DEVELOPER_ROOT = "/Library/Developer"
    DEVELOPER_DIR = "/usr/local/xcode-cache/{}/Developer"
    XCODE_SIMULATOR_CACHE_ROOT = "/usr/local/xcode-simulator-cache"
    XCODE_SIMULATOR_CACHE_DIR = "/usr/local/xcode-simulator-cache/{}"
    XCODE_SIMULATOR_ROOT = "/Library/Developer/CoreSimulator/Profiles/Runtimes/{}.simruntime"
    XCODE_SIMULATOR_DIR = "/usr/local/xcode-simulator-cache/{}/{}.simruntime"
