import os
import stat
import yaml
import ctypes
import subprocess as sp
import collections

import gevent
import gevent.lock

import six

import aniso8601

from sandbox.common import os as common_os
from sandbox.common import errors as common_errors
from sandbox.common import format as common_format
from sandbox.common import patterns as common_patterns

from .. import errors


# add support for OrderedDict to yaml
yaml.SafeDumper.add_representer(
    collections.OrderedDict,
    lambda dumper, data: dumper.represent_mapping(
        yaml.resolver.BaseResolver.DEFAULT_MAPPING_TAG, data.items()
    )
)


def get_disk_usage(path, allow_files=False):
    """
        Get file size information within a directory (in a tree format)

        :param path: path to a directory
        :param allow_files: allow the path to be a single file
        :return: tuple (dictionary *name of file/directory* -> *file size or dictionary for subfolder*, folder size)
        :rtype: tuple(dict, int)
    """
    path = os.path.abspath(path)
    nodes = {}
    dir_size = 0
    if six.PY2:
        import scandir
        fscandir = scandir.scandir
    else:
        fscandir = os.scandir
    try:
        if allow_files and os.path.isfile(path):
            file_size = os.stat(path)[stat.ST_SIZE]
            nodes[os.path.basename(path)] = file_size
            dir_size += file_size
        else:
            for entry in fscandir(path):
                gevent.sleep()
                if entry.is_file(follow_symlinks=False):
                    file_size = entry.stat()[stat.ST_SIZE]
                    nodes[entry.name] = file_size
                    dir_size += file_size
                elif entry.is_dir(follow_symlinks=False):
                    entry_nodes, entry_size = get_disk_usage(entry.path)
                    nodes[entry.name] = {
                        'nodes': entry_nodes,
                        'size': entry_size,
                    }
                    dir_size += entry_size
    except OSError as ex:
        return {'error': str(ex)}, 0

    def sort_key(item):
        value = item[1]
        if isinstance(value, dict):
            return value['size']
        return value

    nodes = collections.OrderedDict(sorted(nodes.items(), key=sort_key, reverse=True))
    return nodes, dir_size


def dump_dir_disk_usage_scandir(path, outpath, mode=None):
    """
        Dump disk usage information of a folder (in yaml format)

        :param path: path to a directory
        :param outfile: path to output
        :return: True if successful
        :rtype: bool
    """
    path = os.path.abspath(path)
    outpath = os.path.abspath(outpath)
    nodes = get_disk_usage(path)[0]
    if mode is None:
        if os.path.isfile(outpath):
            mode = 'r+'
        else:
            mode = 'w'

    with open(outpath, mode) as out:
        yaml.safe_dump(nodes, out, default_flow_style=False)
        if mode == 'r+':
            out.truncate()


def str2dt(value):
    """ Parse ISO 8601 date string to UTC datetime object. """
    return aniso8601.parse_datetime(value).replace(tzinfo=None)


class LogTee(object):
    """ Combines a number of logger objects into a single instance. """

    def __init__(self, *loggers):
        self.__loggers = loggers
        self.__methods = []

    def __getattr__(self, item):
        if self.__methods:
            raise AttributeError("'function' object has no attribute {!r}".format(item))
        self.__methods = [getattr(_, item) for _ in self.__loggers]
        return self

    def __call__(self, *args, **kwargs):
        if not self.__methods:
            raise TypeError("'module' object is not callable")
        ret = [_(*args, **kwargs) for _ in self.__methods][0]
        self.__methods = []
        return ret


class ProjectQuota(object):
    """
    Helper class for working with project quotas (https://patchwork.kernel.org/patch/5813281/):

    Usage examples:

    .. code-block:: python

        # create quota
        p = ProjectQuota("/some/directory")

        # destroy quota:
        p.destroy()

        # get project id:
        project_id = p.project

        # set project id:
        p.project = 123456

        # get project disk usage in bytes:
        disk_usage = p.usage
    """
    class Error(Exception):
        pass

    PRJQUOTA = 2
    Q_GETQUOTA = 0x800007
    Q_SETQUOTA = 0x800008
    QIF_DQBLKSIZE = 1 << 10
    QIF_BLIMITS = 1
    PROJECT_QUOTA_BINARY = "/usr/bin/project_quota"

    # noinspection PyPep8Naming
    class dqblk(ctypes.Structure):
        """
        .. code-block:: c

            struct dqblk {
                u_int64_t dqb_bhardlimit;   /* absolute limit on disk quota blocks alloc */
                u_int64_t dqb_bsoftlimit;   /* preferred limit on disk quota blocks */
                u_int64_t dqb_curspace;     /* current quota block count */
                u_int64_t dqb_ihardlimit;   /* maximum # allocated inodes */
                u_int64_t dqb_isoftlimit;   /* preferred inode limit */
                u_int64_t dqb_curinodes;    /* current # allocated inodes */
                u_int64_t dqb_btime;        /* time limit for excessive disk use */
                u_int64_t dqb_itime;        /* time limit for excessive files */
                u_int32_t dqb_valid;        /* bitmask of QIF_* constants */
            };
        """

        _fields_ = [
            ("bhardlimit", ctypes.c_ulonglong),
            ("bsoftlimit", ctypes.c_ulonglong),
            ("curspace", ctypes.c_ulonglong),
            ("ihardlimit", ctypes.c_ulonglong),
            ("isoftlimit", ctypes.c_ulonglong),
            ("curinodes", ctypes.c_ulonglong),
            ("btime", ctypes.c_ulonglong),
            ("itime", ctypes.c_ulonglong),
            ("valid", ctypes.c_uint)
        ]

    MountPoint = collections.namedtuple("MountPoint", "device fstype root_path stat")
    QuotaInfo = collections.namedtuple("QuotaInfo", "usage iusage limit ilimit")

    supported = False

    def __init__(self, path, create=True, popen=None):
        """
        :param path: path to set quota on
        :param create: if False then create object only
        :param popen: replace subprocess.Popen, used internally, with the own one
        """
        self.__project_id = None
        self.__path = path
        self.__popen = popen or sp.Popen
        if self.supported and create:
            cmd = [self.PROJECT_QUOTA_BINARY, "create", path]
            process = self.__popen(cmd, stdout=sp.PIPE, stderr=sp.PIPE)
            _, err = process.communicate()
            if process.returncode and "already in the project" not in err:
                raise ProjectQuota.Error(err.strip())

    def __repr__(self):
        return "<{}: {}>".format(type(self).__name__, self.__path)

    @staticmethod
    def __qcmd(cmd, type_):
        return (cmd << 8) | (type_ & 0x00ff)

    @classmethod
    def check_support(cls):
        if not (common_os.User.has_root and os.path.exists(cls.PROJECT_QUOTA_BINARY)):
            return
        process = sp.Popen(
            [cls.PROJECT_QUOTA_BINARY, "project", "/"], stdout=sp.PIPE, stderr=sp.PIPE
        )
        process.communicate()
        cls.supported = not process.returncode

    @property
    def path(self):
        return self.__path

    @common_patterns.singleton_property
    def mountpoint(self):
        st = os.stat(self.__path)
        with open("/proc/self/mountinfo") as f:
            for line in f:
                parts = line.split()
                major, minor = map(int, parts[2].split(":"))
                root_path = parts[4]
                if os.makedev(major, minor) != st.st_dev:
                    continue
                _ = parts.index("-")
                fstype, device = parts[_ + 1:_ + 3]
                return self.MountPoint(device=device, fstype=fstype, root_path=root_path, stat=st)

    def destroy(self, ignore_errors=False):
        if not self.supported:
            return
        process = self.__popen(
            [self.PROJECT_QUOTA_BINARY, "destroy", self.__path], stdout=sp.PIPE, stderr=sp.PIPE
        )
        _, err = process.communicate()
        if process.returncode and not ignore_errors:
            raise ProjectQuota.Error(err.strip())

    @property
    def project(self):
        if not self.supported:
            return 0
        if self.__project_id is None:
            process = self.__popen(
                [self.PROJECT_QUOTA_BINARY, "project", self.__path], stdout=sp.PIPE, stderr=sp.PIPE
            )
            out, err = process.communicate()
            if process.returncode:
                raise ProjectQuota.Error(err.strip())
            self.__project_id = int(out)
        return self.__project_id

    @project.setter
    def project(self, project_id):
        if not self.supported:
            return
        process = self.__popen(
            [self.PROJECT_QUOTA_BINARY, "project", self.__path, str(project_id)], stdout=sp.PIPE, stderr=sp.PIPE
        )
        _, err = process.communicate()
        if process.returncode:
            raise ProjectQuota.Error(err.strip())
        self.__project_id = project_id

    @common_patterns.singleton_classproperty
    def quotactl(self):
        libc = ctypes.CDLL("libc.so.6")
        quotactl = libc.quotactl
        return quotactl

    @property
    def info(self):
        if not self.supported:
            return
        blk = self.dqblk()
        self.quotactl(
            self.__qcmd(self.Q_GETQUOTA, self.PRJQUOTA),
            self.mountpoint.device,
            self.project,
            ctypes.byref(blk)
        )
        return self.QuotaInfo(blk.curspace, blk.curinodes, blk.bhardlimit, blk.ihardlimit)

    @property
    def usage(self):
        info = self.info
        if not info:
            return 0
        return info.usage

    @property
    def limit(self):
        info = self.info
        if not info:
            return 0
        return info.limit * self.QIF_DQBLKSIZE

    @limit.setter
    def limit(self, limit):
        if not self.supported:
            return
        blk = self.dqblk()
        blocks, remainder = divmod(limit, self.QIF_DQBLKSIZE)
        if remainder:
            blocks += 1
        blk.bhardlimit = blocks
        blk.bsoftlimit = 0
        blk.valid = self.QIF_BLIMITS
        self.quotactl(
            self.__qcmd(self.Q_SETQUOTA, self.PRJQUOTA),
            self.mountpoint.device,
            self.project,
            ctypes.byref(blk)
        )


class TranslateSkyboneErrors(object):
    def __init__(self, logger, msg="Temporary copier error"):
        import api.copier
        api.copier.Copier()
        self.logger = logger
        self.msg = msg

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if not exc_type:
            return
        try:
            import ya.skynet.services.copier.rpc.errors as rpc_errors
        except ImportError:
            import ya.skynet.services.skybone.rpc.errors as rpc_errors

        import api.copier.errors
        if issubclass(
                exc_type,
                (
                    api.copier.errors.ApiConnectionError,
                    api.copier.errors.Timeout,
                    rpc_errors.Timeout,
                    rpc_errors.HandshakeError,
                    rpc_errors.HandshakeTimeout,
                    rpc_errors.CallTimeout,
                )
        ):
            self.logger.error(self.msg, exc_info=(exc_type, exc_val, exc_tb))
            raise common_errors.TemporaryError("{}: {}".format(self.msg, str(exc_val)))
        if issubclass(
                exc_type,
                (
                    api.copier.errors.FilesystemError,
                    api.copier.errors.UnshareableResource,
                )
        ):
            raise errors.InvalidData("{}: {}".format(self.msg, str(exc_val)))

        if issubclass(exc_type, api.copier.errors.CopierError):
            raise errors.CopierError("{}: {}".format(self.msg, str(exc_val)))


class Synchronized(object):
    """
    Synchronize key adding into a global storage between several greenlets.
    It is used for example to wait resource synchronization by second greenlet while fist one already synchronizing it.
    """

    def __init__(self, logger, storage, key, message, obfuscate=False):
        """
        Constructor.

        :param logger:      Logger object to be used on logging message
        :param storage:     Global storage the key will be registered in
        :param key:         Key to be registered in a storage
        :param message:     Message to be logged if the object will block on entering critical section.
        :param obfuscate:   Obfuscate key on logging the message
        """
        self.logger = logger
        self.storage = storage
        self.key = key
        self.message = message
        self.obfuscate = obfuscate
        self.lock = None

    def __enter__(self):
        my_lock = gevent.lock.RLock()
        self.lock = self.storage.setdefault(self.key, my_lock)
        if my_lock != self.lock:
            self.logger.info(self.message, common_format.obfuscate_token(self.key) if self.obfuscate else self.key)
        self.lock.acquire()

    def __exit__(self, *_):
        self.storage.pop(self.key, None)
        self.lock.release()
