#!/home/zomb-sandbox/venv/bin/python -W ignore::UserWarning

"""
Sandbox client
"""

from __future__ import print_function
from __future__ import absolute_import

import os
import sys
import signal
import logging
# FIXME: temporary turn off container delta limit [SANDBOX-5069]
# import platform

SANDBOX_DIR = os.path.realpath(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))  # noqa
sys.path = ["/skynet", os.path.dirname(SANDBOX_DIR), SANDBOX_DIR] + sys.path  # noqa

from sandbox import common
from sandbox.common import config
import sandbox.common.types.misc as ctm
common.import_hook.setup_sandbox_namespace()  # noqa

import sandbox.client as sc

from sandbox.client.bin import launch


logger = logging.getLogger("client")


class Workers(common.process.Master):
    """
    Client's workers pool. Singleton.
    """

    class AgentR(common.process.Slave):
        def __init__(self):
            logger.info('Starting AgentR.')
            pidfile = os.path.join(config.Registry().client.dirs.run, 'agentr.pid')
            super(Workers.AgentR, self).__init__(logger, pidfile, None)

        def process(self, data):
            # Daemon cannot process any IPC messages.
            pass

        def main(self):
            import agentr.bin
            cmd = [sys.executable, os.path.join(os.path.dirname(agentr.bin.__file__), 'daemon.py')]
            self.logger.info('AgentR started with PID #%s. execv(%r)', self.mypid, cmd)
            os.environ[common.os.User.SERVICE_USER_ENV] = sc.system.SERVICE_USER.login
            with common.os.User.Privileges():
                os.execv(sys.executable, cmd)
            assert False, "This point should not be reached"

    class Fileserver(common.process.Slave):
        def __init__(self):
            logger.info('Starting Fileserver.')
            pidfile = os.path.join(config.Registry().client.dirs.run, 'fileserver.pid')
            super(Workers.Fileserver, self).__init__(logger, pidfile, None)

        def process(self, data):
            # Daemon cannot process any IPC messages.
            pass

        def main(self):
            cmd = [sys.executable, "-m", "sandbox.fileserver.fileserver"]
            self.logger.info("Fileserver started with PID #%s. execv(%r)", self.mypid, cmd)
            os.environ[common.os.User.SERVICE_USER_ENV] = sc.system.SERVICE_USER.login
            os.chdir(os.path.dirname(config.Registry().common.dirs.service))
            os.environ["PYTHONPATH"] += ":{}:{}".format(
                os.path.join(config.Registry().common.dirs.service),
                os.path.join(config.Registry().common.dirs.service, "fileserver"),
            )
            with common.os.User.Privileges():
                os.execv(sys.executable, cmd)
            assert False, "This point should not be reached"

    def _slaves_builder(self):
        if config.Registry().client.agentr.enabled:
            yield self.AgentR()
        if config.Registry().client.fileserver.enabled:
            yield self.Fileserver()

    def stop(self):
        for p in self.processes:
            if not p.is_alive:
                continue
            logger.info('Asking %s (pid %s) to stop.', p.__class__.__name__, p.pid)
            try:
                with common.os.User.Privileges():
                    os.kill(p.pid, signal.SIGTERM)
            except OSError:
                pass


def daemonize():
    if config.Registry().client.daemonize:
        logger.info("Daemonize self.")
        pidfile_name = os.path.join(config.Registry().client.dirs.run, "client.py.pid")
        d = common.daemon.Daemon(pidfile_name, finalize_fn=lambda: sc.pinger.PingSandboxServerThread().stop())
        d.daemonize(ch_to=config.Registry().client.tasks.data_dir)

        # Reset porto connection after fork.
        # Don't care for other threads because there can be only one after fork.
        if config.Registry().client.porto.enabled:
            sc.platforms.porto.PortoContainerRegistry()._tls.connection = None


def run():
    Workers(logger).init().start()
    # setup tasks dir
    sys.path.insert(0, config.Registry().client.tasks.code_dir)

    launch.start(Workers)
    os._exit(-2)


def main():
    # Set HOME for OS X
    os.environ["HOME"] = sc.system.SERVICE_USER.home
    try:
        launch.initialize()
        launch.log_system_info(start_type="from filesystem")
        daemonize()
        run()
    except Exception:
        if logger.handlers:
            logger.exception("Error during client initialization.")
        print("Error during client initialization.", file=sys.stderr)
        import traceback
        traceback.print_exc(file=sys.stderr)
        os._exit(1)


if __name__ == "__main__":
    common.process.run_as_root()
    launch.prepare_main_process()

    limit = (
        config.Registry().common.installation == ctm.Installation.LOCAL and
        common.os.User.has_root
    )

    with common.os.User.Privileges(sc.system.SERVICE_USER.login, limit=limit):
        launch.check_client_stop()
        main()
