#!/usr/bin/env python -W ignore::UserWarning

from __future__ import print_function

import os
import sys
import time
import random
import signal
import logging
import resource
import functools
import threading
# Do not remove the line below. It's here to avoid deadlock.
import unicodedata  # noqa

logger = logging.getLogger('server')  # noqa

import faulthandler

import tornado.web
import tornado.ioloop
import tornado.autoreload
from tornado.platform.auto import set_close_exec

# Setup custom picklers explicitly (SANDBOX-7356)
# noinspection PyUnresolvedReferences
import kernel.util.pickle  # noqa

from kernel.util import console

SANDBOX_DIR = reduce(lambda p, _: os.path.dirname(p), xrange(2), os.path.abspath(__file__))  # noqa
sys.path = ["/skynet", os.path.dirname(SANDBOX_DIR), SANDBOX_DIR] + sys.path  # noqa

from sandbox import common
common.encoding.setup_default_encoding()

import sandbox.common.types.misc as ctm
import sandbox.common.atfork.stdlib_fixer
common.import_hook.setup_sandbox_namespace()  # noqa

from sandbox.yasandbox import manager
# Enable local mode for managers before loading anything from `yasandbox` namespace.
# But detain their initialization because of upcoming daemonization.
manager.use_locally(detain_init=True)  # noqa

from sandbox.yasandbox import services
from sandbox.yasandbox import controller
from sandbox.yasandbox.database import mapping

from sandbox.sdk2.helpers import misc

import web.server
import web.server.request
import web.server.workers


class Dozer(object):
    def __init__(self, port):
        # Check for module existance before the thread started.
        import dozer  # noqa

        t = threading.Thread(target=self._main, args=(port,))
        # this is important, if you forget this your script won't terminate
        t.daemon = True
        t.start()

    @staticmethod
    def _main(port):
        """
        Create empty wsgi app and wrap it into Dozer memory profiler;
        start the app on a given port
        """
        from wsgiref.simple_server import make_server
        from wsgiref.util import setup_testing_defaults

        import dozer

        # taken from wsgiref documentation
        def simple_app(environ, start_response):
            setup_testing_defaults(environ)
            status = '200 OK'
            headers = [('Content-type', 'text/plain')]
            start_response(status, headers)
            return ""
        simple_app = dozer.Dozer(simple_app)
        httpd = make_server('', port, simple_app)
        sys.stderr.write("Starting memory profiler on port %s...\n" % port)
        httpd.serve_forever()


def shutdown():
    deadline = time.time() + settings.server.autorestart.timeout
    web.server.workers.Workers().stop()

    logger.info('Waiting for event loop to finish.')
    io_loop = tornado.ioloop.IOLoop.instance()

    def stop_loop():
        now = time.time()
        if now < deadline and (io_loop._callbacks):
            io_loop.add_timeout(now + 1, stop_loop)
        else:
            logger.info("Shutdown the event loop.")
            io_loop.stop()
            services.Brigadier().join()

    stop_loop()

# Remember original Tornado's reloader method for future patching it.
tornado_reloader = tornado.autoreload._reload
# HTTP server instance
http_server = None


def reloader(delay=0):
    # Avoid future requests for reload - blank it.
    tornado.autoreload._check_file = lambda *_: None

    if delay:
        if delay < 0:
            logger.info('Sleeping %d sec before autorestart.', abs(delay))
            time.sleep(abs(delay))
        else:
            th = threading.Thread(target=reloader, args=(-delay,))
            th.daemon = True
            th.start()
            return

    http_server.stop()
    logger.info('Initiating autorestart.')
    tornado.ioloop.IOLoop.instance().add_callback(shutdown)
    web.server.workers.KamikadzeThread(settings.server.autorestart.timeout, on_finish=tornado_reloader)


def finalize(*args, **kw):
    if web.server.workers.Workers().stopping:
        return
    http_server.stop()
    logger.info('Terminating the application.')
    tornado.ioloop.IOLoop.instance().add_callback(shutdown)
    web.server.workers.KamikadzeThread(settings.server.autorestart.timeout, on_finish=lambda: os._exit(-1)).join()


# The Web server's entry point.
def main():
    settings = common.config.Registry()
    autoreload = settings.server.autoreload
    logger.info("Running main(). Auto-reload mode is %s.", 'ENABLED' if autoreload else 'DISABLED')

    # Block automatic server restarts for some period
    checkf = os.path.join(settings.common.dirs.runtime, "server_check_stop")
    try:
        open(checkf, "w").close()
    except OSError:
        pass

    # log db settings
    logger.info("Mongo settings: %s", settings.server.mongodb.connection_url)

    # Set common signal handlers.
    faulthandler.enable()
    signal.signal(signal.SIGUSR2, lambda *_: common.threading.dump_threads(logger))

    # Set custom process title.
    console.setProcTitle("[sandbox] Web Server")

    # loading owners for projects if it's possible
    owners_file_path = os.path.join(
        common.config.Registry().client.tasks.code_dir,
        "projects",
        ctm.EnvironmentFiles.OWNERS_STORAGE
    )
    try:
        owners_storage = misc.SingleFileStorage(owners_file_path)
        if owners_storage:
            owners_storage.load()
    except IOError:
        logger.warning("There are no pre-stored owners in %s", owners_file_path)

    # start the service process.
    services.Brigadier().start()
    # Initialize threads pool singleton and start it.
    web.server.workers.Workers(settings.server.web.threads_per_process, settings.server.web.instances, finalize).start()
    # Set common reply headers.
    from sandbox import sdk2
    sdk2.Task.current = None
    from sandbox import projects
    web.server.request.SandboxHandler.reply_headers[ctm.HTTPHeader.TASKS_REVISION] = str(projects.__revision__)

    if settings.server.profiler.performance.enabled:
        try:
            if not os.path.exists(settings.server.profiler.performance.data_dir):
                os.makedirs(settings.server.profiler.performance.data_dir)
            logging.info(
                "Performance profiler enabled with data dir '%s'",
                settings.server.profiler.performance.data_dir
            )
        except OSError as ex:
            logging.error("Unable to prepare performance profiler data dir: %s", ex)
            settings.server.profiler.performance.enabled = False

    global http_server
    http_server = web.server.SandboxApp(
        [
            # Own REST API handlers for UING requests.
            (r"^/api.*$", web.server.request.SandboxHandler),
            # Legacy UI handlers
            (r"^/sandbox/?$", tornado.web.RedirectHandler, {"url": "/sandbox/tasks/list", "permanent": False}),
            (r"^/http_check$", web.server.request.SandboxHandler),
            (r"^/sandbox.*$", web.server.request.SandboxHandler),
            # setup common rule as last one
            (
                r"^/docs/(.*)",
                tornado.web.StaticFileHandler,
                {'path': settings.server.web.static.docs_link}
            ),

            # Redirect swagger to index.html
            (r"^/media/swagger-ui/?$", tornado.web.RedirectHandler, {
                "url": "/media/swagger-ui/index.html", "permanent": False
            }),
            (r"^/(media.*)", tornado.web.StaticFileHandler, {'path': settings.server.web.static.root_path}),

            # Handle all other prefixes proxying them to `index.html` file.
            (
                r"^/.*", web.server.request.MDSProxyHandler, {"installation_type": settings.common.installation}
            ),
        ],
        autoreload=autoreload
    )

    # Set various signal handler.
    def reload_me(sig=None, frame=None, delay=0):
        if sig:
            signal.signal(signal.SIGHUP, signal.SIG_IGN)
            signal.signal(signal.SIGUSR1, signal.SIG_IGN)
            print('Caught signal: {}'.format(sig), file=sys.stderr)
        # Block automatic server restarts for some period
        try:
            open(checkf, "w").close()
        except OSError:
            pass
        reloader(delay)

    signal.signal(signal.SIGINT, finalize)
    signal.signal(signal.SIGHUP, reload_me)
    signal.signal(
        signal.SIGUSR1,
        functools.partial(reload_me, delay=random.randrange(*settings.server.autorestart.delay))
    )

    # Patch tornado's reloader to gracefully restart server with some delay.
    tornado.autoreload._reload = reload_me

    def before_reload():
        logger.info('------GO-GO-GO------')
        for fd in range(3, resource.getrlimit(resource.RLIMIT_NOFILE)[1]):
            try:
                set_close_exec(fd)
            except IOError:
                pass
    tornado.autoreload.add_reload_hook(before_reload)

    logger.info('Establishing database connection.')
    mapping.ensure_connection()
    logger.info('Initializing manager objects.')
    manager.initialize_locally()
    logger.info('Initializing controllers')
    controller.initialize()
    logger.info('Setting up statistics processing in the main thread')
    common.statistics.Signaler(
        common.statistics.ServerSignalHandler(),
        logger=logger,
        component=ctm.Component.SERVER,
        update_interval=common.config.Registry().server.statistics.update_interval,
    )

    port = settings.server.web.address.port
    logger.info('Listening port %d', port)
    http_server.listen(
        port, xheaders=True, max_buffer_size=settings.server.web.max_body_size << 20, address="localhost"
    )

    logger.info("Starting loop")
    tornado.ioloop.IOLoop.instance().start()
    web.server.workers.KamikadzeThread.finish()
    logger.info("Exit main()")


def check_for_updates_and_reload():
    from sandbox.yasandbox.services import update_server_code

    logger.info("Establishing database connection.")
    mapping.ensure_connection()

    logger.info("Checking for server code updates.")
    if update_server_code.update_packages(logger=logger):
        logger.info("Reloading the process.")
        os.execve(sys.executable, [sys.executable] + sys.argv + ["foo"], os.environ)
        raise Exception("This point should not be reached")

    tasks_version = common.package.package_version("tasks")
    if tasks_version is None or tasks_version == 0:
        raise Exception("Can't start with tasks code archive version = {}".format(tasks_version))


# The script's main entry point starts here.
if __name__ == "__main__":
    # fix deadlocks while using os.fork with threads
    sandbox.common.atfork.monkeypatch_os_fork_functions()
    sandbox.common.atfork.stdlib_fixer.fix_logging_module()
    # setup logging
    settings = common.config.Registry()
    common.log.setup_log(
        os.path.join(settings.server.log.root, settings.server.log.name),
        settings.server.log.level
    )

    logger.info('Start Sandbox server [%s]', sys.argv[0])

    # ensure working dirs
    common.fs.make_folder(settings.client.dirs.run)
    common.fs.make_folder(settings.server.log.root)

    # init projects
    sys.path.insert(0, settings.client.tasks.code_dir)

    common.rest.Client._default_component = common.proxy.ReliableServerProxy._default_component = ctm.Component.SERVER
    if settings.client.auth.oauth_token:
        logger.info("Passing token to channel")
        common.proxy.ReliableServerProxy._external_auth = common.proxy.OAuth(common.utils.read_settings_value_from_file(
            settings.client.auth.oauth_token, True
        ))

    if settings.server.services.packages_updater.enabled and settings.common.installation != ctm.Installation.LOCAL:
        check_for_updates_and_reload()
    common.projects_handler.load_project_types(settings.server.services.packages_updater.enabled)

    if settings.common.installation == ctm.Installation.LOCAL:
        from sandbox import projects

        sdk1_warning = "It's forbidden to create new SDK1-based tasks. Use SDK2, please."
        try:
            bl = projects.BLACK_LIST
        except AttributeError:
            pass
        else:
            mapping.ensure_connection()
            mapping.UINotification.objects(content__startswith=sdk1_warning).delete()
            if bl:
                message = "{} Ignored task{}: {}".format(sdk1_warning, "s" if len(bl) > 1 else "", " ".join(bl))
                print(message)
                mapping.UINotification(severity="WARNING", content=message).save()

    # We should disconnect from the database before forking.
    mapping.disconnect()

    # dowser
    if settings.server.profiler.memory.enabled:
        Dozer(settings.server.profiler.memory.port)

    # daemonize
    if settings.server.daemonize:
        logger.debug("Daemonizing")
        pidfile = os.path.join(settings.client.dirs.run, 'server.py.pid')
        common.daemon.Daemon(
            pidfile_name=pidfile,
            stdout=os.path.join(settings.server.log.root, settings.server.log.name),
            stderr=os.path.join(settings.server.log.root, settings.server.log.name),
            finalize_fn=finalize
        ).daemonize(None)
        logger.debug("I'm daemon!")

    # main
    main()
