#!/usr/bin/env python
# coding: utf-8

import os
import py
import sys
import logging
import optparse
import subprocess as sp

import colorlog


def setup_environ():
    sandbox_path = py.path.local(__file__).dirpath(os.path.pardir)
    sys.path = [str(sandbox_path.dirname), str(sandbox_path)] + sys.path

    from sandbox import common
    common.encoding.setup_default_encoding()
    common.import_hook.setup_sandbox_namespace()

    settings = common.config.Registry()
    sys.path.insert(0, settings.client.tasks.code_dir)

    with common.console.LongOperation("Establish database connection"):
        from sandbox.yasandbox.database import mapping
        mapping.ensure_connection()

    with common.console.LongOperation("Initializing controllers"):
        from sandbox.yasandbox import manager
        from sandbox.yasandbox import controller
        manager.use_locally()
        controller.initialize()

    # Setup logging and log some audit information
    logger = common.log.setup_log(
        os.path.join(settings.server.log.root, settings.server.log.name),
        settings.server.log.level
    )
    try:
        keys = "\n" + sp.check_output(["/usr/bin/ssh-add", "-l"])
    except (OSError, sp.CalledProcessError):
        keys = "None"
    logger.info(
        "Started interactive terminal session for SSH client %r, loaded keys are: " + keys,
        os.environ.get("SSH_CLIENT")
    )

    # And now switch logging to stderr
    for no, alias in {
        logging.DEBUG: "DBG",
        logging.INFO: "INF",
        logging.WARNING: "WRN",
        logging.ERROR: "ERR",
        logging.CRITICAL: "CRI",
        logging.NOTSET: "---",
    }.iteritems():
            logging.addLevelName(no, alias)

    root = logging.getLogger()
    map(root.removeHandler, root.handlers[:])
    map(root.removeFilter, root.filters[:])
    root.setLevel(logging.DEBUG)

    handler = logging.StreamHandler(sys.stderr)
    handler.setFormatter(colorlog.ColoredFormatter(
        "%(cyan)s%(asctime) s%(log_color)s%(bold)s%(levelname)4s%(reset)s "
        "%(bold)s%(cyan)s[%(name)s]%(reset)s %(message)s",
        log_colors={"DBG": "black", "INF": "green", "WRN": "yellow", "ERR": "red", "CRI": "red"}
    ))
    handler.setLevel(logging.DEBUG)
    root.addHandler(handler)


def ipython(srv_rev, tasks_rev, tasks_cnt, res_cnt):
    """
    Some IPython tricks:

    * **Editing**

      Firstly, set `EDITOR` enviroment variable or set editor in `~/.ipython/ipythonrc`
      To edit file, type `edit filename.py`, after closing the editor
      the file will be executed. To prevent execution, type `edit -x`

    * **Interactive debugging**

      Type `pdb` in IPython shell after exception has occurred, and
      you will be given an `ipdb>` shell. Type `?` for help on ipdb or `q` for quit.

      Hint: to observe local state it's convinient to raise an exception by doing `1/0`.

    * **Helpful commands**

      `%history -g foo` -- search history;

      `%history?` -- help regarding history management;

      `%rep <number_of_command>` -- retype a command;

      `Out[N]` -- get the object returned in command `N`;

      `%cpaste` -- copy stuff crom clipboard, type `--` to finish;

      `<object>?` - get help on `object`;

      `<object>??` -- peek into `object`'s source code;

      `edit _iN` -- edit input from command `N` in editor in a temporary file;

      `run -d -b LINE_NO file` -- run file and stop at `LINE_NO` with `ipdb`;

      `import pdb; pdb.set_trace()`

      `%time command` -- measure time of the `command` execution

      `%prun -l substr command` -- profile `command`, output only lines related to functions with `substr` in name
    """

    import IPython.terminal.interactiveshell as ipsh
    banner = 'Sandbox shell r{}/r{}. Known task types: {}, resource types: {}'.format(
        srv_rev, tasks_rev, tasks_cnt, res_cnt
    )
    shell = ipsh.TerminalInteractiveShell(banner2=banner)
    shell.show_banner()
    shell.mainloop()


def main():
    setup_environ()

    from sandbox import sdk2
    from sandbox import common
    from sandbox import projects

    params = get_parameters()
    if params["load_projects"]:
        with common.console.LongOperation("Loading tasks code"):
            common.projects_handler.load_project_types()
    else:
        projects.TYPES = ()
    if params["file_to_execute"]:
        execfile(params["file_to_execute"])
    elif params["eval"]:
        exec(params["eval"])
    else:
        rev_path = os.path.join(os.path.dirname(os.path.dirname(__file__)), ".revision")
        if os.path.exists(rev_path):
            with open(rev_path) as f:
                rev = f.read()
        else:
            rev = 0
        ipython(rev, projects.__revision__, len(projects.TYPES), len(list(sdk2.resource.AbstractResource)))


def get_parameters():
    parser = optparse.OptionParser()
    parser.add_option("-C", "--config", default=None,
                      help="Path to Sandbox server config file.")
    parser.add_option("-c", "--command", default=None,
                      help="Execute (evaluate) specified command")
    parser.add_option("-e", "--execute", dest="execute", default=None,
                      help="Execute specific file with initialized environment")
    parser.add_option("-T", "--without-tasks", action="store_true", help="Don't import tasks code")
    options, args = parser.parse_args()
    if not options.config:
        config_file = None
    else:
        config_path = os.path.abspath(os.path.expanduser(options.config))
        if os.path.isfile(config_path):
            config_file = config_path
        else:
            print "Config file %s was not found." % config_path
            config_file = None
    return {
        "server_config": config_file,
        "file_to_execute": options.execute,
        "eval": options.command,
        "load_projects": not options.without_tasks,
    }


if __name__ == "__main__":
    main()
